﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Windows.Forms.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Configurations;
    using NintendoWare.SoundMaker.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Preview;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    /// 波形ファイルの出力 コマンドを処理します。
    /// </summary>
    public class QueryAndOutputWaveFileHandler : CommandHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryParameter">パラメータの問い合わせ先。</param>
        public QueryAndOutputWaveFileHandler(IQueryCommandParameter queryParameter)
            : base(queryParameter)
        {
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return PreviewCommands.OutputWaveFile.ID; }
        }

        private SoundProjectConfigurationCommon ProjectConfiguration
        {
            get { return FormsApplicationCommon.Instance.ProjectConfiguration; }
        }

        private XmlPreviewCommonOutputWaveFile OutputWaveFileConfiguration
        {
            get { return ProjectConfiguration.PreviewCommon.OutputWaveFile; }
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            if (null == GetTargetProjectService(parameters)) { return CommandStatus.SupportedAndVisible; }
            if (0 == GetTargetSounds(parameters).Length) { return CommandStatus.SupportedAndVisible; }
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            if (FormsApplicationCommon.Instance.IsErrorAudioDevice == true)
            {
                FormsApplicationCommon.Instance.ShowWarningMessageCanNotInitializeAudioDevice();
                return false;
            }

            Sound[] targets = GetTargetSounds(parameters);
            if (0 == targets.Length) { return false; }

            string initDestinationFolderPath = string.Empty;

            if (ProjectConfiguration.Uris.ContainsKey(Uris.UriName_PreviewOutputWaveFile))
            {
                initDestinationFolderPath = ProjectConfiguration.Uris[Uris.UriName_PreviewOutputWaveFile].Value;
            }
            else
            {
                initDestinationFolderPath = GetInitDestinationFolderPath(GetTargetProjectService(parameters));
            }

            OutputWaveFileDialog dialog = new OutputWaveFileDialog(targets, initDestinationFolderPath);
            dialog.MaxDuration = OutputWaveFileConfiguration.MaxDuration;
            dialog.IsOpenDestinationFolder = OutputWaveFileConfiguration.OpenDestinationFolder;
            dialog.IsMonaural = OutputWaveFileConfiguration.IsMonaural;
            dialog.Is48kHz = (OutputWaveFileConfiguration.SamplingRate == XmlOutputWaveFileSamplingRate.k48KHz);

            if (dialog.ShowDialog() != DialogResult.OK) { return false; }

            OutputWaveFileConfiguration.MaxDuration = dialog.MaxDuration;
            OutputWaveFileConfiguration.OpenDestinationFolder = dialog.IsOpenDestinationFolder;
            OutputWaveFileConfiguration.IsMonaural = dialog.IsMonaural;
            if (dialog.Is48kHz == true)
            {
                OutputWaveFileConfiguration.SamplingRate = XmlOutputWaveFileSamplingRate.k48KHz;
            }
            else
            {
                OutputWaveFileConfiguration.SamplingRate = XmlOutputWaveFileSamplingRate.k32KHz;
            }

            try
            {

                if (!Directory.Exists(dialog.DestinationFolderPath))
                {
                    Directory.CreateDirectory(dialog.DestinationFolderPath);
                }

                OutputWaveFileRenderType samplingRate = OutputWaveFileRenderType.k32KHz;
                if (AppConfigurationCommon.Enabled48kHz == true && dialog.Is48kHz == true)
                {
                    samplingRate = OutputWaveFileRenderType.k48KHz;
                }
                OutputWaveFiles(targets, dialog.DestinationFilePaths, dialog.MaxDuration, dialog.IsMonaural, samplingRate);

            }
            catch (DirectoryNotFoundException exception)
            {
                FormsApplicationCommon.Instance.UIService.ShowMessageBox(exception.Message);
                return false;
            }
            catch (Exception)
            {
                FormsApplicationCommon.Instance.UIService.ShowMessageBox(
                    FrameworkResources.MessageResource.Message_UnknownError);
                return false;
            }


            if (dialog.IsOpenDestinationFolder)
            {
                Process.Start("explorer.exe", "/e,/select, " + dialog.DestinationFilePaths[0]);
            }

            string outputFolderPath = Path.GetDirectoryName(dialog.DestinationFilePaths[0]);

            if (ProjectConfiguration.Uris.ContainsKey(Uris.UriName_PreviewOutputWaveFile))
            {
                ProjectConfiguration.Uris[Uris.UriName_PreviewOutputWaveFile].Value = outputFolderPath;
            }
            else
            {
                ProjectConfiguration.Uris.Add(Uris.UriName_PreviewOutputWaveFile,
                                               new XmlUri()
                                               {
                                                   Name = Uris.UriName_PreviewOutputWaveFile,
                                                   Value = outputFolderPath,
                                               });
            }

            return true;
        }

        /// <summary>
        /// 対象プロジェクトサービスを取得します。
        /// </summary>
        private SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                   as SoundProjectService;
        }

        /// <summary>
        /// 対象サウンドの一覧を取得します。
        /// </summary>
        private Sound[] GetTargetSounds(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            return (from Component component in
                        (parameters.GetParameter(CommandParameterNames.TargetComponents)
                          as IEnumerable<Component>)
                    where component is Sound
                    select component as Sound).ToArray();
        }

        /// <summary>
        /// 初期出力フォルダを取得します。
        /// </summary>
        private string GetInitDestinationFolderPath(SoundProjectService projectService)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (null == projectService.ProjectDocument) { return string.Empty; }
            return Path.GetDirectoryName(projectService.ProjectDocument.Resource.Key) + "\\";
        }

        private void OutputWaveFiles(Sound[] sounds, string[] filePaths, int maxDuration, bool isMonaural, OutputWaveFileRenderType samplingRate)
        {
            if (null == sounds) { throw new ArgumentNullException("sounds"); }
            if (null == filePaths) { throw new ArgumentNullException("filePaths"); }

            if (sounds.Length != filePaths.Length)
            {
                throw new ArgumentException("sounds.Length != filePaths.Length");
            }

            FormsApplicationCommon.Instance.ExecuteCommand(PreviewCommands.StopAll);
            FormsApplicationCommon.Instance.ExecuteCommand(ToolCommands.StopAllSounds);
            FormsApplicationCommon.Instance.ShutdownWaveOutput();


            int index = 0;

            foreach (Sound sound in sounds)
            {

                OutputWaveFile(sound, filePaths[index], maxDuration, isMonaural, samplingRate);

                index++;

            }

            FormsApplicationCommon.Instance.StartupWaveOutput();
        }

        private void OutputWaveFile(Sound sound, string filePath, int maxDuration, bool isMonaural, OutputWaveFileRenderType samplingRate)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            Preview.PreviewSound previewSound = Preview.PreviewPlayer.CreatePreviewSound(sound, samplingRate);

            if (null == previewSound)
            {
                throw new ApplicationException(FrameworkResources.MessageResource.Message_UnknownError);
            }

            previewSound.OutputWaveFile(filePath, maxDuration, isMonaural);
        }
    }
}
