﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Preview.Communications;
    using NintendoWare.SoundMaker.Preview.MCS;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    public class PreviewPlayerSettings : IDisposable
    {
        private PreviewPlayerManager _manager = null;

        private bool _dirty = false;
        private bool _autoWriteSeqVariables = true;                                         // シーケンス変数の自動書き込み設定
        private SeqVariableSyncType _seqVariablesSyncType = SeqVariableSyncType.Auto500ms;  // シーケンス変数の更新設定
        private SeqVariableContainer _seqSyncTargetVariables = null;                     // 更新対象のシーケンス変数コンテナ

        // ユーティリティ
        private PlayerStateWatcher _previewPlayerWatcher = null;     // プレビュープレイヤー状態の同期コンポーネント
        private PreviewPlayerParameterWatcher _previewPlayerParameterWatcher = null;     // プレビュープレイヤーパラメータの同期コンポーネント
        private SeqVariableWatcher _seqVariableWatcher = null;       // シーケンス変数の同期コンポーネント
        private EffectWatcher _effectWatcher = null;       // エフェクトの同期コンポーネント

        public PreviewPlayerSettings(PreviewPlayerManager manager)
        {
            if (null == manager) { throw new ArgumentNullException("manager"); }
            _manager = manager;
        }

        public event EventHandler AutoWriteVariablesChanged;

        public event EventHandler VariablesSyncTypeChanged;

        public bool IsDirty
        {
            get { return _dirty; }
            set { _dirty = value; }
        }

        public bool AutoWriteVariables
        {
            get { return _autoWriteSeqVariables; }
            set
            {
                if (value == _autoWriteSeqVariables) { return; }

                _autoWriteSeqVariables = value;
                _dirty = true;

                if (null != AutoWriteVariablesChanged)
                {
                    AutoWriteVariablesChanged(this, new EventArgs());
                }
            }
        }

        public SeqVariableSyncType VariablesSyncType
        {
            get { return _seqVariablesSyncType; }
            set
            {
                if (value == _seqVariablesSyncType) { return; }

                _seqVariablesSyncType = value;
                _dirty = true;

                if (null != VariablesSyncTypeChanged)
                {
                    VariablesSyncTypeChanged(this, new EventArgs());
                }

                _seqVariableWatcher.Reset();
            }
        }

        public SeqVariableContainer SyncTargetVariables
        {
            get { return _seqSyncTargetVariables; }
            set
            {
                if (value == _seqSyncTargetVariables) { return; }

                _seqSyncTargetVariables = value;

                _seqVariableWatcher.Reset();
            }
        }

        public PlayerStateWatcher PlayerStateWatcher
        {
            get { return _previewPlayerWatcher; }
        }

        public PreviewPlayerParameterWatcher PreviewPlayerParameterWatcher
        {
            get { return _previewPlayerParameterWatcher; }
        }

        public SeqVariableWatcher SeqVariableWatcher
        {
            get { return _seqVariableWatcher; }
        }

        public EffectWatcher EffectWatcher
        {
            get { return _effectWatcher; }
        }

        public void SwitchPlayer(PreviewPlayerType type, CommunicationService communicationService)
        {
            DisposeWatchers();

            switch (type)
            {
                case PreviewPlayerType.PCPlayer:
                    _previewPlayerWatcher = new PlayerStateWatcherForPC(_manager);
                    _previewPlayerParameterWatcher = new PreviewPlayerParameterWatcherForPC(_manager);
                    _seqVariableWatcher = new SeqVariableWatcherForPC(_manager, communicationService);
                    _effectWatcher = new EffectWatcherForPC(_manager);
                    break;

                case PreviewPlayerType.ViewerPlayer:
                    _previewPlayerWatcher = new PlayerStateWatcherForViewer(_manager);
                    _previewPlayerParameterWatcher = new PreviewPlayerParameterWatcherForViewer(_manager, communicationService);
                    _seqVariableWatcher = new SeqVariableWatcherForViewer(_manager, communicationService);
                    _effectWatcher = new EffectWatcherForViewer(_manager, communicationService);
                    break;

                default:
                    throw new ArgumentException("invalid preview player type");
            }
        }

        public void Dispose()
        {
            DisposeWatchers();
        }

        private void DisposeWatchers()
        {
            if (null != _previewPlayerWatcher)
            {
                _previewPlayerWatcher.Dispose();
                _previewPlayerWatcher = null;
            }
            if (null != _previewPlayerParameterWatcher)
            {
                _previewPlayerParameterWatcher.Dispose();
                _previewPlayerParameterWatcher = null;
            }
            if (null != _seqVariableWatcher)
            {
                _seqVariableWatcher.Dispose();
                _seqVariableWatcher = null;
            }
            if (null != _effectWatcher)
            {
                _effectWatcher.Dispose();
                _effectWatcher = null;
            }
        }
    }
}
