﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using NW4R.ProtocolSound;

    public enum DeviceType : uint
    {
        TV = 0,
        DRC0 = 1,
        DRC1 = 2,
    }

    public class EffectManager
    {
        #region ** 定数

        private const uint _defaultEffectCount = 3;
        private const uint _defaultDeviceCount = 2;

        #endregion

        private EffectParameterCollection[] _effectParameters = new EffectParameterCollection[_defaultDeviceCount];

        public uint DeviceTypeToUint(DeviceType deviceType)
        {
            switch (deviceType)
            {
                case DeviceType.TV:
                    return 0;

                case DeviceType.DRC0:
                    return 1;
#if false
            case DeviceType.DRC1:
                return 2;
#endif
                default:
                    Debug.Assert(false);
                    return 0;
            }
        }

        public EffectManager() : this(_defaultEffectCount) { }

        public EffectManager(uint effectCount)
        {
            for (uint device = 0; device < _defaultDeviceCount; device++)
            {
                _effectParameters[device] = new EffectParameterCollection();
            }

            InitializeEffectParameter(this.GetEffectParameters(DeviceType.TV), effectCount);
            InitializeEffectParameter(this.GetEffectParameters(DeviceType.DRC0), effectCount);
#if false
            InitializeEffectParameter(this.GetEffectParameters(DeviceType.DRC1), effectCount);
#endif
        }

        #region ** プロパティ

        #endregion

        #region ** イベント

        public event EventHandler EffectParameterChanged
        {
            add
            {
                this.EffectParameterChangedAdd(this.GetEffectParameters(DeviceType.TV), value);
                this.EffectParameterChangedAdd(this.GetEffectParameters(DeviceType.DRC0), value);
#if false
                this.EffectParameterChangedAdd(this.GetEffectParameters(DeviceType.DRC1), value);
#endif
            }
            remove
            {
                this.EffectParameterChangedRemove(this.GetEffectParameters(DeviceType.TV), value);
                this.EffectParameterChangedRemove(this.GetEffectParameters(DeviceType.DRC0), value);
#if false
                this.EffectParameterChangedRemove(this.GetEffectParameters(DeviceType.DRC1), value);
#endif
            }
        }

        #endregion

        public EffectParameterCollection ParameterSets(DeviceType deviceType)
        {
            return this.GetEffectParameters(deviceType);
        }

        private void EffectParameterChangedAdd(EffectParameterCollection effectParameter, EventHandler value)
        {
            foreach (EffectParameterSet item in effectParameter)
            {
                item.EffectParameterChanged += value;
            }
        }

        private void EffectParameterChangedRemove(EffectParameterCollection effectParameter, EventHandler value)
        {
            foreach (EffectParameterSet item in effectParameter)
            {
                item.EffectParameterChanged -= value;
            }
        }

        private void InitializeEffectParameter(EffectParameterCollection effectParameter, uint effectCount)
        {
            for (uint i = 0; i < effectCount; i++)
            {
                EffectID auxA_ID = new EffectID(ToolSoundAuxBusType.AuxBusType_AuxA, i);
                EffectID auxB_ID = new EffectID(ToolSoundAuxBusType.AuxBusType_AuxB, i);
                EffectID auxC_ID = new EffectID(ToolSoundAuxBusType.AuxBusType_AuxC, i);

                effectParameter.Add(new EffectParameterSet(auxA_ID));
                effectParameter.Add(new EffectParameterSet(auxB_ID));
                effectParameter.Add(new EffectParameterSet(auxC_ID));
            }
        }

        private EffectParameterCollection GetEffectParameters(DeviceType deviceType)
        {
            switch (deviceType)
            {
                case DeviceType.TV:
                    return _effectParameters[0];

                case DeviceType.DRC0:
                    return _effectParameters[1];
#if false
            case DeviceType.DRC1:
                return _effectParameters[2];
#endif
                default:
                    return _effectParameters[0];
            }
        }
    }

    /// <summary>
    /// EffectParameterSet コンテナ
    /// </summary>
    public class EffectParameterCollection : KeyedCollection<EffectID, EffectParameterSet>
    {
        #region ** KeyedCollection の実装

        /// <summary>
        /// 派生クラスで実装された場合、指定した要素からキーを抽出します。
        /// </summary>
        /// <param name="item">キーの抽出元要素。</param>
        /// <returns>指定した要素のキー。</returns>
        protected override EffectID GetKeyForItem(EffectParameterSet item)
        {
            return item.ID;
        }

        #endregion
    }

    public class EffectParameterSet
    {
        private EffectParameter _selected;

        private NoEffectParameter _noEffect;
        private ReverbHiParameter _reverbHi;
        private ReverbStdParameter _reverbStd;
        private ChorusParameter _chorus;
        private DelayParameter _delay;
        private MultiChannelReverbParameter _multiChannelReverb;
        private MultiChannelChorusParameter _multiChannelChorus;
        private MultiChannelDelayParameter _multiChannelDelay;

        public EffectParameterSet(EffectID id)
        {
            _noEffect = new NoEffectParameter(id);
            _reverbHi = new ReverbHiParameter(id);
            _reverbStd = new ReverbStdParameter(id);
            _chorus = new ChorusParameter(id);
            _delay = new DelayParameter(id);
            _multiChannelReverb = new MultiChannelReverbParameter(id);
            _multiChannelChorus = new MultiChannelChorusParameter(id);
            _multiChannelDelay = new MultiChannelDelayParameter(id);

            _noEffect.ParameterChanged += OnEffectParamterChanged;
            _reverbHi.ParameterChanged += OnEffectParamterChanged;
            _reverbStd.ParameterChanged += OnEffectParamterChanged;
            _chorus.ParameterChanged += OnEffectParamterChanged;
            _delay.ParameterChanged += OnEffectParamterChanged;
            _multiChannelReverb.ParameterChanged += OnEffectParamterChanged;
            _multiChannelChorus.ParameterChanged += OnEffectParamterChanged;
            _multiChannelDelay.ParameterChanged += OnEffectParamterChanged;

            _selected = _noEffect;
        }

        #region ** プロパティ

        public EffectID ID
        {
            get { return _selected.EffectID; }
        }

        public EffectParameter Selected
        {
            get { return _selected; }
        }

        #endregion

        #region ** イベント

        public event EventHandler EffectParameterChanged;

        #endregion

        #region ** イベントハンドラ

        private void OnEffectParamterChanged(object sender, EventArgs e)
        {
            if (null != EffectParameterChanged)
            {
                EffectParameterChanged(sender, e);
            }
        }

        #endregion

        #region ** メソッド

        public EffectParameter GetParameter(ToolSoundEffectType effectType)
        {
            switch (effectType)
            {
                case ToolSoundEffectType.EffectType_NoEffect:
                    return _noEffect;

                case ToolSoundEffectType.EffectType_ReverbHi:
                    return _reverbHi;

                case ToolSoundEffectType.EffectType_ReverbStd:
                    return _reverbStd;

                case ToolSoundEffectType.EffectType_Chorus:
                    return _chorus;

                case ToolSoundEffectType.EffectType_Delay:
                    return _delay;

                case ToolSoundEffectType.EffectType_MultiChannelReverb:
                    return _multiChannelReverb;

                case ToolSoundEffectType.EffectType_MultiChannelChorus:
                    return _multiChannelChorus;

                case ToolSoundEffectType.EffectType_MultiChannelDelay:
                    return _multiChannelDelay;
            }

            Debug.Assert(false);
            return _noEffect;
        }

        /// <summary>
        /// エフェクトの種類を選択します。
        /// </summary>
        /// <param name="effectType"></param>
        /// <returns></returns>
        public EffectParameter Select(ToolSoundEffectType effectType)
        {
            switch (effectType)
            {
                case ToolSoundEffectType.EffectType_NoEffect:
                    _selected = _noEffect;
                    break;

                case ToolSoundEffectType.EffectType_ReverbHi:
                    _selected = _reverbHi;
                    break;

                case ToolSoundEffectType.EffectType_ReverbStd:
                    _selected = _reverbStd;
                    break;

                case ToolSoundEffectType.EffectType_Chorus:
                    _selected = _chorus;
                    break;

                case ToolSoundEffectType.EffectType_Delay:
                    _selected = _delay;
                    break;

                case ToolSoundEffectType.EffectType_MultiChannelReverb:
                    _selected = _multiChannelReverb;
                    break;

                case ToolSoundEffectType.EffectType_MultiChannelChorus:
                    _selected = _multiChannelChorus;
                    break;

                case ToolSoundEffectType.EffectType_MultiChannelDelay:
                    _selected = _multiChannelDelay;
                    break;

                default:
                    Debug.Assert(false);
                    break;
            }

            if (null != EffectParameterChanged)
            {
                EffectParameterChanged(_selected, new EventArgs());
            }

            return _selected;
        }

        #endregion
    }
}
