﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Windows.Forms;
using NW4R.ProtocolSound;

namespace NintendoWare.SoundMaker.Preview.MCS.Tool
{
    //=========================================================================
    #region MCSツールシステムパケット
    /// <summary>
    /// MCSツールシステムパケット
    /// <para>
    /// ツールからビューアに送る、システム関連パケットのベースクラスです。
    /// </para>
    /// </summary>
    public abstract class MCSToolSystemPacket : MCSToolPacket
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        protected MCSToolSystemPacket()
        {
            Debug.Assert( Category == Constants.CATEGORY_SYSTEM );
        }
    }
    #endregion

    //=========================================================================
    #region プロトコルのバージョン
    /// <summary>
    /// プロトコルのバージョン
    /// <para>
    /// プロトコルのバージョンを確認するためのパケットです。
    /// </para>
    /// </summary>
    public class MCSVersionPacket : MCSToolSystemPacket
    {
        //---------------------------------------------------------------------
        /// <summary>
        /// 送る
        /// </summary>
        public static void Send()
        {
            if ( !( MCSManager.IsConnecting || MCSManager.IsConnected ) ) { return; }
            MCSManager.SendPacket( new MCSVersionPacket() );
        }

        /// <summary>
        /// 書き出し
        /// </summary>
        public override MCSPacketHeader Write( ProtocolSoundWriter writer )
        {
            MCSPacketHeader header = CreateHeader();
            header.Write( writer );
            ToolSystemVersion packet = new ToolSystemVersion();
            packet.version = Constants.PROTOCOL_VERSION;
            packet.Write( writer );
            return header;
        }

        /// <summary>
        /// 読み込み
        /// </summary>
        public override void Read(
            ProtocolSoundReader reader, MCSPacketHeader header )
        {
            Debug.Assert( header.Size == ToolSystemVersion.StructSize );
            ToolSystemVersion packet = new ToolSystemVersion();
            packet.Read( reader );
            if ( packet.version != Constants.PROTOCOL_VERSION ) {
                ShowError(
                    string.Format( "プロトコルのバージョンが違います。{0:x8} {1:x8}",
                                   Constants.PROTOCOL_VERSION, packet.version ) );
            }

            MCS.MCSManager.SetState( ConnectionState.Connected );
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SYSTEM_VERSION; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get { return ToolSystemVersion.StructSize; }
        }

        /// <summary>
        /// 文字列変換
        /// </summary>
        public override string ToString()
        {
            return base.ToString() + " " +
                Constants.PROTOCOL_VERSION.ToString( "x8" );
        }
    }
    #endregion

    //=========================================================================
    #region リセット
    /// <summary>
    /// リセット
    /// <para>
    /// ビューアをリセットするためのパケットです。
    /// </para>
    /// </summary>
    public class MCSResetPacket : MCSToolSystemPacket
    {
        //---------------------------------------------------------------------
        /// <summary>
        /// 送る
        /// </summary>
        public static void Send()
        {
            if ( !MCSManager.IsConnected ) { return; }
            MCSResetPacket packet = new MCSResetPacket();
            MCSManager.SendPacket( packet );
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SYSTEM_RESET; }
        }
    }
    #endregion

    //=========================================================================
    #region 接続を閉じる
    /// <summary>
    /// 接続を閉じる
    /// <para>
    /// 現在送信しようとしているパケットを送りきった後に接続を閉じるパケットです。
    /// </para>
    /// </summary>
    public class DisconnectPacket : MCSToolSystemPacket
    {
        //---------------------------------------------------------------------
        /// <summary>
        /// 送る
        /// </summary>
        public static void Send( Control mainControl )
        {
            if ( !MCSManager.IsConnected ) { return; }
            MCSManager.SendPacket( new DisconnectPacket( mainControl ) );
        }

        /// <summary>
        /// 書き出し
        /// </summary>
        public override MCSPacketHeader Write( ProtocolSoundWriter writer )
        {
            MCSPacketHeader header = CreateHeader();
            header.Write( writer );
            ToolSystemSync packet = new ToolSystemSync();
            packet.tagName = 0;
            packet.Write( writer );
            return header;
        }

        /// <summary>
        /// 読み込み
        /// </summary>
        public override void Read(
            ProtocolSoundReader reader, MCSPacketHeader header )
        {
            Debug.Assert( header.Size == ToolSystemSync.StructSize );
            ToolSystemSync packet = new ToolSystemSync();
            packet.Read( reader );
            Debug.Assert( packet.tagName == 0 );
            _mainControl.BeginInvoke(
                new MethodInvoker( MCSManager.Disconnect ) );
        }

        /// <summary>
        /// 圧縮
        /// </summary>
        public override bool Compress( MCSToolPacket packet )
        {
            return ( packet is DisconnectPacket );
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SYSTEM_SYNC; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get { return ToolSystemSync.StructSize; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        protected DisconnectPacket( Control mainControl )
        {
            _mainControl = mainControl;
        }

        // メインスレッドコントロール
        Control _mainControl;
    }
    #endregion

    //=========================================================================
}
