﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Generic.Preview.MCS
{
    using System;
    using System.IO;
    using System.Diagnostics;
    using System.Net;
    using System.Net.Sockets;
    using System.Threading;
    using System.Text;

    using NintendoWare.SoundMaker.Framework.Preview.Communications;
#if !USE_MCS
    using Nintendo.Alto.Foundation.Communications;
#endif
    /// <summary>
    /// MCS接続
    /// <para>
    /// ツール接続、ビューア接続、Ping接続の親クラスです。MCS接続の共通部分を
    /// 実装しています。実際に別スレッドで通信を行う部分の実装は、内部クラスの
    /// CommunicationThreadクラスにて行っています。
    /// </para>
    /// </summary>
    public abstract class MCSConnection : CommConnection
    {
        //---------------------------------------------------------------------
        // 定数
        //---------------------------------------------------------------------
        protected const uint _baseChannel =
        (((uint)'S' << 8) + (uint)'D') + 0x8000;

        // 通信
        private volatile CommunicationThread _communication = null;

        //---------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MCSConnection() { }

        /// <summary>
        /// メインループ
        /// </summary>
        public abstract void MainLoop();

        /// <summary>
        /// チャンネルの取得
        /// </summary>
        public abstract uint Channel { get; }

        /// <summary>
        /// 文字列変換
        /// </summary>
        public override string ToString() { return GetType().Name; }

        //---------------------------------------------------------------------
        // 接続関係
        //---------------------------------------------------------------------
        public override bool Connect()
        {
            return false;
        }
#if !USE_MCS
        /// <summary>
        ///
        /// </summary>
        public IComEndPoint ComEndPoint
        {
            get;
            set;
        }

        public object Port
        {
            get;
            set;
        }
#endif
        /// <summary>
        /// 接続
        /// </summary>
        public virtual bool Connect(IPEndPoint ipEndPoint, string hostName)
        {
            Debug.Assert(_communication == null);
#if USE_MCS
            _communication = new CommunicationThread( this, hostName, ipEndPoint.Port, Channel);
#else
            _communication = new CommunicationThread(this, hostName, 0, 0);
#endif

#if USE_MCS
            if( !_communication.Connect( ipEndPoint, Channel ))
#else
            if (_communication.Connect(ComEndPoint, Port) == false)
#endif
            {
                _communication = null;
                return false;
            }
            ShowMessage("通信開始 : " + _communication);
            return true;
        }

        /// <summary>
        /// 接続中か
        /// </summary>
        public override bool IsConnected { get { return (_communication != null); } }

        /// <summary>
        /// 切断
        /// </summary>
        public override void Disconnect()
        {
            Disconnect(Timeout.Infinite);
        }

        /// <summary>
        /// 切断
        /// </summary>
        public override void Disconnect(int timeout)
        {
            if (!IsConnected) { return; }
            string name = _communication.ToString();
            _communication.Disconnect(timeout);
            _communication = null;
            ShowMessage("通信停止 : " + name);
        }

        /// <summary>
        /// 終了待ち
        /// </summary>
        public bool IsWantToExit
        {
            get
            {
                if (_communication.DisconnectingFlag)
                    return true;
                if (!IsConnected)
                    return false;
                return _communication.IsWantToExit;
            }
        }

        /// <summary>
        /// 切断中
        /// </summary>
        public bool DisconnectingFlag
        {
            get
            {
                if (_communication != null)
                {
                    return _communication.DisconnectingFlag;
                }
                else
                {
                    return false;
                }
            }
            set
            {
                if (_communication != null)
                {
                    _communication.DisconnectingFlag = value;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void SendPacket(CommPacket packet)
        {
            Debug.Assert(false, "SendPacket is not implimented");
        }

        //---------------------------------------------------------------------
        // 非公開メンバ
        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージ
        /// </summary>
        protected void ShowMessage(string text)
        {
#if MCSDEBUG
           MCSManager.Console.WriteMCSMessage( text );
#endif
            Debug.WriteLine(text);
        }

        /// <summary>
        /// エラー
        /// </summary>
        protected void ShowError(string text)
        {
#if MCSDEBUG
           MCSManager.Console.WriteMCSError( text );
#endif
            Debug.WriteLine(text);
        }

#if USE_MCS
        //---------------------------------------------------------------------
        /// <summary>
        /// クライアント
        /// </summary>
        protected TcpClient Client
        {
            get { return _communication.Client; }
        }
#endif

        /// <summary>
        /// リーダ
        /// </summary>
        protected ProtocolSoundReader Reader
        {
            get { return _communication.Reader; }
        }

        /// <summary>
        /// ライタ
        /// </summary>
        protected ProtocolSoundWriter Writer
        {
            get { return _communication.Writer; }
        }
#if !USE_MCS
        /// <summary>
        ///
        /// </summary>
        protected int ReceiveTimeout
        {
            set
            {
                this._communication.ReceiveTimeout = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected int SendTimeout
        {
            set
            {
                this._communication.SendTimeout = value;
            }
        }
#endif
        //=====================================================================
        // 通信スレッド内部クラス
        //=====================================================================
        #region CommunicationThread
        /// <summary>
        /// 通信スレッド
        /// <para>
        /// スレッド実行によるソケット通信を行うクラスです。接続時にMCSチャンネルの
        /// 初期化も行います。
        /// </para>
        /// </summary>
        protected class CommunicationThread
        {
            //-----------------------------------------------------------------
            // 定数
            //-----------------------------------------------------------------
            /// <summary>MCSデフォルトタイムアウト(ms)</summary>
            private const int MCSDefaultTimeout = 5000;
            /// <summary>MCSチャンネルオープンタイムアウト(ms)</summary>
            private const int MCSChannelOpenTimeout = 5000;
            /// <summary>スレッド終了待ち時間(ms)</summary>
            private const int WaitForThreadExitTime = 100;

            //-----------------------------------------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public CommunicationThread(MCSConnection connection,
                string hostName, int port, uint channel)
            {
                Debug.Assert(connection != null);
                _connection = connection;
                _name = _connection + " " + hostName + ":" + port +
                    " channel 0x" + channel.ToString("x");
            }

            //-----------------------------------------------------------------
            /// <summary>
            /// 接続
            /// </summary>
#if USE_MCS
            public bool Connect( IPEndPoint ipEndPoint, uint channel )
#else
            public bool Connect(IComEndPoint comEndPoint, object port)
#endif
            {
#if USE_MCS
                Debug.Assert( ( _client == null ) && ( _writer == null ) &&
                              ( _reader == null ) && ( _thread == null ) );
#else
                Debug.Assert((this.comChannel == null) && (_writer == null) &&
                              (_reader == null) && (_thread == null));
#endif
                // TCP/IP接続
                try
                {
#if USE_MCS
                    //_client = new TcpClient();
                    _client = new TcpClient( ipEndPoint.AddressFamily );

                    _client.Connect( ipEndPoint );
                    _client.NoDelay = true;
                    //_client.SendTimeout    = MCSDefaultTimeout;
                    //_client.ReceiveTimeout = MCSDefaultTimeout;

                    _writer = new MCSProtocolSoundWriter( _client.GetStream() );
                    _reader = new MCSProtocolSoundReader( _client.GetStream() );
#else

                    IComChannel result = null;

                    using (var timer = new Timer(
                        state => comEndPoint.Disconnect(),
                        null,
                        3000,
                        Timeout.Infinite))
                    {
                        while (true)
                        {
                            lock (comEndPoint)
                            {
                                if (!comEndPoint.IsConnected)
                                {
                                    break;
                                }

                                result = comEndPoint.OpenChannel(port);
                            }

                            if (result != null)
                            {
                                break;
                            }
                            else
                            {
                                Thread.Sleep(10);
                            }
                        }
                    }

                    this.comChannel = result;
                    if (this.comChannel == null)
                    {
                        return false;
                    }

                    Stream stream = null;
                    if (this.comChannel is HostIOChannel)
                    {
                        HostIOChannel channel = this.comChannel as HostIOChannel;
                        stream = channel.TcpClient.GetStream();
                    }
                    else
                    {
                        stream = this.comChannel.GetStream();
                    }

                    this._writer = new MCSProtocolSoundWriter(stream);
                    this._reader = new MCSProtocolSoundReader(stream);
#endif
                }
                catch (Exception exception)
                {
                    ShowError(string.Format(
                        "TCP/IP接続に失敗しました。{0}\r\n{1}",
                        ToString(), exception.Message));
                    Disconnect(0);
                    return false;
                }

#if USE_MCS
                // MCSチャンネルを開く
                if ( !OpenMCSChannel( channel ) )
                {
                    Disconnect( 0 );
                    return false;
                }
#endif

                // スレッドの開始
                try
                {
                    _thread = new Thread(new ThreadStart(ThreadMain));
                    _thread.Name = ToString();
                    _thread.IsBackground = true;
                    _thread.Start();
                }
                catch (Exception exception)
                {
                    ShowError(string.Format(
                        "スレッドの開始に失敗しました。{0}\r\n{1}",
                        ToString(), exception.Message));
                    Disconnect(0);
                    return false;
                }
                return true;
            }

            /// <summary>
            /// スレッドのメインループ
            /// </summary>
            public void ThreadMain()
            {
                try
                {
                    while (!_wantToExit)
                    {
                        _connection.MainLoop();
                        // ループ毎にyieldする
                        Thread.Sleep(0);
                    }
                    Disconnect(Timeout.Infinite);
                }
                catch (Exception exception)
                {
                    // 終了フラグを立てていないときならば、例外処理をする
                    if (!_wantToExit)
                    {
                        string message = exception.Message;
                        if (exception.InnerException != null)
                        {
                            message += "\r\n" +
                                exception.InnerException.Message;
                        }
                        ShowError(string.Format(
                            "MCS通信に問題が発生しました。{0}\r\n{1}",
                            ToString(), message));
                    }
                    // 終了フラグを立てて、MCS接続全てを閉じる
                    _wantToExit = true;

                    if (null != exception.Message)
                    {
                        //MCSManager.LastError = exception.Message;
                        CommManager.Instance.LastError = exception.Message;
                    }
                    //MCSManager.Terminate();
                    CommManager.Instance.Terminate();
                }
            }

            /// <summary>
            /// 切断
            /// </summary>
            public void Disconnect(int timeout)
            {
                // スレッドの停止を待つ
                if (_thread != null)
                {
                    // 終了フラグが立っていない時に閉じられたら、
                    // 終了フラグを立てて待つ
                    if (!_wantToExit)
                    {
                        _wantToExit = true;
                        _thread.Join(timeout);

                        if (null != _thread)
                        {
                            _thread.Abort();
                        }

                    }
                    else
                    {
                        // すでに終了しようとしているので待つ
                        Thread.Sleep(WaitForThreadExitTime);
                    }
                    _thread = null;
                }

                // TCP/IP接続を閉じる
                if (_reader != null)
                {
                    _reader.Close();
                    _reader = null;
                }

                if (_writer != null)
                {
                    _writer.Close();
                    _writer = null;
                }

#if USE_MCS
                if ( _client != null )
                {
                    _client.Close();
                    _client = null;
                }
#else
                if (this.comChannel != null)
                {
                    this.comChannel.Close();
                    this.comChannel = null;
                }
#endif
            }
#if !USE_MCS
            /// <summary>
            ///
            /// </summary>
            public int ReceiveTimeout
            {
                set
                {
                    if (this.comChannel != null)
                    {
                        this.comChannel.ReceiveTimeout = value;
                    }
                }
            }

            /// <summary>
            ///
            /// </summary>
            public int SendTimeout
            {
                set
                {
                    if (this.comChannel != null)
                    {
                        this.comChannel.SendTimeout = value;
                    }
                }
            }
#endif
#if USE_MCS
            //-----------------------------------------------------------------
            // MCSチャンネル関連
            //-----------------------------------------------------------------
            #region MCSChannel
            // MCSのコマンド
            private enum MCSChannelCommand : uint
            {
                Open = 1,	// チャンネルを開く
            }

            // チャンネルリクエストフラグ
            [Flags()]
            private enum ChannelRequestFlags : uint
            {
                NoCheckConnect = 1 << 0,	// ターゲットとの接続状態に依存しない
            }

            // チャンネルリクエスト返り値
            private enum ChannelRequestResult : uint
            {
                Success,                // 成功
                ChannelProtocolError,   // MCSチャンネルプロトコルのエラーです。
                AlreadyRegistedChannel, // 既に登録されているチャンネルです。
                SystemReservedChannel,  // システムで予約しています。
                NotConnectedToViewer,   // ターゲットと接続されていません。
            }

            private static string[] _channelRequestResultString = new string[]
                {
                    "成功",
                    "MCSチャンネルプロトコルに問題がありました。",
                    "既に登録されているチャンネルを開こうとしました。",
                    "システムで予約しているチャンネルを開こうとしました。",
                    "MCSサーバがビューアと接続されていません。",
                };

            /// <summary>
            /// MCSチャンネルを開く
            /// </summary>
            private bool OpenMCSChannel( uint channel )
            {
                Console.WriteLine( "Open MCS channel = {0}", channel);

                // タイムアウトを設定する
                int swapReceiveTimeout = Client.ReceiveTimeout;
                int swapSendTimeout = Client.SendTimeout;
                Client.ReceiveTimeout = MCSChannelOpenTimeout;
                Client.SendTimeout = MCSChannelOpenTimeout;

                try {
                    // MCSチャンネルオープン要求を出す
                    const uint sizeofUint = 4;
                    const uint type = (uint)MCSChannelCommand.Open;
                    const uint size = sizeofUint * 2;
                    const uint flag = 0;
                    Writer.Write( type );
                    Writer.Write( size );
                    Writer.Write( channel );
                    Writer.Write( flag );
                    Writer.Flush();

                    // レスポンスを受け取る
                    uint chunkID = Reader.ReadUInt32();
                    uint messageSize = Reader.ReadUInt32();
                    if ( messageSize != sizeofUint ) { return false; }
                    uint result = Reader.ReadUInt32();
                    if ( (ChannelRequestResult)result !=
                        ChannelRequestResult.Success ) {
                        ShowError( string.Format(
                            "MCSチャンネルを開くのに失敗しました。{0}\r\n{1}",
                            ToString(), _channelRequestResultString[ result ] ) );
                        return false;
                    }
                }
                catch ( Exception exception ) {
                    ShowError( string.Format(
                        "MCSチャンネルを開くのに失敗しました。{0}\r\n{1}",
                        ToString(), exception.Message ) );
                    return false;
                }
                finally {
                    // タイムアウトの設定を元に戻す
                    Client.ReceiveTimeout = swapReceiveTimeout;
                    Client.SendTimeout = swapSendTimeout;
                }
                return true;
            }
            #endregion
#endif

            //-----------------------------------------------------------------
            // 公開メンバ
            //-----------------------------------------------------------------

            /// <summary>
            /// クライアント
            /// </summary>
#if USE_MCS
            public TcpClient Client { get { return _client; } }
#else
#if false
            public TcpClient Client
            {
                get
                {
                    return this.comChannel != null ? this.comChannel.tcpClient : null;
                }
            }
#endif
#endif

            /// <summary>
            /// リーダ
            /// </summary>
            public ProtocolSoundReader Reader { get { return _reader; } }

            /// <summary>
            /// ライタ
            /// </summary>
            public ProtocolSoundWriter Writer { get { return _writer; } }

            /// <summary>
            /// 文字列変換
            /// </summary>
            public override string ToString() { return _name; }

            /// <summary>
            /// 終了待ち
            /// </summary>
            public bool IsWantToExit { get { return _wantToExit; } }

            /// <summary>
            /// 切断中
            /// </summary>
            public bool DisconnectingFlag
            {
                get { return _disconnectingFlag; }
                set { _disconnectingFlag = value; }
            }

            //-----------------------------------------------------------------
            // 非公開メンバ
            //-----------------------------------------------------------------
            /// <summary>
            /// メッセージ
            /// </summary>
            protected void ShowMessage(string text)
            {
#if MCSDEBUG
                MCSManager.Console.WriteMCSMessage(text);
#endif
            }

            /// <summary>
            /// エラー
            /// </summary>
            protected void ShowError(string text)
            {
#if MCSDEBUG
                MCSManager.Console.WriteMCSError(text);
#endif
            }

            //-----------------------------------------------------------------
            // 接続
            private volatile MCSConnection _connection;
            // 名前
            private volatile string _name;

            // TCPクライアント
#if USE_MCS
            private volatile TcpClient _client = null;
#else
            private IComChannel comChannel = null;
#endif
            // ライタ
            private volatile ProtocolSoundWriter _writer = null;
            // リーダ
            private volatile ProtocolSoundReader _reader = null;
            // スレッド
            private volatile Thread _thread = null;
            // 終了フラグ
            private volatile bool _wantToExit = false;
            // 切断中フラグ
            private volatile bool _disconnectingFlag = false;
        }
        #endregion
    }

    //-----------------------------------------------------------------------
    /// <summary>
    /// ProtocolSoundReader
    /// short,ushort,int,uint,floatのエンディアンを変換し、読み込みます
    /// </summary>
    public class MCSProtocolSoundReader : ProtocolSoundReader
    {
        /// <summary>コンストラクタ</summary>
        public MCSProtocolSoundReader(Stream input) : base(input) { }
        /// <summary>コンストラクタ</summary>
        public MCSProtocolSoundReader(Stream input, Encoding encoding) :
            base(input, encoding)
        { }

        ///	<summary>short読み込み</summary>
        public override short ReadInt16()
        {
            return IPAddress.NetworkToHostOrder(base.ReadInt16());
            //return base.ReadInt16();
        }

        ///	<summary>ushort読み込み</summary>
        public override ushort ReadUInt16()
        {
#if true
            short result = (short)base.ReadUInt16();
            return (ushort)IPAddress.NetworkToHostOrder(result);
#else
            return (ushort)base.ReadUInt16();
#endif
        }

        ///	<summary>int読み込み</summary>
        public override int ReadInt32()
        {
#if true
            int result = (int)base.ReadInt32();
            return (int)IPAddress.NetworkToHostOrder(result);
#else
            return base.ReadInt32();
#endif
        }

        ///	<summary>uint読み込み</summary>
        public override uint ReadUInt32()
        {
#if true
            int result = (int)base.ReadUInt32();
            return (uint)IPAddress.NetworkToHostOrder(result);
#else
            return base.ReadUInt32();
#endif
        }

        ///	<summary>float読み込み</summary>
        public override float ReadSingle()
        {
#if true
            // ネットワークとエンディアンが違うことが前提になっています
            byte[] floatBytes = BitConverter.GetBytes(base.ReadSingle());
            Array.Reverse(floatBytes);
            return BitConverter.ToSingle(floatBytes, 0);
#else
            return base.ReadSingle();
#endif
        }
    }

    //-----------------------------------------------------------------------
    /// <summary>
    /// ProtocolSoundWriter
    /// short,ushort,int,uint,floatのエンディアンを変換し、書き込みます
    /// </summary>
    public class MCSProtocolSoundWriter : ProtocolSoundWriter
    {
        /// <summary>コンストラクタ</summary>
        public MCSProtocolSoundWriter() : base() { }
        /// <summary>コンストラクタ</summary>
        public MCSProtocolSoundWriter(Stream input) : base(input) { }
        /// <summary>コンストラクタ</summary>
        public MCSProtocolSoundWriter(Stream input, Encoding encoding) :
            base(input, encoding)
        { }

        ///	<summary>short書き出し</summary>
        public override void Write(short value)
        {
            base.Write(IPAddress.HostToNetworkOrder(value));
        }

        ///	<summary>ushort書き出し</summary>
        public override void Write(ushort value)
        {
            base.Write((ushort)IPAddress.HostToNetworkOrder((short)value));
            //base.Write(value);
        }

        ///	<summary>int書き出し</summary>
        public override void Write(int value)
        {
            base.Write(IPAddress.HostToNetworkOrder(value));
            //base.Write(value);
        }

        ///	<summary>uint書き出し</summary>
        public override void Write(uint value)
        {
            base.Write((uint)IPAddress.HostToNetworkOrder((int)value));
            //base.Write(value);
        }

        ///	<summary>float書き出し</summary>
        public override void Write(float value)
        {
#if true
            // ネットワークとエンディアンが違うことが前提になっています
            byte[] floatBytes = BitConverter.GetBytes(value);
            Array.Reverse(floatBytes);
            base.Write(BitConverter.ToSingle(floatBytes, 0));
#else
            base.Write(value);
#endif
        }
    }
}
