﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Nintendo.Alto.Foundation.Communications
{
    using System;
    using System.IO;
    using System.Net.Sockets;

    /// <summary>
    /// Win HostIO のチャンネルを制御するクラスです。
    /// </summary>
    public class HostIOWinChannel : IComChannel
    {
        private readonly int port;
        private readonly string portName;

        private TcpClient tcpClient;
        private int sendTimeout = 1000;
        private int receiveTimeout = 1000;
        private int keepAliveTimeout = 1000;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="port">ポート番号を指定します。</param>
        protected HostIOWinChannel(int port)
        {
            //Assertion.Argument.True(port > 0);
            this.port = port;
            this.portName = string.Empty;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="port">ポート番号を指定します。</param>
        /// <param name="portName">ポート名を指定します。</param>
        private HostIOWinChannel(int port, string portName)
        {
            //Assertion.Argument.True(port > 0);
            this.port = port;
            this.portName = portName;
        }

        /// <summary>
        /// デストラクタです。
        /// </summary>
        ~HostIOWinChannel()
        {
            this.Close();
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// ポート番号を取得します。
        /// </summary>
        public int Port
        {
            get { return this.port; }
        }

        /// <summary>
        /// ポート名を取得します。
        /// </summary>
        public string PortName
        {
            get { return this.portName; }
        }

        /// <summary>
        /// 開閉状態の有無を取得します。
        /// </summary>
        public bool IsOpened
        {
            get
            {
                return this.tcpClient == null ? false : this.tcpClient.Connected;
            }
        }

        /// <summary>
        /// 送信タイムアウトを取得または設定します。
        /// </summary>
        public int SendTimeout
        {
            get { return this.sendTimeout; }
            set
            {
                if (this.sendTimeout == value)
                {
                    return;
                }

                this.sendTimeout = value;

                if (this.tcpClient != null)
                {
                    this.tcpClient.SendTimeout = value;
                }
            }
        }

        /// <summary>
        /// 受信タイムアウトを取得または設定します。
        /// </summary>
        public int ReceiveTimeout
        {
            get { return this.receiveTimeout; }
            set
            {
                if (this.receiveTimeout == value)
                {
                    return;
                }

                this.receiveTimeout = value;

                if (this.tcpClient != null)
                {
                    this.tcpClient.ReceiveTimeout = value;
                }
            }
        }

        /// <summary>
        /// キープアライブタイムアウトを取得または設定します。
        /// </summary>
        public int KeepAliveTimeout
        {
            get { return this.keepAliveTimeout; }
            set
            {
                if (this.keepAliveTimeout == value)
                {
                    return;
                }

                this.keepAliveTimeout = value;
                this.ApplyKeepAliveTimeout();
            }
        }

        /// <summary>
        /// TCP クライアントを取得します。
        /// </summary>
        protected TcpClient TcpClient
        {
            get { return this.tcpClient; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 新しいチャンネルを開きます。
        /// </summary>
        /// <param name="port">ポート番号を指定します。</param>
        /// <param name="portName">ポート名を指定します。</param>
        public static IComChannel Open(int port, string portName)
        {
            var newChannel = new HostIOWinChannel(port, portName);

            try
            {
                newChannel.Open();
            }
            catch
            {
                newChannel.Dispose();
                throw;
            }

            return newChannel;
        }

        /// <summary>
        /// チャンネルを閉じます。
        /// </summary>
        public void Close()
        {
            TcpClient client = this.tcpClient;
            this.tcpClient = null;

            this.OnClosing();

            if (client != null)
            {
                client.Close();
            }

            this.OnClosed();
        }

        /// <summary>
        /// オブジェクトを破棄します。
        /// </summary>
        public void Dispose()
        {
            this.Close();
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// ストリームを取得します。
        /// </summary>
        public Stream GetStream()
        {
            return this.tcpClient != null ? this.tcpClient.GetStream() : null;
        }

        protected virtual void OnOpened() { }
        protected virtual void OnClosing() { }
        protected virtual void OnClosed() { }

        /// <summary>
        /// チャンネルを開きます。
        /// </summary>
        protected void Open()
        {
            // 接続先は "localhost" 固定。
            this.tcpClient = new TcpClient("127.0.0.1", this.port);
            //Ensure.Operation.NotNull(this.tcpClient);

            this.ApplyKeepAliveTimeout();

            // 送受信タイムアウトは TcpClient のデフォルト値を利用。
            this.SendTimeout = this.tcpClient.SendTimeout;
            this.ReceiveTimeout = this.tcpClient.ReceiveTimeout;
        }

        /// <summary>
        /// キープアライブタイムアウトを反映します。
        /// </summary>
        private void ApplyKeepAliveTimeout()
        {
            if (this.tcpClient == null)
            {
                return;
            }

            byte[] buffer = new byte[12];
            BitConverter.GetBytes(this.keepAliveTimeout == 0 ? 0 : 1).CopyTo(buffer, 0);    // スイッチ
            BitConverter.GetBytes(this.keepAliveTimeout).CopyTo(buffer, 4);                 // msec

            this.tcpClient.Client.IOControl(IOControlCode.KeepAliveValues, buffer, null);
        }
    }
}
