﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using NW4R.ProtocolSound;

namespace NintendoWare.SoundMaker.Preview.Htcs.Tool.Sound
{
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;

    internal class HtcsGetSeqVariablesPacket : ToolSoundPacket
    {
        private SeqVariableContainerCollection _variableSets;

        public HtcsGetSeqVariablesPacket(SeqVariableContainerCollection variableSets)
        {
            Debug.Assert(null != variableSets, "Variable set is null");
            _variableSets = variableSets;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_GET_SEQVARIABLES; }
        }

        /// <summary>
        /// データサイズ
        /// </summary>
        public override ushort Size
        {
            get { return (ushort)(ToolSoundItemContainer.StructSize + ItemsSize); }
        }

        private ushort ItemsSize
        {
            get
            {
                int size = 0;

                foreach (SeqVariableContainer item in _variableSets)
                {
                    size += item.GetWritableSize(true);
                }

                Debug.Assert(ushort.MaxValue >= size, "Value is over ushort.MaxValue");
                Debug.Assert(ushort.MinValue <= size, "Value is under ushort.MinValue");

                return (ushort)size;
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// パケットを作成して送信します。
        /// </summary>
        /// <param name="effectParam">エフェクトパラメータ</param>
        public static void Send(SeqVariableContainerCollection variables)
        {
            if (!CommManager.Instance.IsConnected) { return; }
            SendPacket(new HtcsGetSeqVariablesPacket(variables));
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// 送信パケットの圧縮
        /// </summary>
        public override bool Compress(ToolPacket nextPacket)
        {
            HtcsGetSeqVariablesPacket next = nextPacket as HtcsGetSeqVariablesPacket;
            if (null == next) { return false; }

            if (_variableSets.Count != next._variableSets.Count) { return false; }

            for (int i = 0; i < _variableSets.Count; i++)
            {

                if (_variableSets[i].ID != next._variableSets[i].ID) { return false; }

            }

            return true;
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            Debug.Fail("unexpected function call.");
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            //MemoryStream        memStream = new MemoryStream();
            //ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );


            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            //header.Write( memWriter );
            header.Write(writer);

            Debug.Assert(null != _variableSets, "Variable set is null");
            Debug.Assert(_variableSets.Count > 0, "Variable set count is zero");


            // シーケンス変数取得用ヘッダを出力する
            ToolSoundItemContainer container = new ToolSoundItemContainer();

            container.offsetForData = ToolSoundItemContainer.StructSize;
            container.itemCount = (uint)_variableSets.Count;
            container.itemSize = (uint)_variableSets[0].GetWritableSize(true);

            //container.Write( memWriter );
            container.Write(writer);

            // シーケンス変数取得パラメータを出力する
            foreach (SeqVariableContainer item in _variableSets)
            {
                //item.Write( memWriter, true );
                item.Write(writer, true);
            }

            // この１行は仮実装
            // Send用ToolPacket, Recv用ViewerPacketとして１つのクラスにまとめる
            Htcs.Viewer.HtcsGetSeqVariablesPacket.Received = false;

            //writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

            return header;
        }

        #endregion
    }

    internal class HtcsSetSeqVariablesPacket : ToolSoundPacket
    {
        private SeqVariableContainerCollection _variableSets;

        public HtcsSetSeqVariablesPacket(SeqVariableContainerCollection variableSets)
        {
            Debug.Assert(null != variableSets, "Variable set is null");
            _variableSets = variableSets;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_SET_SEQVARIABLES; }
        }

        /// <summary>
        /// データサイズ
        /// </summary>
        public override ushort Size
        {
            get { return (ushort)(ToolSoundItemContainer.StructSize + ItemsSize); }
        }

        private ushort ItemsSize
        {
            get
            {
                int size = 0;

                foreach (SeqVariableContainer item in _variableSets)
                {
                    size += item.GetWritableSize(false);
                }

                Debug.Assert(ushort.MaxValue >= size, "Valuei s over ushort.MaxValue");
                Debug.Assert(ushort.MinValue <= size, "Value is under ushort.MinValue");

                return (ushort)size;
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// パケットを作成して送信します。
        /// </summary>
        /// <param name="effectParam">エフェクトパラメータ</param>
        public static void Send(SeqVariableContainerCollection variables)
        {
            if (!CommManager.Instance.IsConnected) { return; }
            SendPacket(new HtcsSetSeqVariablesPacket(variables));
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// 送信パケットの圧縮
        /// </summary>
        public override bool Compress(ToolPacket nextPacket)
        {
            return false;
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            // TOOL_SOUND_SET_SEQVARIABLES は Write のみ
            Debug.Assert(false, "Don't call this method");
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            //MemoryStream        memStream = new MemoryStream();
            //ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            //header.Write( memWriter );
            header.Write(writer);

            Debug.Assert(null != _variableSets, "Variable set is null");
            Debug.Assert(_variableSets.Count > 0, "Variable set count is zero");


            // シーケンス変数取得用ヘッダを出力する
            ToolSoundItemContainer container = new ToolSoundItemContainer();

            container.offsetForData = ToolSoundItemContainer.StructSize;
            container.itemCount = (uint)_variableSets.Count;
            container.itemSize = (uint)_variableSets[0].GetWritableSize(false);

            //container.Write( memWriter );
            container.Write(writer);

            // シーケンス変数取得パラメータを出力する
            foreach (SeqVariableContainer item in _variableSets)
            {
                //item.Write( memWriter, false );
                item.Write(writer, false);
            }


            //writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

            return header;
        }

        #endregion
    }
}
