﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using NW4R.ProtocolSound;

namespace NintendoWare.Preview.HIO
{
    using Communications;
    using NintendoWare.Preview.HIO.Sound;

    /// <summary>
    ///
    /// </summary>
    public class HIOClient : IDisposable
    {
        private class HIOStream : Stream
        {
            private HIOClient client = null;

            ///
            public HIOStream(HIOClient client)
            {
                this.client = client;
            }

            /// <summary>
            ///
            /// </summary>
            public override int Read(byte[] buffer, int offset, int count)
            {
                throw new NotImplementedException();
            }

            public override void Write(byte[] buffer, int offset, int count)
            {
                // TODO : offset 対応。
                HostIO.Send(new MemoryStream(buffer, offset, count), this.client.channel);
            }

            public override void Flush()
            {
                // TODO: 未実装
            }

            public override long Position
            {
                get
                {
                    //Debug.Assert(false);
                    return 0;
                }
                set
                {
                    Debug.Assert(false, "Can not set value");
                }
            }

            public override long Length
            {
                get
                {
                    Debug.Assert(false, "Can not get value");
                    return 0;
                }
            }

            public override bool CanRead
            {
                get
                {
                    return true;
                }
            }

            public override bool CanWrite
            {
                get
                {
                    return true;
                }
            }

            public override bool CanSeek
            {
                get
                {
                    return false;
                }
            }

            public override long Seek(long offset, SeekOrigin origin)
            {
                Debug.Assert(false, "Can not seek");
                return 0;
            }

            public override void SetLength(long value)
            {
                Debug.Assert(false, "Can not set value");
            }
        }

        ///
        private int channel = -1;
        private HIOStream stream = null;

        ///
        public HIOClient()
        {
            this.stream = new HIOStream(this);
        }

        ///
        public void Dispose()
        {
        }

        ///
        public Stream GetStream()
        {
            return this.stream;
        }

        ///
        public bool Connect(int channel)
        {
            this.channel = channel;

            try
            {
                HostIO.Connect();
                HostIO.OpenSerialIO(this.channel);
                HostIO.ConnectSerialIO(this.channel);
            }
            catch
            {
                return false;
            }
            return true;
        }

        ///
        public void Disconnect()
        {
            if (channel < 0)
            {
                return;
            }

            try
            {
                HostIO.DisconnectSerialIO(this.channel);
                HostIO.Disconnect();
            }
            catch
            {
            }
        }
    }

    /// <summary>
    /// HIO接続
    /// <para>
    /// ツール接続、ビューア接続、Ping接続の親クラスです。MCS接続の共通部分を
    /// 実装しています。実際に別スレッドで通信を行う部分の実装は、内部クラスの
    /// CommunicationThreadクラスにて行っています。
    /// </para>
    /// </summary>
    public abstract class HIOConnection : CommConnection
    {
        //---------------------------------------------------------------------
        // 定数
        //---------------------------------------------------------------------
#if false
        protected const int _baseChannel =
        (int)(((uint)'S' << 8) + (uint)'D') + 0x8000;
#endif

        //---------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public HIOConnection()
        {
        }

        /// <summary>
        /// メインループ
        /// </summary>
        public abstract void MainLoop();

        /// <summary>
        /// チャンネルの取得
        /// </summary>
        public abstract int Channel { get; }

        /// <summary>
        /// 文字列変換
        /// </summary>
        public override string ToString()
        {
            return GetType().Name;
        }

        //---------------------------------------------------------------------
        // 接続関係
        //---------------------------------------------------------------------
        /// <summary>
        /// 接続
        /// </summary>
        public override bool Connect()
        {
            Debug.Assert(_communication == null, "Communication is null");
            _communication = new CommunicationThread(this, Channel);
            if (!_communication.Connect(Channel))
            {
                _communication = null;
                return false;
            }
            ShowMessage("通信開始 : " + _communication);
            return true;
        }


        /// <summary>
        /// 接続中か
        /// </summary>
        public override bool IsConnected { get { return (_communication != null); } }

        /// <summary>
        /// 切断
        /// </summary>
        public override void Disconnect()
        {
            Disconnect(Timeout.Infinite);
        }

        /// <summary>
        /// 切断
        /// </summary>
        public virtual void Disconnect(int timeout)
        {
            if (!IsConnected) { return; }
            string name = _communication.ToString();
            _communication.Disconnect(timeout);
            _communication = null;
            ShowMessage("通信停止 : " + name);
        }

        /// <summary>
        /// 終了待ち
        /// </summary>
        public bool IsWantToExit
        {
            get
            {
                if (_communication.DisconnectingFlag) return true;
                if (!IsConnected) return false;
                return _communication.IsWantToExit;
            }
        }

        /// <summary>
        /// 切断中
        /// </summary>
        public bool DisconnectingFlag
        {
            get
            {
                if (_communication != null)
                {
                    return _communication.DisconnectingFlag;
                }
                else
                {
                    return false;
                }
            }
            set
            {
                if (_communication != null)
                {
                    _communication.DisconnectingFlag = value;
                }
            }
        }


        //---------------------------------------------------------------------
        // 非公開メンバ
        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージ
        /// </summary>
        protected void ShowMessage(string format, params object[] args)
        {
            //MCSManager.Console.WriteMCSMessage(format, args);
            CommManager.Instance.Console.WriteMessage(format, args);
        }

        /// <summary>
        /// エラー
        /// </summary>
        protected void ShowError(string format, params object[] args)
        {
            //MCSManager.Console.WriteMCSError(format, args);
            CommManager.Instance.Console.WriteMessage(format, args);
        }

        //---------------------------------------------------------------------
#if false
        /// <summary>
        /// クライアント
        /// </summary>
        protected TcpClient Client
        {
            get{ return _communication.Client; }
        }
#endif

        /// <summary>
        /// リーダ
        /// </summary>
        protected ProtocolSoundReader Reader
        {
            get { return _communication.Reader; }
        }

        /// <summary>
        /// ライタ
        /// </summary>
        protected ProtocolSoundWriter Writer
        {
            get { return _communication.Writer; }
        }

        //---------------------------------------------------------------------
        // 通信
        private volatile CommunicationThread _communication = null;

        //=====================================================================
        // 通信スレッド内部クラス
        //=====================================================================
        #region CommunicationThread
        /// <summary>
        /// 通信スレッド
        /// <para>
        /// スレッド実行によるソケット通信を行うクラスです。接続時にMCSチャンネルの
        /// 初期化も行います。
        /// </para>
        /// </summary>
        protected class CommunicationThread
        {
            //-----------------------------------------------------------------
            // 定数
            //-----------------------------------------------------------------
            /// <summary>MCSデフォルトタイムアウト(ms)</summary>
            /// private const int MCSDefaultTimeout = 5000;

            /// <summary>MCSチャンネルオープンタイムアウト(ms)</summary>
            private const int MCSChannelOpenTimeout = 5000;

            /// <summary>スレッド終了待ち時間(ms)</summary>
            private const int WaitForThreadExitTime = 100;

            ///--------------------------------
            ///
            private static string[] _channelRequestResultString = new string[]
                {
                    "成功",
                    "MCSチャンネルプロトコルに問題がありました。",
                    "既に登録されているチャンネルを開こうとしました。",
                    "システムで予約しているチャンネルを開こうとしました。",
                    "MCSサーバがビューアと接続されていません。",
                };

            //-----------------------------------------------------------------
            // 接続
            private volatile HIOConnection _connection;

            // 名前
            private volatile string _name = "CommunicationThread";

            // ライタ
            private volatile ProtocolSoundWriter _writer = null;
            // リーダ
            private volatile ProtocolSoundReader _reader = null;
            // スレッド
            private volatile Thread _thread = null;
            // 終了フラグ
            private volatile bool _wantToExit = false;
            // 切断中フラグ
            private volatile bool _disconnectingFlag = false;

            ///
            //private int channel = -1;

            ///
            private HIOClient client = null;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public CommunicationThread(HIOConnection connection, int channel)
            {
                Debug.Assert(connection != null, "Connection is null");
                _connection = connection;
                //this.channel = channel;
            }

            /// <summary>
            /// 接続
            /// </summary>
            public bool Connect(int channel)
            {
                this.client = new HIOClient();
                if (this.client.Connect(channel) == false)
                {
                    return false;
                }

                this._reader = new HioProtocolSoundReader(this.client.GetStream());
                this._writer = new HioProtocolSoundWriter(this.client.GetStream());
                //this._writer = new ProtocolSoundWriterImpersonation(this.client.GetStream());

                // スレッドの開始
                try
                {
                    _thread = new Thread(new ThreadStart(ThreadMain));
                    _thread.Name = ToString();
                    _thread.IsBackground = true;
                    _thread.Start();
                }
                catch (Exception exception)
                {
                    ShowError("スレッドの開始に失敗しました。{0}\r\n{1}",
                              ToString(), exception.Message);
                    Disconnect(0);
                    return false;
                }
                return true;
            }

            /// <summary>
            /// スレッドのメインループ
            /// </summary>
            public void ThreadMain()
            {
                try
                {
                    while (!_wantToExit)
                    {
                        _connection.MainLoop();
                        // ループ毎にyieldする
                        Thread.Sleep(0);
                    }
                    Disconnect(Timeout.Infinite);
                }
                catch (Exception exception)
                {
                    // 終了フラグを立てていないときならば、例外処理をする
                    if (!_wantToExit)
                    {
                        string message = exception.Message;
                        if (exception.InnerException != null)
                        {
                            message += "\r\n" +
                                exception.InnerException.Message;
                        }
                        ShowError("MCS通信に問題が発生しました。{0}\r\n{1}",
                                  ToString(), message);
                    }
                    // 終了フラグを立てて、MCS接続全てを閉じる
                    _wantToExit = true;

                    if (null != exception.Message)
                    {
                        CommManager.Instance.LastError = exception.Message;
                    }
                    CommManager.Instance.Terminate();
                }
            }

            /// <summary>
            /// 切断
            /// </summary>
            public void Disconnect(int timeout)
            {
                // スレッドの停止を待つ
                if (_thread != null)
                {
                    // 終了フラグが立っていない時に閉じられたら、
                    // 終了フラグを立てて待つ
                    if (!_wantToExit)
                    {
                        _wantToExit = true;
                        _thread.Join(timeout);

                        if (null != _thread)
                        {
                            _thread.Abort();
                        }

                    }
                    else
                    {
                        // すでに終了しようとしているので待つ
                        Thread.Sleep(WaitForThreadExitTime);
                    }
                    _thread = null;
                }

                /// HIO接続を閉じる
                if (this._reader != null)
                {
                    this._reader.Close();
                    this._reader = null;
                }
                if (this._writer != null)
                {
                    this._writer.Close();
                    this._writer = null;
                }

                if (this.client != null)
                {
                    this.client.Disconnect();
                }
            }

            //-----------------------------------------------------------------
            // MCSチャンネル関連
            //-----------------------------------------------------------------
            #region MCSChannel

            // MCSのコマンド
            private enum MCSChannelCommand : uint
            {
                Open = 1,   // チャンネルを開く
            }

            // チャンネルリクエストフラグ
            [Flags()]
            private enum ChannelRequestFlags : uint
            {
                NoCheckConnect = 1 << 0,    // ターゲットとの接続状態に依存しない
            }

            // チャンネルリクエスト返り値
            private enum ChannelRequestResult : uint
            {
                Success,                // 成功
                ChannelProtocolError,   // MCSチャンネルプロトコルのエラーです。
                AlreadyRegistedChannel, // 既に登録されているチャンネルです。
                SystemReservedChannel,  // システムで予約しています。
                NotConnectedToViewer,   // ターゲットと接続されていません。
            }

#if false
            /// <summary>
            /// MCSチャンネルを開く
            /// </summary>
            private bool OpenMCSChannel(uint channel)
            {
                // タイムアウトを設定する
                int swapReceiveTimeout = Client.ReceiveTimeout;
                int swapSendTimeout = Client.SendTimeout;
                Client.ReceiveTimeout = MCSChannelOpenTimeout;
                Client.SendTimeout = MCSChannelOpenTimeout;

                try
                {
                    // MCSチャンネルオープン要求を出す
                    const uint sizeofUint = 4;
                    const uint type = (uint)MCSChannelCommand.Open;
                    const uint size = sizeofUint * 2;
                    const uint flag = 0;
                    Writer.Write(type);
                    Writer.Write(size);
                    Writer.Write(channel);
                    Writer.Write(flag);
                    Writer.Flush();

                    // レスポンスを受け取る
                    uint chunkID = Reader.ReadUInt32();
                    uint messageSize = Reader.ReadUInt32();
                    if(messageSize != sizeofUint){ return false; }
                    uint result = Reader.ReadUInt32();
                    if((ChannelRequestResult)result !=
                       ChannelRequestResult.Success)
                    {
                        ShowError(
                                  "MCSチャンネルを開くのに失敗しました。{0}\r\n{1}",
                                  ToString(), _channelRequestResultString[result]);
                        return false;
                    }
                }
                catch(Exception exception)
                {
                    ShowError(
                              "MCSチャンネルを開くのに失敗しました。{0}\r\n{1}",
                              ToString(), exception.Message);
                    return false;
                }
                finally
                {
                    // タイムアウトの設定を元に戻す
                    Client.ReceiveTimeout = swapReceiveTimeout;
                    Client.SendTimeout = swapSendTimeout;
                }
                return true;
            }
#endif
            #endregion

            //-----------------------------------------------------------------
            // 公開メンバ
            //-----------------------------------------------------------------
            /// <summary>
            /// クライアント
            /// </summary>
            //public TcpClient Client{ get{ return _client; } }

            /// <summary>
            /// リーダ
            /// </summary>
            public ProtocolSoundReader Reader { get { return _reader; } }

            /// <summary>
            /// ライタ
            /// </summary>
            public ProtocolSoundWriter Writer { get { return _writer; } }

            /// <summary>
            /// 文字列変換
            /// </summary>
            public override string ToString() { return _name; }

            /// <summary>
            /// 終了待ち
            /// </summary>
            public bool IsWantToExit { get { return _wantToExit; } }

            /// <summary>
            /// 切断中
            /// </summary>
            public bool DisconnectingFlag
            {
                get { return _disconnectingFlag; }
                set { _disconnectingFlag = value; }
            }
            //-----------------------------------------------------------------
            // 非公開メンバ
            //-----------------------------------------------------------------
            /// <summary>
            /// メッセージ
            /// </summary>
            protected void ShowMessage(string format, params object[] args)
            {
                CommManager.Instance.Console.WriteMessage(format, args);
            }

            /// <summary>
            /// エラー
            /// </summary>
            protected void ShowError(string format, params object[] args)
            {
                CommManager.Instance.Console.WriteError(format, args);
            }
        }
        #endregion

    }
}
