﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.FileFormats
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Runtime.InteropServices;
    using System.Text;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMakerPlugin;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.Collections;

    /// <summary>
    /// SeqFileManager の概要の説明です。
    /// </summary>
    public class SeqFileManager
    {
        private static SeqFileManager InstanceInternal = new SeqFileManager();

        private Dictionary<string, SeqFileInfo> seqFilesSmf = new Dictionary<string, SeqFileInfo>();
        private Dictionary<string, SeqFileInfo> seqFilesText = new Dictionary<string, SeqFileInfo>();
        private Dictionary<string, SeqFileInfo> seqFilesBin = new Dictionary<string, SeqFileInfo>();

        private SeqFileManager() { }

        public static SeqFileManager Instance
        {
            get { return SeqFileManager.InstanceInternal; }
        }

        public SeqFileInfo LoadFile(string filePath, FileType fileType)
        {
            Dictionary<string, SeqFileInfo> seqFiles;
            switch (fileType)
            {
                case FileType.Smf:
                    seqFiles = seqFilesSmf;
                    break;
                case FileType.Text:
                    seqFiles = seqFilesText;
                    break;
                case FileType.Binary:
                    seqFiles = seqFilesBin;
                    break;
                default:
                    return null;
            }

            SeqFileInfo seqFileInfo;
            string fullPath = filePath.GetFullPath();
            if (seqFiles.ContainsKey(fullPath))
            {
                seqFileInfo = seqFiles[fullPath];

                if (seqFileInfo.IsUpdateFile())
                {
                    if (!seqFileInfo.Open(filePath, fileType))
                    {
                        seqFiles.Remove(fullPath);
                        return null;
                    }
                }
            }
            else
            {
                seqFileInfo = new SeqFileInfo();
                if (!seqFileInfo.Open(filePath, fileType))
                {
                    return null;
                }
                seqFiles[fullPath] = seqFileInfo;
            }

            return seqFileInfo;
        }

        public void Reload()
        {
            Reload(seqFilesSmf);
            Reload(seqFilesText);
            Reload(seqFilesBin);
        }

        private void Reload(Dictionary<string, SeqFileInfo> seqFiles)
        {
            foreach (SeqFileInfo seqFileInfo in seqFiles.Values)
            {
                if (seqFileInfo.IsUpdateFile())
                {
                    if (!seqFileInfo.Open(seqFileInfo.FilePath, seqFileInfo.FileType))
                    {
                        seqFiles.Remove(seqFileInfo.FilePath);
                    }
                }
            }
        }

        public enum FileType
        {
            Smf,
            Text,
            Binary
        }

        public class SeqFileInfo : IDisposable
        {
            private Dictionary<string, DateTime> dependFiles = new Dictionary<string, DateTime>();
            private string filePath;
            private FileType fileType;
            private byte[] seqData;
            private GCHandle seqDataHandle;
            //private ISequenceSoundBinaryReader sequenceSoundBinaryReader = new SequenceSoundBinaryReader();
            private ISequenceSoundBinaryReader sequenceSoundBinaryReader = SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.CreateSequenceSoundBinaryReader();

            public IKeyedList<string, LabelInfo> LabelInfoList
            {
                get { return sequenceSoundBinaryReader.LabelInfos; }
            }

            public string FilePath
            {
                get { return filePath; }
            }

            public SeqFileManager.FileType FileType
            {
                get { return fileType; }
            }

            public bool Open(string filePath, FileType fileType)
            {
                if (fileType == FileType.Smf)
                {
                    // #include埋め込み対応のため、
                    // SMFファイルを起点して、一時ファイルの作成を試みる
                    // ダメなら、Path.GetTempFileName()を使う
                    string textFileName = Path.Combine(
                        Path.GetDirectoryName(filePath),
                        Path.GetRandomFileName()
                    );

                    if (File.Exists(textFileName))
                    {
                        textFileName = Path.GetTempFileName();
                    }
                    else
                    {
                        try
                        {
                            // 作成可能かトライ
                            using (FileStream fs = File.Create(textFileName))
                            {
                            }
                        }
                        catch
                        {
                            textFileName = Path.GetTempFileName();
                        }
                    }
                    string binaryFileName = Path.GetTempFileName();
                    try
                    {
                        if (!ConvertSmfFile(filePath, textFileName))
                        {
                            return false;
                        }

                        SequenceSoundTextLexer lexer = new SequenceSoundTextLexer(textFileName);
                        while (lexer.ReadToken() != SequenceSoundTextLexer.Token.End)
                        {
                            // 依存ファイル収集のため空回し
                        }
                        dependFiles.Clear();
                        foreach (string path in lexer.DependFiles)
                        {
                            dependFiles[path] = File.GetLastWriteTime(path);
                        }
                        lexer.Dispose();
                        lexer = null;

                        dependFiles[filePath] = File.GetLastWriteTime(filePath);
                        dependFiles.Remove(textFileName);

                        if (!ConvertTextSeqFile(textFileName, binaryFileName))
                        {
                            return false;
                        }

                        this.LoadBinaryFile(binaryFileName, out seqData);
                    }
                    finally
                    {
                        File.Delete(textFileName);
                        File.Delete(binaryFileName);
                    }
                }
                else if (fileType == FileType.Text)
                {
                    string binaryFileName = Path.GetTempFileName();
                    try
                    {
                        SequenceSoundTextLexer lexer = new SequenceSoundTextLexer(filePath);
                        while (lexer.ReadToken() != SequenceSoundTextLexer.Token.End)
                        {
                            // 依存ファイル収集のため空回し
                        }
                        dependFiles.Clear();
                        foreach (string path in lexer.DependFiles)
                        {
                            dependFiles[path] = File.GetLastWriteTime(path);
                        }
                        lexer.Dispose();
                        lexer = null;

                        if (!ConvertTextSeqFile(filePath, binaryFileName))
                        {
                            return false;
                        }

                        this.LoadBinaryFile(binaryFileName, out seqData);
                    }
                    finally
                    {
                        File.Delete(binaryFileName);
                    }
                }
                else if (fileType == FileType.Binary)
                {
                    dependFiles[filePath] = File.GetLastWriteTime(filePath);
                    this.LoadBinaryFile(filePath, out seqData);
                }
                else
                {
                    return false;
                }

                this.filePath = filePath;
                this.fileType = fileType;

                seqDataHandle = GCHandle.Alloc(seqData, GCHandleType.Pinned);

                return true;
            }

            public IntPtr GetSeqDataPtr()
            {
                return seqDataHandle.AddrOfPinnedObject();
            }

            public bool IsUpdateFile()
            {
                foreach (KeyValuePair<string, DateTime> kvp in dependFiles)
                {
                    DateTime time = File.GetLastWriteTime(kvp.Key);
                    if (time != kvp.Value)
                    {
                        return true;
                    }
                }

                return false;
            }

            public void Dispose()
            {
                Dispose(true);
                // Take yourself off the Finalization queue
                // to prevent finalization code for this object
                // from executing a second time.
                GC.SuppressFinalize(this);
            }

            protected virtual void Dispose(bool disposing)
            {
                if (seqDataHandle.IsAllocated)
                {
                    seqDataHandle.Free();
                }
            }

            private static bool ConvertSmfFile(string smfFilePath, string outFilePath)
            {
                string arguments =
                    "-o \"" + outFilePath + "\" \"" + smfFilePath + "\"";

                StringBuilder sb = new System.Text.StringBuilder();

                try
                {
                    Process process = new Process();

                    process.StartInfo.FileName =
                        ApplicationBase.Instance.Traits.ConversionTraits.SmfConverterPath;
                    process.StartInfo.Arguments = arguments;
                    process.StartInfo.CreateNoWindow = true;
                    process.StartInfo.UseShellExecute = false;
                    process.StartInfo.RedirectStandardError = true;

                    process.ErrorDataReceived += delegate (object sender, DataReceivedEventArgs e)
                    {
                        if (e.Data != null)
                        {
                            sb.AppendLine(e.Data);
                        }
                    };

                    process.Start();
                    process.WaitForExit();

                    if (process.ExitCode != 0)
                    {
                        throw new ApplicationException(MessageResource.Message_FailedToConvertTextSequence
                                                         + "\n\n" + sb.ToString());
                    }

                    return process.ExitCode == 0;
                }
                catch
                {
                    return false;
                }
            }

            private static bool ConvertTextSeqFile(string textFilePath, string outFilePath)
            {
                string arguments =
                    "-o \"" + outFilePath + "\" \"" + textFilePath + "\"";

                StringBuilder sb = new System.Text.StringBuilder();

                try
                {
                    Process process = new Process();

                    process.StartInfo.FileName =
                        ApplicationBase.Instance.Traits.ConversionTraits.SequenceSoundTextConverterPath;
                    process.StartInfo.Arguments = arguments;
                    process.StartInfo.CreateNoWindow = true;
                    process.StartInfo.UseShellExecute = false;
                    process.StartInfo.RedirectStandardError = true;

                    process.ErrorDataReceived += delegate (object sender, DataReceivedEventArgs e)
                    {
                        if (e.Data != null)
                        {
                            sb.AppendLine(e.Data);
                        }
                    };

                    process.Start();
                    process.WaitForExit();

                    if (process.ExitCode != 0)
                    {
                        throw new ApplicationException(MessageResource.Message_FailedToConvertTextSequence
                                                         + "\n\n" + sb.ToString());
                    }

                    return process.ExitCode == 0;
                }
                catch
                {
                    return false;
                }
            }

            private void LoadBinaryFile(string filePath, out byte[] data)
            {
                Assertion.Argument.NotNull(filePath);

                using (FileStream stream = File.Open(filePath, FileMode.Open, FileAccess.Read, FileShare.Read))
                {
                    this.sequenceSoundBinaryReader.Parse(
                        LittleEndianBinaryReader.Create(stream)
                        );
                }

                data = File.ReadAllBytes(filePath);
            }
        }
    }
}



