﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Configurations
{
    using System;
    using System.IO;
    using System.Xml;
    using System.Xml.Schema;
    using NintendoWare.SoundFoundation;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Resources;

    /// <summary>
    /// アプリケーション設定を格納します。
    /// </summary>
    public class SoundProjectConfigurationCommon : SoundProjectConfiguration
    {
        private const string Platform = Platforms.Any.PlatformName;
        private const string Version = "1.4.0.0";

        private Schemas.XmlPreviewCommon xmlPreviewCommon;
        private Schemas.XmlSoundProjectConfigurationCommon xmlSoundProjectCommon;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public SoundProjectConfigurationCommon() : base(Platform, Version) { }

        /// <summary>
        /// Revolution 向けプレビュー設定を取得します。
        /// </summary>
        public Schemas.XmlPreviewCommon PreviewCommon
        {
            get { return this.xmlPreviewCommon; }
        }

        public Schemas.XmlSoundProjectConfigurationCommon SoundProjectCommon
        {
            get { return this.xmlSoundProjectCommon; }
        }

        protected override string RootElementName
        {
            get { return "SoundProjectConfiguration"; }
        }

        /// <summary>
        /// リストのデフォルト設定を読み込みます。
        /// </summary>
        /// <returns>リスト設定。</returns>
        public override XmlLists LoadListsDefaultInternal()
        {
            return ReadConfigurationPart<XmlLists>(
                                FileResourcesProxy.SoundProjectConfigurationListsDefaultCommon,
                                CreateReaderSettings());
        }

        /// <summary>
        /// デフォルト設定を読み込みます。
        /// </summary>
        /// <param name="settings">XmlReader 設定。</param>
        /// <param name="forced">現在の設定を全て破棄する場合は true、読み込み済みの設定を残す場合は false。</param>
        protected override void LoadDefaultsInternal(XmlReaderSettings settings, bool forced)
        {
            base.LoadDefaultsInternal(settings, forced);

            if (forced || null == this.xmlPreviewCommon)
            {
                this.xmlPreviewCommon = ReadConfigurationPart<Schemas.XmlPreviewCommon>(
                                            FileResourcesProxy.PreviewConfigurationDefaultCommon, settings);

            }
            if (forced || null == this.xmlSoundProjectCommon)
            {
                this.xmlSoundProjectCommon = ReadConfigurationPart<Schemas.XmlSoundProjectConfigurationCommon>(FileResourcesProxy.SoundProjectConfigurationDefaultCommon, settings);

            }
        }

        /// <summary>
        /// スキーマセットを作成します。
        /// </summary>
        /// <returns>スキーマセット。</returns>
        protected override XmlSchemaSet CreateSchemaSet()
        {
            XmlSchemaSet schemas = base.CreateSchemaSet();
            schemas.Add(XmlSchema.Read(new StringReader(FileResourcesProxy.SchemaOptionConfigurationCommon), null));

            return schemas;
        }

        /// <summary>
        /// コンフィギュレーション要素を読み込みます。
        /// </summary>
        /// <param name="xmlElement">コンフィギュレーション要素。</param>
        /// <param name="basePath">基準パス。</param>
        protected override void ReadConfigurationElement(XmlElement xmlElement, string basePath)
        {
            if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

            switch (xmlElement.LocalName)
            {
                case "PreviewCommon":
                    this.xmlPreviewCommon =
                        ReadConfigurationPart<Schemas.XmlPreviewCommon>(xmlElement.OuterXml);
                    break;

                case "SoundProjectConfigurationCommon":
                    this.xmlSoundProjectCommon =
                        ReadConfigurationPart<Schemas.XmlSoundProjectConfigurationCommon>(xmlElement.OuterXml);
                    break;

                default:
                    base.ReadConfigurationElement(xmlElement, basePath);
                    break;
            }
        }

        /// <summary>
        /// コンフィギュレーション要素を書き出します。
        /// </summary>
        /// <param name="reader">XmlReader。</param>
        /// <param name="basePath">基準パス。</param>
        protected override void WriteConfigurationElements(XmlWriter writer, string basePath)
        {
            base.WriteConfigurationElements(writer, basePath);

            WriteConfigurationPart<Schemas.XmlPreviewCommon>(writer, this.xmlPreviewCommon);

            WriteConfigurationPart<Schemas.XmlSoundProjectConfigurationCommon>(writer, this.xmlSoundProjectCommon);
        }

        /// <summary>
        /// コンフィギュレーションルートを書き出します。
        /// </summary>
        /// <param name="writer">XmlWriter。</param>
        protected override void WriteStartConfigurationRoot(XmlWriter writer)
        {
            base.WriteStartConfigurationRoot(writer);

            writer.WriteAttributeString("xmlns", "nwrsp", null,
                                        "NintendoWare.SoundMaker.PreviewConfiguration");

            writer.WriteAttributeString("xmlns", "nwspc", null,
                                        "NintendoWare.SoundMaker.SoundProjectConfigurationCommon");
        }

        /// <summary>
        /// 読み込んだコンフィギュレーションを検証します。
        /// </summary>
        /// <returns>読み込んだ結果が正常な場合は true、それ以外の場合は false。</returns>
        protected override bool Validate()
        {
            if (null == this.xmlPreviewCommon.System)
            {
                this.xmlPreviewCommon.System = new Schemas.XmlPreviewCommonSystem();
            }
            if (null == this.xmlPreviewCommon.SequenceVariable)
            {
                this.xmlPreviewCommon.SequenceVariable = new Schemas.XmlPreviewCommonSequenceVariable();
            }
            if (null == this.xmlPreviewCommon.OutputWaveFile)
            {
                this.xmlPreviewCommon.OutputWaveFile = new Schemas.XmlPreviewCommonOutputWaveFile();
            }

            return base.Validate();
        }
    }
}
