﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareReport;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// Ctr サウンドプロジェクトドキュメントをサウンドプロジェクトドキュメントに変換します。
    /// </summary>
    public class SoundProjectCtrToSoundProjectDocumentConverter : IDocumentConverter
    {
        private readonly SoundIntermediateOutputTraits intermediateOutputTraits;
        private IObjectFactory<Type, Component> componentFactory;

        public SoundProjectCtrToSoundProjectDocumentConverter(
            SoundIntermediateOutputTraits intermediateOutputTraits)
            : this(intermediateOutputTraits, new SoundProjectComponentFactory())
        {
        }

        public SoundProjectCtrToSoundProjectDocumentConverter(
            SoundIntermediateOutputTraits intermediateOutputTraits,
            IObjectFactory<Type, Component> componentFactory)
        {
            Ensure.Argument.NotNull(intermediateOutputTraits);
            Ensure.Argument.NotNull(componentFactory);

            this.intermediateOutputTraits = intermediateOutputTraits;
            this.componentFactory = componentFactory;
        }

        public string InputDocumentTypeName
        {
            get { return Platforms.Ctr.SoundProjectDocument; }
        }

        public string OutputDocumentTypeName
        {
            get { return this.intermediateOutputTraits.SoundProjectDocumentTypeName; }
        }

        public void Convert(Document inputDocument, Document outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);
            this.ConvertInternal(
                inputDocument as SoundProjectDocument,
                outputDocument as SoundProjectDocument);
        }

        private void ConvertInternal(SoundProjectDocument inputDocument, SoundProjectDocument outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);

            outputDocument.Project = this.ConvertSoundProject
                (inputDocument.Project, inputDocument.Resource.Key);

            if (inputDocument.Resource is FileResource)
            {
                outputDocument.Resource = new FileResource(
                    Path.ChangeExtension(
                        inputDocument.Resource.Key,
                        this.intermediateOutputTraits.SoundProjectFileExtension
                        )
                    );
            }
        }

        private SoundProject ConvertSoundProject(SoundProject source, string filePath)
        {
            Assertion.Argument.NotNull(source);

            SoundProject dest = this.componentFactory.Create(typeof(SoundProject)) as SoundProject;
            Ensure.Operation.ObjectNotNull(dest);

            dest.Name = Path.GetFileNameWithoutExtension(filePath);

            string projectDirectoryPath = Path.GetDirectoryName(filePath);

            // サウンドセットエントリのコンバート
            foreach (ProjectSoundSetComponent child in source.Children)
            {
                string soundSetFilePath = GetSoundSetFilePath(projectDirectoryPath, child.TargetFilePath);
                dest.Children.Add(new ProjectSoundSetComponent(child.Name, soundSetFilePath));
            }

            // プロジェクト設定のコンバート
            List<string> filters = new List<string> { };
            ComponentConverter.Converter(dest, source, filters);

            this.ConvertProjectConvertSettings(source, dest);
            this.ConvertHtmlSoundListSettings(source, dest);
            this.ConvertUserParameterStructureSettings(source, dest);
            this.ConvertUserCommandSettings(source, dest);

            return dest;
        }

        ///
        private string GetSoundSetFilePath(string projectDirectoryPath, string soundSetFilePath)
        {
            return Path.Combine(
                projectDirectoryPath,
                Path.ChangeExtension(
                    soundSetFilePath,
                    this.intermediateOutputTraits.SoundSetFileExtension
                    )
                ).GetFullPath();
        }

        //-----------------------------------------------------------------
        // プロジェクト設定のコンバート
        //-----------------------------------------------------------------

        private void ConvertProjectConvertSettings(SoundProject source, SoundProject dest)
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(dest);

            // サウンドアーカイブの出力パスが "."と同等なのかを調べます。
            if (FilePathValidator.ValidateRelativePathIsCurrent
                (dest.OutputDirectoryPath) == true)
            {
                dest.OutputDirectoryPath = "output";
            }
        }

        private void ConvertHtmlSoundListSettings(SoundProject source, SoundProject dest)
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(dest);

            foreach (SoundListOutput sourceSoundListOutput in source.SoundListOutputSettings.ListOutputs)
            {
                dest.SoundListOutputSettings.ListOutputs.Add(
                    this.ConvertHtmlSoundListOutput(source, sourceSoundListOutput));
            }
        }

        //-----------------------------------------------------------------
        // HTMLサウンドリスト設定のコンバート
        //-----------------------------------------------------------------

        private SoundListOutput ConvertHtmlSoundListOutput(
            SoundProject sourceProject,
            SoundListOutput source)
        {
            Assertion.Argument.NotNull(sourceProject);
            Assertion.Argument.NotNull(source);

            SoundListOutput dest = new SoundListOutput()
            {
                Name = source.Name,
                Title = source.Title,
                FilePath = source.FilePath,
                OutputType = source.OutputType,
                IsIgnoreDisableItems = source.IsIgnoreDisableItems,
            };

            foreach (ListOutput listOutput in source.ListOutputs)
            {
                dest.ListOutputs.Add(listOutput.Clone());
            }

            return dest;
        }

        //-----------------------------------------------------------------
        // ユーザーパラメータ設定のコンバート
        //-----------------------------------------------------------------
        private void ConvertUserParameterStructureSettings(SoundProject source, SoundProject dest)
        {
            List<UserParameterStructureSetting> settings = dest.UserDataStructureSettings.Settings;
            settings.Clear();

            foreach (UserParameterStructureSetting setting in source.UserDataStructureSettings.Settings)
            {
                settings.Add(setting.Clone() as UserParameterStructureSetting);
            }
        }

        //-----------------------------------------------------------------
        // ユーザーコマンド設定のコンバート
        //-----------------------------------------------------------------
        private void ConvertUserCommandSettings(SoundProject source, SoundProject dest)
        {
            for (int i = 0; i < UserCommand.MaxCount; i++)
            {
                dest.UserCommands[i] = source.UserCommands[i].Clone();
            }
        }
    }
}
