﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Xml.Serialization;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveXmlElements;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    public delegate IFileIDAggregateFactory CreateFileIDAggregateFactoryDelegate(string basePath);
    public delegate IFileID CreateFileIDDelegate(string value);
    public delegate ISoundArchiveBinaryXml CreateSoundArchiveBinaryXmlDelegate(Stream stream);

    /// <summary>
    /// コンバート依存関係に関わる機能を提供します。
    /// </summary>
    public class SoundProjectConvertDependencyService : ISoundProjectConvertDependencyService
    {
        private static readonly CreateFileIDAggregateFactoryDelegate DefaultCreateFileIDAggregateFactory =
            (basePath) => new FileIDAggregateFactory(basePath, false);

        private CreateFileIDAggregateFactoryDelegate CreateFileIDAggregateFactory;
        private CreateFileIDDelegate CreateFileID;
        private CreateSoundArchiveBinaryXmlDelegate CreateSoundArchiveBinaryXml;

        private DependencyManager dependencies;
        private ISoundArchiveBinaryXml soundArchiveBinaryXml;
        private SoundProjectService projectService;

        private IFileIDAggregateFactory fileIDFactory;

        public SoundProjectConvertDependencyService()
        {
            this.CreateFileIDAggregateFactory = DefaultCreateFileIDAggregateFactory;
            this.CreateSoundArchiveBinaryXml = this.CreateSoundArchiveBinaryXmlMethod;
            this.CreateFileID = ((value) => new FileID(value));
        }

        ~SoundProjectConvertDependencyService()
        {
            Detach();
        }

        /// <summary>
        /// 関連付けられたサウンドプロジェクトサービスを取得します。
        /// </summary>
        public SoundProjectService SoundProjectService
        {
            get { return this.projectService; }
        }

        private string ProjectFilePath
        {
            get { return this.projectService.ProjectDocument.Resource.Key; }
        }

        private string OutputDirectoryPath
        {
            get
            {
                return Path.Combine(
                        Path.GetDirectoryName(ProjectFilePath),
                        this.projectService.Project.OutputDirectoryPath);
            }
        }

        private string BinarySoundArchiveFilePath
        {
            get
            {
                return Path.Combine(
                        OutputDirectoryPath,
                        Path.GetFileNameWithoutExtension(ProjectFilePath) + ".brsar");
            }
        }

        /// <summary>
        /// プロジェクトサービスを関連付けます。
        /// </summary>
        /// <param name="projectService">プロジェクトサービス。</param>
        public void Attach(SoundProjectService projectService)
        {
            if (this.projectService == projectService) { return; }

            Detach();

            this.projectService = projectService;

            if (null != projectService)
            {
                projectService.Opened += OnProjectOpened;
                projectService.Closed += OnProjectClosed;
            }
        }

        /// <summary>
        /// プロジェクトサービスの関連付けを解除します。
        /// </summary>
        public void Detach()
        {
            if (null != this.projectService)
            {
                this.projectService.Opened -= OnProjectOpened;
                this.projectService.Closed -= OnProjectClosed;
            }

            this.fileIDFactory = null;
            this.projectService = null;
        }

        /// <summary>
        /// サウンドプロジェクトのバイナリ情報を更新します。
        /// </summary>
        public void UpdateSoundProjectBinaryInformations()
        {
            if (null == this.projectService) { return; }
            if (null == this.projectService.ProjectDocument) { return; }

            // リロードします。
            Close();
            OpenOnlyIfNotOpened();

            if (this.dependencies != null)
            {
                UpdateSoundBinaryDataSizes();
            }

            UpdateGroupDataSizes();
        }

        /// <summary>
        /// バンクのバイナリ情報を更新します。
        /// </summary>
        public void UpdateBankBinaryInformations(BankService bankService)
        {
            if (null == bankService) { throw new ArgumentNullException("bankService"); }
            if (null == bankService.BankDocument) { return; }

            OpenOnlyIfNotOpened();
            if (null == this.dependencies) { return; }

            UpdateBankItemBinaryDataSizes(bankService);
            UpdateSoundSetBankBinaryDataSizes(bankService);
        }

        /// <summary>
        /// コンポーネントに対応するバイナリファイルパスを取得します。
        /// </summary>
        /// <param name="component">コンポーネント。</param>
        /// <returns>バイナリファイルパス。</returns>
        public string[] GetBinaryFilePath(Component component)
        {
            if (null == component) { throw new ArgumentNullException("component"); }

            IFileID fileID = this.fileIDFactory.Create(component.GetType().Name, component);
            if (fileID == null)
            {
                return new string[0];
            }

            return this.GetBinaryFilePathFromFileID(fileID.Value);
        }

        public void SetCreateFileIDAggregateFactory(CreateFileIDAggregateFactoryDelegate createFileIDAggregateFactory)
        {
            if (createFileIDAggregateFactory == null)
            {
                this.CreateFileIDAggregateFactory = DefaultCreateFileIDAggregateFactory;
            }
            else
            {
                this.CreateFileIDAggregateFactory = createFileIDAggregateFactory;
            }
        }

        public void SetCreateFileID(CreateFileIDDelegate createFileID)
        {
            if (createFileID == null)
            {
                this.CreateFileID = ((value) => new FileID(value));
            }
            else
            {
                this.CreateFileID = createFileID;
            }
        }

        public void SetCreateSoundArchiveBinaryXml(CreateSoundArchiveBinaryXmlDelegate createSoundArchiveBinaryXml)
        {
            if (createSoundArchiveBinaryXml == null)
            {
                this.CreateSoundArchiveBinaryXml = this.CreateSoundArchiveBinaryXmlMethod;
            }
            else
            {
                this.CreateSoundArchiveBinaryXml = createSoundArchiveBinaryXml;
            }
        }

        private ISoundArchiveBinaryXml CreateSoundArchiveBinaryXmlMethod(Stream stream)
        {
            SoundArchiveBinaryXml soundArchiveBinaryXml;

            if (stream == null)
            {
                soundArchiveBinaryXml = new SoundArchiveBinaryXml()
                {
                    Sounds = new List<SoundXml>(),
                    SoundSetBanks = new List<SoundSetBankXml>(),
                    SoundGroups = new List<SoundGroupXml>(),
                    Groups = new List<GroupXml>(),
                    WaveArchives = new List<WaveArchiveXml>(),
                    Players = new List<PlayerXml>(),
                    Files = new List<FileXml>(),
                };
            }
            else
            {
                soundArchiveBinaryXml = new XmlSerializer(typeof(SoundArchiveBinaryXml)).Deserialize(stream) as SoundArchiveBinaryXml;
            }

            return new SoundArchiveBinaryXmlCommon(soundArchiveBinaryXml);
        }


        /// <summary>
        /// 依存関係ファイルが開かれていないは依存関係ファイルを開きます。
        /// </summary>
        private void OpenOnlyIfNotOpened()
        {
            if (this.projectService == null) { return; }
            if (this.dependencies != null && this.soundArchiveBinaryXml != null) { return; }

            Open();
        }

        /// <summary>
        /// ファイルを開きます。
        /// </summary>
        private void Open()
        {
            Close();

            try
            {
                this.dependencies = new DependencyManager();
                this.dependencies.Load(this.GetDependencyFilePath());

                using (Stream stream = File.Open(
                    this.GetSoundArchiveBinaryXmlFilePath(),
                    FileMode.Open,
                    FileAccess.Read,
                    FileShare.Read))
                {
                    this.soundArchiveBinaryXml = this.CreateSoundArchiveBinaryXml(stream);
                }
            }
            catch (FileNotFoundException)
            {
                this.soundArchiveBinaryXml = this.CreateSoundArchiveBinaryXml(null);
            }
            catch
            {
                this.Close();
            }
        }

        /// <summary>
        /// 依存関係ファイルを閉じます。
        /// </summary>
        private void Close()
        {
            this.dependencies = null;
            this.soundArchiveBinaryXml = null;
        }

        private string GetDependencyFilePath()
        {
            return Path.Combine(
                this.projectService.ProjectIntermediateOutputPath,
                this.projectService.Project.Name + ".depend"
                );
        }

        private string GetSoundArchiveBinaryXmlFilePath()
        {
            return Path.Combine(
                this.projectService.ProjectOutputPath,
                this.projectService.Project.Name + ".xml");
        }

        private void UpdateSoundBinaryDataSizes()
        {
            foreach (WaveSoundBase sound in this.projectService.WaveSounds)
            {
                sound.Parameters[WaveSoundBase.ParameterNameOfDataSize].Value = (int)GetDataSize(sound);
            }
        }

        private void UpdateGroupDataSizes()
        {
            if (this.soundArchiveBinaryXml == null)
            {
                ResetGroupDataSizes();
                return;
            }

            try
            {
                foreach (GroupBase group in this.projectService.Groups)
                {
                    ulong totalSize;

                    if (this.soundArchiveBinaryXml.GetGroupXmlTotalSize(group.Name, out totalSize) == false)
                    {
                        this.UpdateGroupDataSize(group, -1);
                        continue;
                    }

                    this.UpdateGroupDataSize(group, (int)totalSize);
                }
            }
            catch
            {
                ResetGroupDataSizes();
                return;
            }
        }

        private void UpdateGroupDataSize(GroupBase group, int size)
        {
            Assertion.Argument.NotNull(group);
            group.DataSize = (group.OutputType == GroupOutputType.None) ? 0 : size;
        }

        private void ResetGroupDataSizes()
        {
            foreach (GroupBase group in this.projectService.Groups)
            {
                this.UpdateGroupDataSize(group, -1);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateBankItemBinaryDataSizes(BankService bankService)
        {
            if (null == bankService) { throw new ArgumentNullException("bankService"); }

            foreach (Instrument instrument in bankService.Instruments)
            {
                instrument.Parameters[Instrument.ParameterNameOfDataSize].Value =
                                                                (int)GetInstrumentDataSize(instrument);
            }

            foreach (VelocityRegion velocityRegion in bankService.VelocityRegions)
            {
                velocityRegion.Parameters[VelocityRegion.ParameterNameOfDataSize].Value =
                                                                        (int)GetDataSize(velocityRegion);
            }
        }

        /// <summary>
        /// バンクのバイナリサイズを計算してサウンドセットバンクに記録します。
        /// </summary>
        private void UpdateSoundSetBankBinaryDataSizes(BankService bankService)
        {
            HashSet<string> filePathHashSet = new HashSet<string>();
            long totalSize = 0;

            foreach (VelocityRegion velocityRegion in bankService.VelocityRegions)
            {
                string filePath = GetConvertedFilePath(velocityRegion);
                if (File.Exists(filePath) == false)
                {
                    // ファイルが存在しない場合、バイナリサイズに含めません。
                    continue;
                }

                // 重複したファイルパスの場合は、バイナリサイズに加算しないようにします。
                if (filePathHashSet.Contains(filePath) != false)
                {
                    continue;
                }
                filePathHashSet.Add(filePath);

                long size = new FileInfo(filePath).Length;
                totalSize += size;
            }

            //
            string bankFilePath = bankService.BankDocument.Resource.Key;

            foreach (SoundSetBankBase soundSetBank in this.projectService.SoundSetBanks)
            {
                if (String.Compare(soundSetBank.FilePath, bankFilePath) == 0)
                {
                    Console.WriteLine(soundSetBank.FilePath);
                    soundSetBank.BankDataSize = (int)totalSize;
                }
            }
        }

        /// <summary>
        /// Componentのコンバート後のファイルパスを取得します。
        /// </summary>
        private string GetConvertedFilePath(Component component)
        {
            Debug.Assert(component != null, "Component is null");

            IFileID fileID = this.fileIDFactory.Create(component.GetType().Name, component);
            if (fileID == null)
            {
                return null;
            }

            string[] filePaths = GetBinaryFilePathFromFileID(fileID.Value);
            if (filePaths.Length == 0)
            {
                return null;
            }

            return filePaths[0];
        }

        /// <summary>
        ///
        /// </summary>
        private long GetDataSize(Component component)
        {
            if (null == component) { throw new ArgumentNullException("component"); }

            IFileID fileID = this.fileIDFactory.Create(component.GetType().Name, component);
            if (fileID == null)
            {
                return -1;
            }

            string[] binaryFilePaths = GetBinaryFilePathFromFileID(fileID.Value);
            if (binaryFilePaths.Length == 0)
            {
                return -1;
            }

            if (!File.Exists(binaryFilePaths[0]))
            {
                return -1;
            }

            return new FileInfo(binaryFilePaths[0]).Length;
        }

        private long GetInstrumentDataSize(Instrument instrument)
        {
            if (null == instrument) { throw new ArgumentNullException("component"); }

            HashSet<string> filePaths = new HashSet<string>();
            long totalSize = 0;

            foreach (KeyRegion keyRegion in instrument.Children)
            {
                foreach (VelocityRegion velocityRegion in keyRegion.Children)
                {
                    string filePath = velocityRegion.FilePath;

                    // 一度加算したファイルは処理しない
                    if (filePaths.Contains(filePath)) { continue; }

                    long size = GetDataSize(velocityRegion);
                    if (0 > size) { return -1; }

                    totalSize += size;

                }
            }

            return totalSize;
        }

        private string[] GetBinaryFilePathFromFileID(string fileIDValue)
        {
            if (null == fileIDValue) { return new string[0]; }
            if (0 == fileIDValue.Length) { return new string[0]; }
            if (null == this.dependencies) { return new string[0]; }
            if (0 == this.dependencies.ManagementFilePath.Length) { return new string[0]; }

            IFileID fileID = this.CreateFileID(fileIDValue);

            if (Path.IsPathRooted(fileID.Source))
            {
                string relativePath = PathEx.MakeRelative(
                                        fileID.Source,
                                        this.dependencies.BaseDirectoryAbsolutePath +
                                            Path.DirectorySeparatorChar).
                                      Replace(Path.DirectorySeparatorChar, '/');

                fileID = this.CreateFileID(string.Empty);
                fileID.Source = relativePath;
                fileID.Format = fileID.Format;
                fileID.Sound = fileID.Sound;
            }

            if (!this.dependencies.Outputs.ContainsUserKey("FileID", fileID.Value))
            {
                return new string[0];
            }

            List<String> filePaths = new List<String>();
            IDependentOutput output = this.dependencies.Outputs["FileID", fileID.Value];

            foreach (IDependentOutputItem outputItem in output.OutputItems)
            {
                filePaths.Add(outputItem.AbsoluteFilePath);
            }

            return filePaths.ToArray();
        }

        private void OnProjectOpened(object sender, EventArgs e)
        {
            this.fileIDFactory = this.CreateFileIDAggregateFactory(Path.GetDirectoryName(this.GetDependencyFilePath()));

            OpenOnlyIfNotOpened();
        }

        private void OnProjectClosed(object sender, EventArgs e)
        {
            Close();

            this.fileIDFactory = null;
        }
    }
}
