﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.Collections.Generic;
    using Core.Collections;
    using Core.Parameters;
    using Projects;
    using ToolDevelopmentKit;

    public static class WaveSoundEx
    {
        public static uint GetIndex(this WaveSoundBase waveSound)
        {
            Ensure.Argument.NotNull(waveSound);

            if (!waveSound.Parameters.ContainsKey(ConversionParameterNames.WaveSound.Index))
            {
                waveSound.Parameters.AddValue(
                    ConversionParameterNames.WaveSound.Index, new UIntParameterValue(uint.MaxValue));
            }

            return (uint)waveSound.Parameters[ConversionParameterNames.WaveSound.Index].Value;
        }

        public static void SetIndex(this WaveSoundBase waveSound, uint value)
        {
            Ensure.Argument.NotNull(waveSound);

            if (!waveSound.Parameters.ContainsKey(ConversionParameterNames.WaveSound.Index))
            {
                waveSound.Parameters.AddValue(
                    ConversionParameterNames.WaveSound.Index, new UIntParameterValue());
            }

            waveSound.Parameters[ConversionParameterNames.WaveSound.Index].Value = value;
        }

        /// <summary>
        /// ウェーブサウンドセットの部分コンバート用バイナリファイルパスを取得します。
        /// </summary>
        /// <param name="waveSound">ウェーブサウンドセットを指定します。</param>
        /// <returns>ウェーブサウンドセットのバイナリファイルパスを返します。</returns>
        public static string GetBinaryFilePathForPartsConvert(this WaveSoundBase waveSound)
        {
            Ensure.Argument.NotNull(waveSound);

            if (!waveSound.Parameters.ContainsKey(
                ConversionParameterNames.WaveSound.BinaryFilePathForPartsConvert))
            {
                waveSound.Parameters.AddValue(
                    ConversionParameterNames.WaveSound.BinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            return waveSound.Parameters[
                ConversionParameterNames.WaveSound.BinaryFilePathForPartsConvert].Value.ToString();
        }

        /// <summary>
        /// ウェーブサウンドセットの部分コンバート用バイナリファイルパスを設定します。
        /// </summary>
        /// <param name="waveSound">ウェーブサウンドセットを指定します。</param>
        /// <param name="value">ウェーブサウンドセットのバイナリファイルパスを指定します。</param>
        public static void SetBinaryFilePathForPartsConvert(this WaveSoundBase waveSound, string value)
        {
            Ensure.Argument.NotNull(waveSound);

            if (!waveSound.Parameters.ContainsKey(
                ConversionParameterNames.WaveSound.BinaryFilePathForPartsConvert))
            {
                waveSound.Parameters.AddValue(
                    ConversionParameterNames.WaveSound.BinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            waveSound.Parameters[
                ConversionParameterNames.WaveSound.BinaryFilePathForPartsConvert].Value = value;
        }

        /// <summary>
        /// ウェーブサウンドセットが参照する部分コンバート用波形アーカイブバイナリファイルパスを取得します。
        /// </summary>
        /// <param name="waveSound">ウェーブサウンドセットを指定します。</param>
        /// <returns>ウェーブサウンドセットが参照する部分コンバート用波形アーカイブバイナリファイルパスを返します。</returns>
        public static string GetWaveArchiveBinaryFilePathForPartsConvert(this WaveSoundBase waveSound)
        {
            Ensure.Argument.NotNull(waveSound);

            if (!waveSound.Parameters.ContainsKey(
                ConversionParameterNames.WaveSound.WaveArchiveBinaryFilePathForPartsConvert))
            {
                waveSound.Parameters.AddValue(
                    ConversionParameterNames.WaveSound.WaveArchiveBinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            return waveSound.Parameters[
                ConversionParameterNames.WaveSound.WaveArchiveBinaryFilePathForPartsConvert].Value.ToString();
        }

        /// <summary>
        /// ウェーブサウンドセットが参照する部分コンバート用波形アーカイブバイナリファイルパスを設定します。
        /// </summary>
        /// <param name="waveSound">ウェーブサウンドセットを指定します。</param>
        /// <param name="value">ウェーブサウンドセットが参照する部分コンバート用波形アーカイブバイナリファイルパスを指定します。</param>
        public static void SetWaveArchiveBinaryFilePathForPartsConvert(this WaveSoundBase waveSound, string value)
        {
            Ensure.Argument.NotNull(waveSound);

            if (!waveSound.Parameters.ContainsKey(
                ConversionParameterNames.WaveSound.WaveArchiveBinaryFilePathForPartsConvert))
            {
                waveSound.Parameters.AddValue(
                    ConversionParameterNames.WaveSound.WaveArchiveBinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            waveSound.Parameters[
                ConversionParameterNames.WaveSound.WaveArchiveBinaryFilePathForPartsConvert].Value = value;
        }

        public static IDictionary<string, WaveArchiveBase> GetWaveArchiveDictionary(this WaveSound waveSound)
        {
            Ensure.Argument.NotNull(waveSound);
            Ensure.Argument.NotNull(waveSound.Parent);

            // 親 WSDSET の WARC を使うのは、新 WSD バイナリ出力の場合のみ
            Ensure.Argument.True(waveSound.IsWaveSound2BinaryRequired());

            if (!waveSound.Parent.Parameters.ContainsKey(ConversionParameterNames.TargetWaveArchive))
            {
                waveSound.Parent.Parameters.AddValue(
                    ConversionParameterNames.TargetWaveArchive,
                    new ObjectParameterValue<IDictionary<string, WaveArchiveBase>>(new Dictionary<string, WaveArchiveBase>()));
            }

            return waveSound.Parent.Parameters[ConversionParameterNames.TargetWaveArchive].Value as IDictionary<string, WaveArchiveBase>;
        }

        public static bool IsWaveSound2BinaryRequired(this WaveSound waveSound)
        {
            Ensure.Argument.NotNull(waveSound);
            return waveSound.WaveSoundResource != null;
        }

        public static IList<WaveSoundTrackConvertModel> GetTrackConvertModels(this WaveSoundBase waveSound)
        {
            Ensure.Argument.NotNull(waveSound);

            if (!waveSound.Parameters.ContainsKey(ConversionParameterNames.WaveSound.WaveSoundTrackConvertModels))
            {
                waveSound.Parameters.AddValue(
                    ConversionParameterNames.WaveSound.WaveSoundTrackConvertModels,
                    new ObjectParameterValue<IList<WaveSoundTrackConvertModel>>(new List<WaveSoundTrackConvertModel>()));
            }

            return waveSound.Parameters[ConversionParameterNames.WaveSound.WaveSoundTrackConvertModels].Value
                as IList<WaveSoundTrackConvertModel>;
        }

        public static IList<WaveSoundClipConvertModel> GetClipConvertModels(this WaveSoundBase waveSound)
        {
            Ensure.Argument.NotNull(waveSound);

            if (!waveSound.Parameters.ContainsKey(ConversionParameterNames.WaveSound.WaveSoundClipConvertModels))
            {
                waveSound.Parameters.AddValue(
                    ConversionParameterNames.WaveSound.WaveSoundClipConvertModels,
                    new ObjectParameterValue<IList<WaveSoundClipConvertModel>>(new List<WaveSoundClipConvertModel>()));
            }

            return waveSound.Parameters[ConversionParameterNames.WaveSound.WaveSoundClipConvertModels].Value
                as IList<WaveSoundClipConvertModel>;
        }

        public static IEnumerable<Component> EnumerateWaveComponents(this WaveSound waveSound, bool isWaveSound2BinaryEnabled)
        {
            if (!isWaveSound2BinaryEnabled)
            {
                Ensure.Operation.True(!waveSound.IsWaveSound2BinaryRequired());
                yield return waveSound;
            }
            else
            {
                if (!waveSound.IsWaveSound2BinaryRequired())
                {
                    Ensure.Operation.True(false, "WaveSound must has WaveSoundResource.");
                    yield return waveSound;
                }

                foreach (var waveSoundClip in waveSound.GetClipConvertModels())
                {
                    yield return waveSoundClip;
                }
            }
        }

        public static int? GetTargetSampleRate(this WaveSoundBase waveSound)
        {
            Ensure.Argument.NotNull(waveSound);

            if (waveSound.IsResampleEnabled)
            {
                return waveSound.SampleRate;
            }
            else
            {
                return null;
            }
        }

        public static int? GetTargetChannelCount(this WaveSoundBase waveSound)
        {
            Ensure.Argument.NotNull(waveSound);

            if (waveSound.IsDownMixEnabled)
            {
                return 1; // モノラル化
            }
            else
            {
                return null;
            }
        }
    }
}
