﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Codecs;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.FileFormats.Audio;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;
    using Nintendo.Atk.Binary;
    using NintendoWare.SoundFoundation.Core.IO;

    /// <summary>
    /// StreamSoundBase の拡張機能を定義します。
    /// </summary>
    public static class StreamSoundExtension
    {
        private const string AACFileExtension = ".aac";

        public static IOutput GetPrefetchOutputTarget(this Component component)
        {
            Ensure.Argument.NotNull(component);

            if (!component.Parameters.ContainsKey(ConversionParameterNames.StreamSound.PrefetchOutputTarget))
            {
                component.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.PrefetchOutputTarget,
                    new ObjectParameterValue<IOutput>());
            }

            return component.Parameters[ConversionParameterNames.StreamSound.PrefetchOutputTarget].Value as IOutput;
        }

        public static void SetPrefetchOutputTarget(this Component component, IOutput value)
        {
            Ensure.Argument.NotNull(component);

            if (!component.Parameters.ContainsKey(ConversionParameterNames.StreamSound.PrefetchOutputTarget))
            {
                component.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.PrefetchOutputTarget,
                    new ObjectParameterValue<IOutput>());
            }

            component.Parameters[ConversionParameterNames.StreamSound.PrefetchOutputTarget].Value = value;
        }

        /// <summary>
        /// ストリームサウンドの TrackAllocationFlags を取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドの TrackAllocationFlags を返します。</returns>
        public static ushort GetTrackAllocationFlags(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.TrackAllocationFlags))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.TrackAllocationFlags,
                    new UIntParameterValue(0));
            }

            return Convert.ToUInt16(streamSound.Parameters[ConversionParameterNames.StreamSound.TrackAllocationFlags].Value);
        }

        /// <summary>
        /// ストリームサウンドの TrackAllocationFlags を取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドの TrackAllocationFlags を返します。</returns>
        public static void SetTrackAllocationFlags(this StreamSoundBase streamSound, ushort value)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.TrackAllocationFlags))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.TrackAllocationFlags,
                    new UIntParameterValue(0));
            }

            streamSound.Parameters[ConversionParameterNames.StreamSound.TrackAllocationFlags].Value = (uint)value;
        }

        /// <summary>
        /// ストリームサウンドの総チャンネル数を取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドの総チャンネル数を返します。</returns>
        public static uint GetTotalChannelCount(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.TotalChannelCount))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.TotalChannelCount,
                    new UIntParameterValue(0));
            }

            return (uint)streamSound.Parameters[ConversionParameterNames.StreamSound.TotalChannelCount].Value;
        }

        /// <summary>
        /// ストリームサウンドの総チャンネル数を取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドの総チャンネル数を返します。</returns>
        public static void SetTotalChannelCount(this StreamSoundBase streamSound, uint value)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.TotalChannelCount))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.TotalChannelCount,
                    new UIntParameterValue(0));
            }

            streamSound.Parameters[ConversionParameterNames.StreamSound.TotalChannelCount].Value = value;
        }

        /// <summary>
        /// ストリームサウンドのループ開始フレームを取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドのループ開始フレームを返します。</returns>
        public static uint GetLoopStartFrame(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.LoopStartFrame))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.LoopStartFrame,
                    new UIntParameterValue(0));
            }

            return (uint)streamSound.Parameters[ConversionParameterNames.StreamSound.LoopStartFrame].Value;
        }

        /// <summary>
        /// ストリームサウンドのループ開始フレームを取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドのループ開始フレームを返します。</returns>
        public static void SetLoopStartFrame(this StreamSoundBase streamSound, uint value)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.LoopStartFrame))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.LoopStartFrame,
                    new UIntParameterValue(0));
            }

            streamSound.Parameters[ConversionParameterNames.StreamSound.LoopStartFrame].Value = value;
        }

        /// <summary>
        /// ストリームサウンドのループ終了フレームを取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドのループ終了フレームを返します。</returns>
        public static uint GetLoopEndFrame(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.LoopEndFrame))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.LoopEndFrame,
                    new UIntParameterValue(0));
            }

            return (uint)streamSound.Parameters[ConversionParameterNames.StreamSound.LoopEndFrame].Value;
        }

        /// <summary>
        /// ストリームサウンドのループ終了フレームを取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドのループ終了フレームを返します。</returns>
        public static void SetLoopEndFrame(this StreamSoundBase streamSound, uint value)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.LoopEndFrame))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.LoopEndFrame,
                    new UIntParameterValue(0));
            }

            streamSound.Parameters[ConversionParameterNames.StreamSound.LoopEndFrame].Value = value;
        }

        /// <summary>
        /// ストリームサウンドのループの有無を取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドのループの有無を返します。</returns>
        public static bool GetHasLoop(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.HasLoop))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.HasLoop,
                    new BoolParameterValue(false));
            }

            return (bool)streamSound.Parameters[ConversionParameterNames.StreamSound.HasLoop].Value;
        }

        /// <summary>
        /// ストリームサウンドのループの有無を取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドのループの有無を返します。</returns>
        public static void SetHasLoop(this StreamSoundBase streamSound, bool value)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.HasLoop))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.HasLoop,
                    new BoolParameterValue(false));
            }

            streamSound.Parameters[ConversionParameterNames.StreamSound.HasLoop].Value = value;
        }

        internal static ContainerType GetContainerType(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.ContainerType))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.ContainerType, new ObjectParameterValue<ContainerType>(ContainerType.Invalid));
            }

            return (ContainerType)streamSound.Parameters[ConversionParameterNames.StreamSound.ContainerType].Value;
        }

        internal static void SetContainerType(this StreamSoundBase streamSound, ContainerType value)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.ContainerType))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.ContainerType, new ObjectParameterValue<ContainerType>(ContainerType.Invalid));
            }

            streamSound.Parameters[ConversionParameterNames.StreamSound.ContainerType].Value = value;
        }

        /// <summary>
        /// ストリームサウンドの部分コンバート用バイナリファイルパスを取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドの部分コンバート用バイナリファイルパスを返します。</returns>
        public static string GetBinaryFilePathForPartsConvert(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(
                ConversionParameterNames.StreamSound.BinaryFilePathForPartsConvert))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.BinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            return streamSound.Parameters[
                ConversionParameterNames.StreamSound.BinaryFilePathForPartsConvert].Value.ToString();
        }

        /// <summary>
        /// ストリームサウンドの部分コンバート用バイナリファイルパスを設定します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <param name="value">ストリームサウンドの部分コンバート用バイナリファイルパスを指定します。</param>
        public static void SetBinaryFilePathForPartsConvert(this StreamSoundBase streamSound, string value)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(
                ConversionParameterNames.StreamSound.BinaryFilePathForPartsConvert))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.BinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            streamSound.Parameters[
                ConversionParameterNames.StreamSound.BinaryFilePathForPartsConvert].Value = value;
        }

        /// <summary>
        /// ストリームサウンドの部分コンバート用プリフェッチバイナリファイルパスを取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドの部分コンバート用プリフェッチバイナリファイルパスを返します。</returns>
        public static string GetPrefetchBinaryFilePathForPartsConvert(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(
                ConversionParameterNames.StreamSound.PrefetchBinaryFilePathForPartsConvert))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.PrefetchBinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            return streamSound.Parameters[
                ConversionParameterNames.StreamSound.PrefetchBinaryFilePathForPartsConvert].Value.ToString();
        }

        /// <summary>
        /// ストリームサウンドの部分コンバート用プリフェッチバイナリファイルパスを設定します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <param name="value">ストリームサウンドの部分コンバート用プリフェッチバイナリファイルパスを指定します。</param>
        public static void SetPrefetchBinaryFilePathForPartsConvert(this StreamSoundBase streamSound, string value)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(
                ConversionParameterNames.StreamSound.PrefetchBinaryFilePathForPartsConvert))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.PrefetchBinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            streamSound.Parameters[
                ConversionParameterNames.StreamSound.PrefetchBinaryFilePathForPartsConvert].Value = value;
        }

        /// <summary>
        /// ストリームサウンドの部分コンバート用PCバイナリファイルパスを取得します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ストリームサウンドの部分コンバート用PCバイナリファイルパスを返します。</returns>
        public static string GetPCBinaryFilePathForPartsConvert(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(
                ConversionParameterNames.StreamSound.PCBinaryFilePathForPartsConvert))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.PCBinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            return streamSound.Parameters[
                ConversionParameterNames.StreamSound.PCBinaryFilePathForPartsConvert].Value.ToString();
        }

        /// <summary>
        /// ストリームサウンドの部分コンバート用PCバイナリファイルパスを設定します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <param name="value">ストリームサウンドの部分コンバート用PCバイナリファイルパスを指定します。</param>
        public static void SetPCBinaryFilePathForPartsConvert(this StreamSoundBase streamSound, string value)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(
                ConversionParameterNames.StreamSound.PCBinaryFilePathForPartsConvert))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.PCBinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            streamSound.Parameters[
                ConversionParameterNames.StreamSound.PCBinaryFilePathForPartsConvert].Value = value;
        }

        public static IList<WaveMarkerInfo> GetWaveMarkers(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.WaveMarkers))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.WaveMarkers,
                    new ObjectParameterValue<IList<WaveMarkerInfo>>(new List<WaveMarkerInfo>()));
            }

            return streamSound.Parameters[ConversionParameterNames.StreamSound.WaveMarkers].Value as IList<WaveMarkerInfo>;
        }

        public static IList<WaveRegionInfo> GetWaveRegions(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (!streamSound.Parameters.ContainsKey(ConversionParameterNames.StreamSound.WaveRegions))
            {
                streamSound.Parameters.AddValue(
                    ConversionParameterNames.StreamSound.WaveRegions,
                    new ObjectParameterValue<IList<WaveRegionInfo>>(new List<WaveRegionInfo>()));
            }

            return streamSound.Parameters[ConversionParameterNames.StreamSound.WaveRegions].Value as IList<WaveRegionInfo>;
        }

        public static AdtsHeader GetAdtsHeader(this StreamSoundBase streamSound)
        {
            if (streamSound.Children.Count == 0)
            {
                return null;
            }

            var track = streamSound.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase;

            if (track == null)
            {
                return null;
            }
#if !AAC_SUPPORTED
            if (string.Compare(Path.GetExtension(track.FilePath), AACFileExtension, ignoreCase: true) != 0)
            {
                return null;
            }
#endif
            try
            {
                using (var fileStream = File.OpenRead(track.FilePath))
                {
                    var result = AdtsHeader.Parse(fileStream);

                    if (result == null)
                    {
                        return null;
                    }

                    if (!result.IsValid())
                    {
                        return null;
                    }
#if AAC_SUPPORTED
                    if (streamSound.Encoding != WaveEncoding.NoConvert)
                    {
                        var errorLine = new ErrorLine(
                            string.Format(Resources.MessageResource.Message_SoundEncodingTypeMustBeNoConvert, "AAC"),
                            streamSound);
                        throw new ConversionException(errorLine);
                    }
#endif
                    return result;
                }
            }
            catch (ConversionException exception)
            {
                throw exception;
            }
            catch
            {
                return null;
            }
        }

        public static AudioOpusBasicInfo GetAudioOpusBasicInfo(this StreamSoundBase streamSound)
        {
            if (streamSound.Encoding != WaveEncoding.NoConvert)
            {
                var errorLine = new ErrorLine(
                    string.Format(Resources.MessageResource.Message_SoundEncodingTypeMustBeNoConvert, "opus"),
                    streamSound);
                throw new ConversionException(errorLine);
            }

            var track = streamSound.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase;

            if (track == null)
            {
                return null;
            }

            try
            {
                using (var fileStream = File.OpenRead(track.FilePath))
                {
                    using (var reader = LittleEndianBinaryReader.Create(fileStream))
                    {
                        return AudioOpusBasicInfo.FromBinary(reader);
                    }
                }
            }
            catch
            {
                throw new ConversionException(
                    new ErrorLine(string.Format(
                        Resources.MessageResourceCommon.Message_FailedToReadOpusFileHeader,
                        track.FilePath)));
            }
        }

        public static string GetLinearPcmWaveFilePath(this StreamSoundBase streamSound)
        {
            if (streamSound.Encoding != WaveEncoding.NoConvert)
            {
                return null;
            }

            if (streamSound.Children.Count == 0)
            {
                return null;
            }

            var track = streamSound.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase;

            if (track == null)
            {
                return null;
            }

            return GetLinearPcmWaveFilePath(track.FilePath);
        }

        public static string GetLinearPcmWaveFilePath(string filePath)
        {
            string waveFilePath = System.IO.Path.ChangeExtension(filePath, "wav");

            if (System.IO.File.Exists(waveFilePath))
            {
                return waveFilePath;
            }

            waveFilePath = System.IO.Path.ChangeExtension(filePath, "aiff");

            if (System.IO.File.Exists(waveFilePath))
            {
                return waveFilePath;
            }

            waveFilePath = System.IO.Path.ChangeExtension(filePath, "aif");

            if (System.IO.File.Exists(waveFilePath))
            {
                return waveFilePath;
            }

            return string.Empty;
        }

        public static int? GetTargetSampleRate(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (streamSound.IsResampleEnabled)
            {
                return streamSound.SampleRate;
            }
            else
            {
                return null;
            }
        }

        public static int? GetTargetChannelCount(this StreamSoundBase streamSound)
        {
            Ensure.Argument.NotNull(streamSound);

            if (streamSound.IsDownMixEnabled)
            {
                return 1; // モノラル化
            }
            else
            {
                return null;
            }
        }
    }
}
