﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using Core.Collections;
    using Core.Parameters;
    using Projects;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Collections;

    internal static class SoundSetBankEx
    {
        public static IDictionary<string, WaveArchiveBase> GetWaveArchiveDictionary(this SoundSetBankBase soundSetBank)
        {
            Ensure.Argument.NotNull(soundSetBank);

            if (!soundSetBank.Parameters.ContainsKey(ConversionParameterNames.TargetWaveArchive))
            {
                soundSetBank.Parameters.AddValue(
                    ConversionParameterNames.TargetWaveArchive,
                    new ObjectParameterValue<IDictionary<string, WaveArchiveBase>>(new Dictionary<string, WaveArchiveBase>()));
            }

            return soundSetBank.Parameters[ConversionParameterNames.TargetWaveArchive].Value as IDictionary<string, WaveArchiveBase>;
        }

        public static WaveArchiveBase GetWaveArchiveForGroup(this SoundSetBankBase soundSetBank, string groupName)
        {
            Ensure.Argument.NotNull(soundSetBank);
            Ensure.Argument.NotNull(groupName);

            if (soundSetBank.TargetWaveArchive != null)
            {
                return soundSetBank.TargetWaveArchive;
            }

            IDictionary<string, WaveArchiveBase> waveArchiveDictionary = soundSetBank.GetWaveArchiveDictionary();
            WaveArchiveBase waveArchive = null;

            if (waveArchiveDictionary.TryGetValue(groupName, out waveArchive))
            {
                return waveArchive;
            }

            if (waveArchiveDictionary.TryGetValue(string.Empty, out waveArchive))
            {
                return waveArchive;
            }

            return null;
        }

        /// <summary>
        /// SoundSetBankBase の部分コンバート用バイナリファイルパスを取得します。
        /// </summary>
        /// <param name="soundSetBank">SoundSetBankBase を指定します。</param>
        /// <returns>SoundSetBankBase のバイナリファイルパスを返します。</returns>
        public static string GetBinaryFilePathForPartsConvert(this SoundSetBankBase soundSetBank)
        {
            Ensure.Argument.NotNull(soundSetBank);

            if (!soundSetBank.Parameters.ContainsKey(
                ConversionParameterNames.SoundSetBank.BinaryFilePathForPartsConvert))
            {
                soundSetBank.Parameters.AddValue(
                    ConversionParameterNames.SoundSetBank.BinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            return soundSetBank.Parameters[
                ConversionParameterNames.SoundSetBank.BinaryFilePathForPartsConvert].Value.ToString();
        }

        /// <summary>
        /// SoundSetBankBase の部分コンバート用バイナリファイルパスを設定します。
        /// </summary>
        /// <param name="soundSetBank">SoundSetBankBase を指定します。</param>
        /// <param name="value">SoundSetBankBase のバイナリファイルパスを指定します。</param>
        public static void SetBinaryFilePathForPartsConvert(this SoundSetBankBase soundSetBank, string value)
        {
            Ensure.Argument.NotNull(soundSetBank);

            if (!soundSetBank.Parameters.ContainsKey(
                ConversionParameterNames.SoundSetBank.BinaryFilePathForPartsConvert))
            {
                soundSetBank.Parameters.AddValue(
                    ConversionParameterNames.SoundSetBank.BinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            soundSetBank.Parameters[
                ConversionParameterNames.SoundSetBank.BinaryFilePathForPartsConvert].Value = value;
        }

        /// <summary>
        /// SoundSetBankBase が参照する部分コンバート用波形アーカイブバイナリファイルパスを取得します。
        /// </summary>
        /// <param name="soundSetBank">SoundSetBankBase を指定します。</param>
        /// <returns>SoundSetBankBase が参照する部分コンバート用波形アーカイブバイナリファイルパスを返します。</returns>
        public static string GetWaveArchiveBinaryFilePathForPartsConvert(this SoundSetBankBase soundSetBank)
        {
            Ensure.Argument.NotNull(soundSetBank);

            if (!soundSetBank.Parameters.ContainsKey(
                ConversionParameterNames.SoundSetBank.WaveArchiveBinaryFilePathForPartsConvert))
            {
                soundSetBank.Parameters.AddValue(
                    ConversionParameterNames.SoundSetBank.WaveArchiveBinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            return soundSetBank.Parameters[
                ConversionParameterNames.SoundSetBank.WaveArchiveBinaryFilePathForPartsConvert].Value.ToString();
        }

        /// <summary>
        /// SoundSetBankBase が参照する部分コンバート用波形アーカイブバイナリファイルパスを設定します。
        /// </summary>
        /// <param name="soundSetBank">SoundSetBankBase を指定します。</param>
        /// <param name="value">SoundSetBankBase が参照する部分コンバート用波形アーカイブバイナリファイルパスを指定します。</param>
        public static void SetWaveArchiveBinaryFilePathForPartsConvert(this SoundSetBankBase soundSetBank, string value)
        {
            Ensure.Argument.NotNull(soundSetBank);

            if (!soundSetBank.Parameters.ContainsKey(
                ConversionParameterNames.SoundSetBank.WaveArchiveBinaryFilePathForPartsConvert))
            {
                soundSetBank.Parameters.AddValue(
                    ConversionParameterNames.SoundSetBank.WaveArchiveBinaryFilePathForPartsConvert,
                    new TextParameterValue());
            }

            soundSetBank.Parameters[
                ConversionParameterNames.SoundSetBank.WaveArchiveBinaryFilePathForPartsConvert].Value = value;
        }
    }
}
