﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using NintendoWare.SoundFoundation.Core.Collections;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    internal static class GroupEx
    {
        public static bool IsEmbeddedWaveSharing(this GroupBase group, bool isWaveSound2BinaryEnabled)
        {
            Ensure.Argument.NotNull(group);

            switch (group.OutputType)
            {
                case GroupOutputType.Embedding:
                case GroupOutputType.UserManagement:
                    break;

                default:
                    return false;
            }

            return group.AutoSharedWaveArchiveIncludes(isWaveSound2BinaryEnabled);
        }

        public static WaveArchiveBase GetTargetWaveArchive(this GroupBase group)
        {
            Ensure.Argument.NotNull(group);

            if (!group.Parameters.ContainsKey(ConversionParameterNames.TargetWaveArchive))
            {
                group.Parameters.AddValue(
                    ConversionParameterNames.TargetWaveArchive,
                    new ObjectParameterValue<WaveArchiveBase>());
            }

            return group.Parameters[ConversionParameterNames.TargetWaveArchive].Value as WaveArchiveBase;
        }

        public static void SetTargetWaveArchive(this GroupBase group, WaveArchiveBase value)
        {
            Ensure.Argument.NotNull(group);
            Ensure.Argument.NotNull(value);

            if (!group.Parameters.ContainsKey(ConversionParameterNames.TargetWaveArchive))
            {
                group.Parameters.AddValue(
                    ConversionParameterNames.TargetWaveArchive,
                    new ObjectParameterValue<WaveArchiveBase>());
            }

            group.Parameters[ConversionParameterNames.TargetWaveArchive].Value = value;
        }

        public static IList<ComponentFile> GetItemFiles(this GroupBase group)
        {
            Ensure.Argument.NotNull(group);

            if (!group.Parameters.ContainsKey(ConversionParameterNames.Group.ItemFiles))
            {
                group.Parameters.AddValue(
                    ConversionParameterNames.Group.ItemFiles,
                    new ObjectParameterValue<IList<ComponentFile>>(new NumberedList<ComponentFile>()));
            }

            return group.Parameters[ConversionParameterNames.Group.ItemFiles].Value as IList<ComponentFile>;
        }

        public static IEnumerable<GroupItemBase> GetAllItems(this GroupBase group)
        {
            return GroupEx.GetAllItems(group, group, new HashSet<GroupBase>());
        }

        public static IEnumerable<SoundSetItem> GetAllItemTargets(this GroupBase group, bool isWaveSound2BinaryEnabled)
        {
            HashSet<SoundSetItem> enumeratedItems = new HashSet<SoundSetItem>();

            foreach (GroupItemBase groupItem in GetAllItems(group, group, new HashSet<GroupBase>()))
            {
                IEnumerable<SoundSetItem> items = null;

                if (groupItem.Target is StreamSoundBase)
                {
                    items = GetAllItemTargets(
                        groupItem.Target as StreamSoundBase,
                        groupItem.RegisterType,
                        group,
                        enumeratedItems);
                }
                else if (groupItem.Target is SequenceSoundSetBase)
                {
                    items = GetAllItemTargets(
                        groupItem.Target as SequenceSoundSetBase,
                        groupItem.RegisterType,
                        group,
                        enumeratedItems);
                }
                else if (groupItem.Target is SequenceSoundBase)
                {
                    items = GetAllItemTargets(
                        groupItem.Target as SequenceSoundBase,
                        groupItem.RegisterType,
                        group,
                        enumeratedItems);
                }
                else if (groupItem.Target is SoundSetBankBase)
                {
                    items = GetAllItemTargets(
                        groupItem.Target as SoundSetBankBase,
                        groupItem.RegisterType,
                        group,
                        enumeratedItems);
                }
                else if (groupItem.Target is WaveSoundSetBase)
                {
                    items = GetAllItemTargets(
                        groupItem.Target as WaveSoundSetBase,
                        groupItem.RegisterType,
                        group,
                        enumeratedItems,
                        isWaveSound2BinaryEnabled);
                }
                else if (groupItem.Target is WaveArchiveBase)
                {
                    items = GetAllItemTargets(
                        groupItem.Target as WaveArchiveBase,
                        groupItem.RegisterType,
                        enumeratedItems);
                }
                else
                {
                    continue;
                }

                foreach (SoundSetItem item in items)
                {
                    if (!item.IsHierarchicalConvertTarget())
                    {
                        continue;
                    }

                    yield return item;
                }
            }
        }

        public static bool AutoSharedWaveArchiveIncludes(this GroupBase group, bool isWaveSound2BinaryEnabled)
        {
            Ensure.Argument.NotNull(group);

            foreach (SoundSetItem item in group.GetAllItemTargets(isWaveSound2BinaryEnabled))
            {
                if (!item.IsHierarchicalConvertTarget())
                {
                    continue;
                }

                string waveArchive = string.Empty;

                if (item is WaveSoundSetBase)
                {
                    waveArchive = (item as WaveSoundSetBase).WaveArchiveReference;
                }
                else if (item is SoundSetBankBase)
                {
                    waveArchive = (item as SoundSetBankBase).WaveArchiveReference;
                }
                else if (item is GroupBase)
                {
                    bool isInnerGroupResult = (item as GroupBase).AutoSharedWaveArchiveIncludes(isWaveSound2BinaryEnabled);

                    if (isInnerGroupResult)
                    {
                        return true;
                    }

                    continue;
                }
                else
                {
                    continue;
                }

                if (waveArchive == WaveArchiveConsts.AutoShared)
                {
                    return true;
                }
            }

            return false;
        }

        private static IEnumerable<GroupItemBase> GetAllItems(GroupBase ownerGroup, GroupBase group, HashSet<GroupBase> groupStack)
        {
            Assertion.Argument.NotNull(ownerGroup);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.NotNull(groupStack);

            if (groupStack.Contains(group))
            {
                throw new Exception(
                    string.Format(
                    Resources.MessageResource.Message_LoopGroupReferenceDetected,
                    ownerGroup.Name, group.Name)
                    );
            }

            groupStack.Add(group);

            foreach (GroupItemBase groupItem in group.Children)
            {
                if (!groupItem.IsConvertTarget())
                {
                    continue;
                }

                if (groupItem.Target is GroupBase)
                {
                    // 内部グループの内容を展開します。
                    GroupBase innerGroup = groupItem.Target as GroupBase;

                    if (!innerGroup.IsConvertTarget())
                    {
                        continue;
                    }

                    foreach (GroupItemBase innerGroupItem in GroupEx.GetAllItems(ownerGroup, innerGroup, groupStack))
                    {
                        yield return innerGroupItem;
                    }
                }
                else
                {
                    yield return groupItem;
                }
            }

            groupStack.Remove(group);
        }

        private static IEnumerable<SoundSetItem> GetAllItemTargets(
            StreamSoundBase streamSound,
            GroupItemRegisterType registerType,
            GroupBase group,
            HashSet<SoundSetItem> enumeratedItems)
        {
            Assertion.Argument.NotNull(streamSound);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.NotNull(enumeratedItems);

            if (!streamSound.IsHierarchicalConvertTarget())
            {
                yield break;
            }

            if (registerType.AllIncludes() &&
                !enumeratedItems.Contains(streamSound))
            {
                enumeratedItems.Add(streamSound);
                yield return streamSound;
            }
        }

        private static IEnumerable<SoundSetItem> GetAllItemTargets(
            SequenceSoundSetBase sequenceSoundSet,
            GroupItemRegisterType registerType,
            GroupBase group,
            HashSet<SoundSetItem> enumeratedItems)
        {
            Assertion.Argument.NotNull(sequenceSoundSet);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.NotNull(enumeratedItems);

            if (!sequenceSoundSet.IsHierarchicalConvertTarget())
            {
                yield break;
            }

            foreach (SequenceSoundBase sequenceSound in sequenceSoundSet.Children)
            {
                foreach (SoundSetItem item in
                   GetAllItemTargets(sequenceSound, registerType, group, enumeratedItems))
                {
                    yield return item;
                }
            }
        }

        private static IEnumerable<SoundSetItem> GetAllItemTargets(
            SequenceSoundBase sequenceSound,
            GroupItemRegisterType registerType,
            GroupBase group,
            HashSet<SoundSetItem> enumeratedItems)
        {
            Assertion.Argument.NotNull(sequenceSound);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.NotNull(enumeratedItems);

            if (!sequenceSound.IsHierarchicalConvertTarget())
            {
                yield break;
            }

            if (registerType.SequenceIncludes() &&
                !enumeratedItems.Contains(sequenceSound))
            {
                enumeratedItems.Add(sequenceSound);
                yield return sequenceSound;
            }

            foreach (SoundSetBankBase soundSetBank in sequenceSound.SoundSetBanks)
            {
                // バンクの欠番をゆるす
                if (soundSetBank == null)
                {
                    continue;
                }

                foreach (SoundSetItem item in
                    GetAllItemTargets(soundSetBank, registerType, group, enumeratedItems))
                {
                    yield return item;
                }
            }
        }

        private static IEnumerable<SoundSetItem> GetAllItemTargets(
            SoundSetBankBase soundSetBank,
            GroupItemRegisterType registerType,
            GroupBase group,
            HashSet<SoundSetItem> enumeratedItems)
        {
            Assertion.Argument.NotNull(soundSetBank);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.NotNull(enumeratedItems);

            if (!soundSetBank.IsHierarchicalConvertTarget())
            {
                yield break;
            }

            if (registerType.BankIncludes() &&
                !enumeratedItems.Contains(soundSetBank))
            {
                enumeratedItems.Add(soundSetBank);
                yield return soundSetBank;
            }

            WaveArchiveBase waveArchiveForGroup = soundSetBank.GetWaveArchiveForGroup(group.Name);

            if (waveArchiveForGroup == null)
            {
                yield break;
            }

            foreach (SoundSetItem item in
                GetAllItemTargets(waveArchiveForGroup, registerType, enumeratedItems))
            {
                yield return item;
            }
        }

        private static IEnumerable<SoundSetItem> GetAllItemTargets(
            WaveSoundSetBase waveSoundSet,
            GroupItemRegisterType registerType,
            GroupBase group,
            HashSet<SoundSetItem> enumeratedItems,
            bool isWaveSound2BinaryEnabled)
        {
            Assertion.Argument.NotNull(waveSoundSet);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.NotNull(enumeratedItems);

            if (!waveSoundSet.IsHierarchicalConvertTarget())
            {
                yield break;
            }

            if (registerType.WaveSoundSetIncludes() &&
                !enumeratedItems.Contains(waveSoundSet))
            {
                enumeratedItems.Add(waveSoundSet);

                if (!isWaveSound2BinaryEnabled)
                {
                    yield return waveSoundSet;
                }
                else
                {
                    foreach (var waveSound in waveSoundSet.Children.Cast<WaveSoundBase>())
                    {
                        if (waveSound.GetIsExcluded())
                        {
                            yield break;
                        }

                        yield return waveSound;
                    }
                }
            }

            WaveArchiveBase waveArchiveForGroup = waveSoundSet.GetWaveArchiveForGroup(group.Name);

            if (waveArchiveForGroup == null)
            {
                yield break;
            }

            foreach (SoundSetItem item in
                GetAllItemTargets(waveArchiveForGroup, registerType, enumeratedItems))
            {
                yield return item;
            }
        }

        private static IEnumerable<SoundSetItem> GetAllItemTargets(
            WaveArchiveBase waveArchive,
            GroupItemRegisterType registerType,
            HashSet<SoundSetItem> enumeratedItems)
        {
            Assertion.Argument.NotNull(waveArchive);
            Assertion.Argument.NotNull(enumeratedItems);

            if (!waveArchive.IsHierarchicalConvertTarget())
            {
                yield break;
            }

            if (registerType.WaveArchiveIncludes() &&
                !enumeratedItems.Contains(waveArchive))
            {
                enumeratedItems.Add(waveArchive);
                yield return waveArchive;
            }
        }
    }
}
