﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Linq;
    using NintendoWare.SoundFoundation.Core.Collections;
    using NintendoWare.SoundFoundation.FileFormats.Audio;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveElements;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Projects;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Collections;

    /// <summary>
    /// サウンドアーカイブバイナリ DOM の構築をサポートします。
    /// </summary>
    internal class SoundArchiveFileBuilder
    {
        private const uint InvalidFileID = 0xFFFFFFFF;

        public SoundArchiveFileBuilder(string signature, BinaryVersion version)
        {
            Ensure.Argument.NotNull(signature);
            this.Signature = signature;
            this.Version = version;
        }

        public string Signature { get; set; }

        public BinaryVersion Version { get; set; }

        public string PreprocessedTag { get; set; }

        public SoundArchiveBinary Build(
            SoundArchiveContext context,
            IDictionary<IOutputItem, object> fileEntityDictionary,
            object[] fileEntities)
        {
            Ensure.Argument.NotNull(context);
            Ensure.Argument.NotNull(fileEntityDictionary);
            Ensure.Argument.NotNull(fileEntities);

            SoundArchiveBinary file = new SoundArchiveBinary(
                this.Signature,
                this.Version.Major,
                this.Version.Minor,
                this.Version.Micro,
                this.Version.BinaryBugFix)
            {
                ExcludeStringBlock = context.Settings.ExcludeStringTable,
            };

            if (!file.ExcludeStringBlock)
            {
                BuildStringBlock(context, file.StringBlock.Body);
            }
            BuildInfoBlock(context, file.InfoBlock.Body, fileEntityDictionary, fileEntities);
            BuildFileBlock(file.FileBlock.Body, fileEntities);

            return file;
        }

        private void BuildStringBlock(SoundArchiveContext context, StringBlockBody body)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(body);

            this.BuildStringTable(context, body.StringTable);

            PatriciaTree patriciaTree = new PatriciaTree();

            this.BuildPatriciaTree(patriciaTree, context.AllSounds);
            this.BuildPatriciaTree(patriciaTree, context.AllSoundGroups);
            this.BuildPatriciaTree(patriciaTree, context.SoundSetBanks);
            this.BuildPatriciaTree(patriciaTree, context.Players);
            this.BuildPatriciaTree(patriciaTree, context.Groups);
            this.BuildPatriciaTree(patriciaTree, context.WaveArchives);

            this.BuildPatriciaTreeBinary(body.PatriciaTree, patriciaTree);
        }

        private void BuildInfoBlock(
            SoundArchiveContext context, InfoBlockBody body,
            IDictionary<IOutputItem, object> fileEntityDictionary, object[] fileEntities)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(body);
            Assertion.Argument.NotNull(fileEntities);

            this.BuildSoundInfoSection(context, body.SoundInfoSection);
            this.BuildBankInfoSection(context, body.BankInfoSection);
            this.BuildWaveArchiveInfoSection(context, body.WaveArchiveInfoSection);
            this.BuildPlayerInfoSection(context, body.PlayerInfoSection);
            this.BuildSoundGroupInfoSection(context, body.SoundGroupInfoSection);
            this.BuildGroupInfoSection(context, body.GroupInfoSection);
            this.BuildFileInfoSection(context, body.FileInfoSection, fileEntityDictionary, fileEntities);
            this.BuildSoundArchivePlayerInfoSection(context, body.SoundArchivePlayerInfoSection);
        }

        private void BuildFileBlock(FileBlockBody body, object[] fileEntities)
        {
            Assertion.Argument.NotNull(body);
            Assertion.Argument.NotNull(fileEntities);

            foreach (object fileEntry in fileEntities)
            {
                body.Items.Add(fileEntry);
            }
        }

        //-----------------------------------------------------------------
        // 文字列ブロック
        //-----------------------------------------------------------------

        private void BuildStringTable(SoundArchiveContext context, StringTable stringTable)
        {
            Assertion.Argument.NotNull(stringTable);

            foreach (SoundSetItem soundSetItem in context.StringTableItems)
            {
                stringTable.Items.Add(new NullTerminateString(soundSetItem.Name));
            }
        }

        private void BuildPatriciaTree(PatriciaTree patriciaTree, IEnumerable soundSetItems)
        {
            Assertion.Argument.NotNull(patriciaTree);
            Assertion.Argument.NotNull(soundSetItems);

            foreach (SoundSetItem soundSetItem in soundSetItems)
            {
                if (soundSetItem.Name.EndsWith(SoundArchiveContext.AutoGeneratedNamePostfix))
                {
                    continue;
                }

                patriciaTree.Add(new PatriciaTreeLeaf(soundSetItem));
            }
        }

        private void BuildPatriciaTreeBinary(PatriciaTreeBinary treeBinary, PatriciaTree patriciaTree)
        {
            Assertion.Argument.NotNull(treeBinary);
            Assertion.Argument.NotNull(patriciaTree);

            treeBinary.RootIndex = (patriciaTree.Root == null) ?
                (UInt32)0xFFFFFFFF : (UInt32)patriciaTree.Root.Index;

            treeBinary.Nodes = (from PatriciaTree.INode node in patriciaTree.Nodes
                                select this.CreatePatriciaTreeNodeBinary(node)).ToList();
        }

        private PatriciaTreeNodeBinary CreatePatriciaTreeNodeBinary(PatriciaTree.INode node)
        {
            Assertion.Argument.NotNull(node);

            PatriciaTreeLeaf leaf = (node is PatriciaTree.ILeaf) ?
                (node as PatriciaTree.ILeaf).Item as PatriciaTreeLeaf : null;

            UInt32 stringID = (UInt32)0xFFFFFFFF;
            if (leaf != null && leaf.SoundSetItem.Parameters.ContainsKey(ConversionParameterNames.StringID))
            {
                stringID = leaf.SoundSetItem.GetStringID();
            }

            return new PatriciaTreeNodeBinary()
            {
                Node = node,
                StringID = stringID,
                ItemID = (leaf == null) ? ItemID.InvalidValue : leaf.SoundSetItem.ID,
            };
        }

        //-----------------------------------------------------------------
        // サウンド情報セクションの構築
        //-----------------------------------------------------------------

        private void BuildSoundInfoSection(SoundArchiveContext context, SoundInfoSection infoSection)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(infoSection);

            foreach (StreamSound sound in context.StreamSounds)
            {
                infoSection.Items.Add(this.CreateStreamSoundInfo(context, sound));
            }

            foreach (WaveSoundBase sound in context.WaveSounds)
            {
                infoSection.Items.Add(this.CreateWaveSoundInfo(context, sound));
            }

            foreach (SequenceSoundBase sound in context.SequenceSounds)
            {
                infoSection.Items.Add(this.CreateSequenceSoundInfo(context, sound));
            }
        }

        private SoundInfo CreateStreamSoundInfo(SoundArchiveContext context, StreamSound sound)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(sound);

            SoundInfo info = new SoundInfo();
            SetSoundInfo(context, sound, info);

            info.Parameters.PanParam = new ParameterPanParam()
            {
                PanMode = sound.PanMode,
                PanCurve = sound.PanCurve,
            };

            if (sound.GetStringID() != 0xFFFFFFFF)
            {
                info.Parameters.StringID = new ParameterUInt32()
                {
                    Value = sound.GetStringID(),
                };
            }

            StreamSoundInfo paramStreamSound = new StreamSoundInfo()
            {
                TrackAllocationFlags = sound.GetTrackAllocationFlags(),
                TotalChannelCount = (UInt16)sound.GetTotalChannelCount(),
                Pitch = sound.Pitch,
                SendInfo = new SendInfo()
                {
                    MainSend = (byte)sound.Sends.MainSend,
                    FxSends = new byte[]
                    {
                        (byte)sound.Sends.AuxASend,
                        (byte)sound.Sends.AuxBSend,
                        (byte)sound.Sends.AuxCSend,
                    },
                },
            };

            var prefetchOutput = sound.GetPrefetchOutputTarget();

            if (prefetchOutput != null)
            {
                paramStreamSound.PrefechFileID = context.GetFileID(context.GetFile(prefetchOutput));
            }

            info.TypeDependentInfo = paramStreamSound;

            uint channelIndex = 0;

            foreach (StreamSoundTrackBase track in sound.Children.Where<Component>(item => item.IsEnabled))
            {
                paramStreamSound.TrackInformationTable.Items.Add(
                    this.CreateTrackInfo(track as StreamSoundTrack, channelIndex)
                    );

                channelIndex += track.GetChannelCount();
            }

            // Adts, Opus の場合のみ設定します。
            switch (sound.GetContainerType())
            {
                case ContainerType.Adts:
                    paramStreamSound.StreamSoundExtensionInfo = new StreamSoundExtensionInfo();
                    paramStreamSound.StreamSoundExtensionInfo.StreamTypeInfo.ContainerType = sound.GetContainerType();
                    paramStreamSound.StreamSoundExtensionInfo.StreamTypeInfo.DecodeType = DecodeType.Default;

                    this.ApplyStreamSoundLoopInfo(
                        sound,
                        paramStreamSound.StreamSoundExtensionInfo,
                        ((StreamSoundTrackBase)sound.Children.First<Component>(item => item.IsEnabled)).FilePath);
                    break;
                case ContainerType.Opus:
                    paramStreamSound.StreamSoundExtensionInfo = new StreamSoundExtensionInfo();
                    paramStreamSound.StreamSoundExtensionInfo.StreamTypeInfo.ContainerType = sound.GetContainerType();
                    if (context.Project.DoUseHardwareOpusDecoder)
                    {
                        paramStreamSound.StreamSoundExtensionInfo.StreamTypeInfo.DecodeType = DecodeType.Accelerator;
                    }
                    else
                    {
                        paramStreamSound.StreamSoundExtensionInfo.StreamTypeInfo.DecodeType = DecodeType.Cpu;
                    }

                    this.ApplyStreamSoundLoopInfo(
                        sound,
                        paramStreamSound.StreamSoundExtensionInfo,
                        ((StreamSoundTrackBase)sound.Children.First<Component>(item => item.IsEnabled)).FilePath);
                    break;
            }

            info.TypeDependentInfo = paramStreamSound;

            return info;
        }

        private StreamSoundTrackInfo CreateTrackInfo(StreamSoundTrack track, uint channelIndexStart)
        {
            Assertion.Argument.NotNull(track);
            Assertion.Argument.True(channelIndexStart >= 0);

            StreamSoundTrackInfo trackInfo = new StreamSoundTrackInfo()
            {
                Volume = (byte)track.Volume,
                Pan = (byte)track.Pan,
                SurroundPan = (byte)track.SurroundPan,
                SurroundMode = this.GetSurroundMode(track),
                LpfFreqency = (byte)track.LPF,
                BiquadType = track.BiquadType,
                BiquadValue = (byte)track.Biquad,
            };

            trackInfo.SendInfo = new SendInfo()
            {
                MainSend = (byte)track.Sends.MainSend,
                FxSends = new byte[]
                {
                    (byte)track.Sends.AuxASend,
                    (byte)track.Sends.AuxBSend,
                    (byte)track.Sends.AuxCSend,
                },
            };

            for (uint i = 0; i < track.GetChannelCount(); i++)
            {
                trackInfo.GlobalChannelIndexTable.Items.Add((byte)(channelIndexStart + i));
            }

            return trackInfo;
        }

        private byte GetSurroundMode(StreamSoundTrackBase track)
        {
            Assertion.Argument.NotNull(track);

#if false   // TODO : Cockpit 対応されたら有効化する
            if (!string.IsNullOrEmpty(this.PreprocessedTag))
            {
                if (track.IsPreprocessed(this.PreprocessedTag))
                {
                    return (byte)StreamSoundTrackInfo.SurroundModeType.Preprocessed;
                }
            }
#endif

            return track.FrontBypass ?
                (byte)StreamSoundTrackInfo.SurroundModeType.FrontBypass : (byte)StreamSoundTrackInfo.SurroundModeType.Normal;
        }

        private void ApplyStreamSoundLoopInfo(StreamSound sound, StreamSoundExtensionInfo extensionInfo, string filePath)
        {
            string waveFilePath = StreamSoundExtension.GetLinearPcmWaveFilePath(filePath);

            if (string.IsNullOrEmpty(waveFilePath))
            {
                return;
            }

            AdtsHeader adtsHeader = null;

            using (var fileStream = System.IO.File.Open(
                filePath,
                System.IO.FileMode.Open,
                System.IO.FileAccess.Read))
            {
                adtsHeader = AdtsHeader.Parse(fileStream);

                if (!adtsHeader.IsValid())
                {
                    adtsHeader = null;
                }
            }

            extensionInfo.StreamTypeInfo.IsLooped = sound.GetHasLoop();

            if (extensionInfo.StreamTypeInfo.IsLooped)
            {
                extensionInfo.LoopStartFrame = sound.GetLoopStartFrame();
                extensionInfo.LoopEndFrame = sound.GetLoopEndFrame();
            }
        }

        private SoundInfo CreateWaveSoundInfo(SoundArchiveContext context, WaveSoundBase sound)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(sound);

            SoundInfo info = new SoundInfo();
            SetSoundInfo(context, sound, info);

            info.Parameters.PanParam = new ParameterPanParam()
            {
                PanMode = sound.PanMode,
                PanCurve = sound.PanCurve,
            };

            if (sound.GetStringID() != 0xFFFFFFFF)
            {
                info.Parameters.StringID = new ParameterUInt32()
                {
                    Value = sound.GetStringID(),
                };
            }

            if (context.Traits.IsWaveSound2BinaryEnabled)
            {
                info.TypeDependentInfo = new WaveSound2Info()
                {
                    WaveArchiveID = ((WaveSoundSetBase)sound.Parent).GetWaveArchiveDictionary().Values.First().ID,
                };
            }
            else
            {
                WaveSoundInfo paramWaveSound = new WaveSoundInfo()
                {
                    Index = sound.GetIndex(),
                    AllocateTrackFlags = 1,     // 1 固定
                };

                paramWaveSound.Parameters.Priority = new ParameterSoundPriority()
                {
                    ChannelPriority = (Byte)sound.ChannelPriority,
                    ReleasePriorityFixed = sound.ReleasePriorityFixed ? (Byte)1 : (Byte)0,
                };

                info.TypeDependentInfo = paramWaveSound;
            }

            return info;
        }

        private SoundInfo CreateSequenceSoundInfo(SoundArchiveContext context, SequenceSoundBase sound)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(sound);

            SoundInfo info = new SoundInfo();
            SetSoundInfo(context, sound, info);

            info.Parameters.PanParam = new ParameterPanParam()
            {
                PanMode = PanMode.Dual,     // 固定値
                PanCurve = PanCurve.Sqrt,   // 固定値
            };

            if (sound.GetStringID() != 0xFFFFFFFF)
            {
                info.Parameters.StringID = new ParameterUInt32()
                {
                    Value = sound.GetStringID(),
                };
            }

            SequenceSoundInfo paramSequenceSound = new SequenceSoundInfo()
            {
                AllocateTrackFlags = sound.GetAllocateTrackFlags(),
            };

            Ensure.Operation.True(
                paramSequenceSound.AllocateTrackFlags <= 0xFFFF,
                "sequence sound allocate track flags are invalid."
                );

            bool foundValidBank = false;
            foreach (SoundSetBankBase soundSetBank in sound.SoundSetBanks.Reverse())
            {
                // 末尾の欠番バンクは出力しない
                if (!foundValidBank && soundSetBank == null)
                {
                    continue;
                }

                foundValidBank = true;

                // バンクの欠番をゆるす
                paramSequenceSound.BankIDTable.Items.Insert(
                    0,
                    soundSetBank == null ? ItemID.InvalidValue : soundSetBank.ID
                    );
            }

            paramSequenceSound.Parameters.StartOffset = new ParameterUInt32()
            {
                Value = sound.GetStartOffset(),
            };

            paramSequenceSound.Parameters.Priority = new ParameterSoundPriority()
            {
                ChannelPriority = (Byte)sound.ChannelPriority,
                ReleasePriorityFixed = sound.ReleasePriorityFixed ? (Byte)1 : (Byte)0,
            };

            info.TypeDependentInfo = paramSequenceSound;

            return info;
        }

        private void SetSoundInfo(SoundArchiveContext context, Sound sound, SoundInfo soundInfo)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(sound);
            Assertion.Argument.NotNull(soundInfo);

            soundInfo.FileID = context.GetFileID(sound);
            soundInfo.PlayerID = sound.TargetPlayer.ID;
            soundInfo.Volume = (Byte)sound.Volume;

            if (sound is StreamSound)
            {
                soundInfo.RemoteFilter = (Byte)((StreamSound)sound).RemoteFilter;
            }
            else if (sound is SequenceSound)
            {
                soundInfo.RemoteFilter = (Byte)((SequenceSound)sound).RemoteFilter;
            }
            else if (sound is WaveSound)
            {
                soundInfo.RemoteFilter = (Byte)((WaveSound)sound).RemoteFilter;
            }

            soundInfo.Parameters.Sound3D = new ParameterSound3D()
            {
                DecayRatio = sound.Sound3D.DecayRatio3D,
                DecayCurveType = sound.Sound3D.DecayCurve3D,
                DopplerFactor = (Byte)sound.Sound3D.DopplerFactor3D,
                Volume3D = sound.Sound3D.Enable3DVolume,
                Priority3D = sound.Sound3D.Enable3DPriority,
                Pan3D = sound.Sound3D.Enable3DPan,
                SPan3D = sound.Sound3D.Enable3DSurroundPan,
                Filter3D = sound.Sound3D.Enable3DFilter
            };

            soundInfo.Parameters.PlayerParam = new ParameterPlayerParam()
            {
                PlayerPriority = (Byte)sound.PlayerPriority,
                ActoryPlayerID = (Byte)sound.ActorPlayer,
            };

            if (sound.SinglePlayType != SinglePlayType.None)
            {
                soundInfo.Parameters.SinglePlayParam = new ParameterSoundSinglePlay()
                {
                    SinglePlayType = sound.SinglePlayType,
                    EffectiveDuration = (short)sound.SinglePlayEffectiveDuration,
                };
            }

            if (!(sound is StreamSoundBase) && sound.Parameters.ContainsKey(ProjectParameterNames.FrontBypass))
            {
                soundInfo.Parameters.FrontBypass =
                    new ParameterUInt32()
                    {
                        Value = (UInt32)((bool)(sound.Parameters[ProjectParameterNames.FrontBypass].Value) ? 1 : 0)
                    };
            }

            if (context.Project.UserDataStructureSettings.Settings.Count >= 1 &&
                context.Project.UserDataStructureSettings.Settings[0].Enabled)
            {
                soundInfo.Parameters.UserParam = new ParameterUInt32() { Value = (UInt32)sound.UserParameter };
            }

            if (context.Project.UserDataStructureSettings.Settings.Count >= 2 &&
                context.Project.UserDataStructureSettings.Settings[1].Enabled)
            {
                soundInfo.Parameters.UserParam1 = new ParameterUInt32() { Value = (UInt32)sound.UserParameter1 };
            }

            if (context.Project.UserDataStructureSettings.Settings.Count >= 3 &&
                context.Project.UserDataStructureSettings.Settings[2].Enabled)
            {
                soundInfo.Parameters.UserParam2 = new ParameterUInt32() { Value = (UInt32)sound.UserParameter2 };
            }

            if (context.Project.UserDataStructureSettings.Settings.Count >= 4 &&
                context.Project.UserDataStructureSettings.Settings[3].Enabled)
            {
                soundInfo.Parameters.UserParam3 = new ParameterUInt32() { Value = (UInt32)sound.UserParameter3 };
            }
        }

        //-----------------------------------------------------------------
        // バンク情報セクションの構築
        //-----------------------------------------------------------------

        private void BuildBankInfoSection(SoundArchiveContext context, BankInfoSection infoSection)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(infoSection);

            foreach (SoundSetBankBase soundSetBank in context.SoundSetBanks)
            {
                infoSection.Items.Add(this.CreateBankInfo(context, soundSetBank));
            }
        }

        private BankInfo CreateBankInfo(SoundArchiveContext context, SoundSetBankBase soundSetBank)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(soundSetBank);

            // メインサウンドアーカイブのバンクを追加サウンドアーカイブで参照する場合はバンク情報のみを出力するので、FileID を無効値にする
            BankInfo info = new BankInfo()
            {
                FileID = context.IsExternalSoundArchiveBank(soundSetBank)
                    ? InvalidFileID : context.GetFileID(soundSetBank),
            };

            if (soundSetBank.WaveArchiveReference != WaveArchiveConsts.AutoShared)
            {
                foreach (WaveArchiveBase waveArchive in soundSetBank.GetWaveArchiveDictionary().Values)
                {
                    info.WaveArchiveIDTable.Items.Add(waveArchive.ID);
                }
            }

            if (soundSetBank.GetStringID() != 0xFFFFFFFF)
            {
                info.Parameters.StringID = new ParameterUInt32()
                {
                    Value = soundSetBank.GetStringID(),
                };
            }

            return info;
        }

        //-----------------------------------------------------------------
        // 波形アーカイブ情報セクションの構築
        //-----------------------------------------------------------------

        private void BuildWaveArchiveInfoSection(SoundArchiveContext context, WaveArchiveInfoSection infoSection)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(infoSection);

            foreach (WaveArchiveBase waveArchive in context.WaveArchives)
            {
                infoSection.Items.Add(this.CreateWaveArchiveInfo(context, waveArchive));
            }
        }

        private WaveArchiveInfo CreateWaveArchiveInfo(SoundArchiveContext context, WaveArchiveBase waveArchive)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(waveArchive);

            WaveArchiveInfo info = new WaveArchiveInfo()
            {
                FileID = context.GetFileID(waveArchive),
                IsLoadTypeIndividual = (waveArchive.LoadType == WaveArchiveLoadType.Individual),
            };

            if (waveArchive.GetStringID() != 0xFFFFFFFF)
            {
                info.Parameters.StringID = new ParameterUInt32()
                {
                    Value = waveArchive.GetStringID(),
                };
            }

            if (info.IsLoadTypeIndividual)
            {
                info.Parameters.WaveCount = new ParameterUInt32()
                {
                    Value = (uint)waveArchive.GetItemOutputTargets().Count,
                };
            }
            return info;
        }

        //-----------------------------------------------------------------
        // プレイヤー情報セクションの構築
        //-----------------------------------------------------------------

        private void BuildPlayerInfoSection(SoundArchiveContext context, PlayerInfoSection infoSection)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(infoSection);

            foreach (PlayerBase player in context.Players)
            {
                infoSection.Items.Add(this.CreatePlayerInfo(context, player));
            }
        }

        private PlayerInfo CreatePlayerInfo(SoundArchiveContext context, PlayerBase player)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(player);

            PlayerInfo info = new PlayerInfo()
            {
                PlayableSoundMax = (UInt32)player.SoundLimit,
            };

            info.Parameters.HeapSize = new ParameterUInt32()
            {
                Value = (UInt32)player.HeapSize,
            };

            if (player.GetStringID() != 0xFFFFFFFF)
            {
                info.Parameters.StringID = new ParameterUInt32()
                {
                    Value = player.GetStringID(),
                };
            }

            return info;
        }

        //-----------------------------------------------------------------
        // サウンドグループ情報セクションの構築
        //-----------------------------------------------------------------

        private void BuildSoundGroupInfoSection(SoundArchiveContext context, SoundGroupInfoSection infoSection)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(infoSection);

            // TODO : WaveSoundResource を利用する場合は、WaveSoundGroupInfo を出力しないようにする（問題があれば修正する）
            if (!context.Traits.IsWaveSound2BinaryEnabled)
            {
                foreach (WaveSoundSetBase waveSoundSet in context.WaveSoundSets)
                {
                    infoSection.Items.Add(this.CreateSoundGroupInfo(context, waveSoundSet));
                }
            }

            foreach (SequenceSoundSetBase sequenceSoundSet in context.SequenceSoundSets)
            {
                infoSection.Items.Add(this.CreateSoundGroupInfo(context, sequenceSoundSet));
            }
        }

        private SoundGroupInfo CreateSoundGroupInfo(SoundArchiveContext context, WaveSoundSetBase waveSoundGroup)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(waveSoundGroup);

            SoundGroupInfo info = new SoundGroupInfo();
            WaveSoundBase firstSound = null;
            WaveSoundBase lastSound = null;

            foreach (WaveSoundBase sound in waveSoundGroup.Children)
            {
                if (!sound.IsConvertTarget())
                {
                    continue;
                }

                if (firstSound == null)
                {
                    firstSound = sound;
                }

                lastSound = sound;
            }

            if (firstSound != null && lastSound != null)
            {
                info.SoundIDFirst = firstSound.ID;
                info.SoundIDLast = lastSound.ID;
            }

            info.FileIDTable.Items.Add(context.GetFileID(waveSoundGroup));

            if (waveSoundGroup.GetStringID() != 0xFFFFFFFF)
            {
                info.Parameters.StringID = new ParameterUInt32()
                {
                    Value = waveSoundGroup.GetStringID(),
                };
            }

            WaveSoundGroupInfo waveSoundGroupInfo = new WaveSoundGroupInfo();

            if (waveSoundGroup.WaveArchiveReference != WaveArchiveConsts.AutoShared)
            {
                foreach (WaveArchiveBase waveArchive in waveSoundGroup.GetWaveArchiveDictionary().Values)
                {
                    waveSoundGroupInfo.WaveArchiveIDTable.Items.Add(waveArchive.ID);
                }
            }

            info.TypeDependentInfo = waveSoundGroupInfo;

            return info;
        }

        private SoundGroupInfo CreateSoundGroupInfo(SoundArchiveContext context, SequenceSoundSetBase sequenceSoundGroup)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(sequenceSoundGroup);

            SoundGroupInfo info = new SoundGroupInfo();
            HashSet<UInt32> fileIDs = new HashSet<uint>();
            SequenceSoundBase firstSound = null;
            SequenceSoundBase lastSound = null;

            foreach (SequenceSoundBase sound in sequenceSoundGroup.Children)
            {
                if (!sound.IsConvertTarget())
                {
                    continue;
                }

                if (firstSound == null)
                {
                    firstSound = sound;
                }

                lastSound = sound;

                UInt32 fileID = context.GetFileID(sound);

                if (fileIDs.Contains(fileID))
                {
                    continue;
                }

                fileIDs.Add(fileID);
                info.FileIDTable.Items.Add(fileID);
            }

            if (firstSound != null && lastSound != null)
            {
                info.SoundIDFirst = firstSound.ID;
                info.SoundIDLast = lastSound.ID;
            }

            if (sequenceSoundGroup.GetStringID() != 0xFFFFFFFF)
            {
                info.Parameters.StringID = new ParameterUInt32()
                {
                    Value = sequenceSoundGroup.GetStringID(),
                };
            }

            return info;
        }

        //-----------------------------------------------------------------
        // グループ情報セクションの構築
        //-----------------------------------------------------------------

        private void BuildGroupInfoSection(SoundArchiveContext context, GroupInfoSection infoSection)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(infoSection);

            foreach (GroupBase group in context.Groups)
            {
                infoSection.Items.Add(this.CreateGroupInfo(context, group));
            }
        }

        private GroupInfo CreateGroupInfo(SoundArchiveContext context, GroupBase group)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(group);

            uint fileID = 0xFFFFFFFF;

            if (group.OutputType != GroupOutputType.UserManagement)
            {
                fileID = context.GetFileID(group);
            }

            GroupInfo info = new GroupInfo()
            {
                FileID = fileID,
            };

            if (group.GetStringID() != 0xFFFFFFFF)
            {
                info.Parameters.StringID = new ParameterUInt32()
                {
                    Value = group.GetStringID(),
                };
            }

            return info;
        }

        //-----------------------------------------------------------------
        // ファイル情報セクションの構築
        //-----------------------------------------------------------------

        private void BuildFileInfoSection(
            SoundArchiveContext context, FileInfoSection infoSection,
            IDictionary<IOutputItem, object> fileEntityDictionary, object[] fileEntities)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(infoSection);
            Assertion.Argument.NotNull(fileEntityDictionary);
            Assertion.Argument.NotNull(fileEntities);

            foreach (ComponentFile file in context.Files)
            {
                var fileInfo = this.CreateFileInfo(context, file, fileEntityDictionary);
                Ensure.Operation.ObjectNotNull(fileInfo);

                infoSection.Items.Add(fileInfo);
            }
        }

        private FileInfo CreateFileInfo(
            SoundArchiveContext context, ComponentFile file,
            IDictionary<IOutputItem, object> fileEntityDictionary)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(file);
            Assertion.Argument.NotNull(fileEntityDictionary);

            FileInfo info = new FileInfo();

            // オリジナルファイルがあれば登録します。
            IOutputItem originalOutputItem = null;
            file.OutputTarget.ItemDictionary.TryGetValue(string.Empty, out originalOutputItem);

            if (file.ExternalFilePath.Length == 0)
            {
                object internalFile = null;

                if (originalOutputItem != null)
                {
                    // 参照ファイルが見つからない場合は、
                    // ユーザー管理グループに含まれているとみなし、ファイル参照なしで続行します。
                    fileEntityDictionary.TryGetValue(originalOutputItem, out internalFile);
                }

                var internalFileInfo = new InternalFileInfo()
                {
                    File = internalFile,
                };

                foreach (SoundSetItem soundSetItem in file.Components)
                {
                    foreach (GroupBase attachedGroup in soundSetItem.GetAttachedGroups())
                    {
                        internalFileInfo.GroupIDTable.Items.Add(attachedGroup.ID);
                    }
                }

                info.OriginalFileInfo = internalFileInfo;
            }
            else
            {
                info.OriginalFileInfo = new ExternalFileInfo()
                {
                    FilePath = new NullTerminateString(file.ExternalFilePath),
                };
            }

            return info;
        }

        //-----------------------------------------------------------------
        // サウンドアーカイブプレイヤー情報セクションの構築
        //-----------------------------------------------------------------

        private void BuildSoundArchivePlayerInfoSection(SoundArchiveContext context, SoundArchivePlayerInfo info)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(info);

            info.SequenceSoundMax = (UInt16)context.Project.SoundArchivePlayerSequenceSoundCount;
            info.SequenceTrackMax = (UInt16)context.Project.SoundArchivePlayerSequenceTrackCount;
            info.StreamChannelMax = (UInt16)context.Project.SoundArchivePlayerStreamChannelCount;
            info.StreamSoundMax = (UInt16)context.Project.SoundArchivePlayerStreamSoundCount;
            info.StreamTrackMax = (UInt16)(context.Project.SoundArchivePlayerStreamSoundCount * 8); // 8track
            info.WaveSoundMax = (UInt16)context.Project.SoundArchivePlayerWaveSoundCount;
            info.WaveTrackMax = (UInt16)context.Project.SoundArchivePlayerWaveSoundCount;

            info.StreamBufferTimes = (byte)context.Project.SoundArchivePlayerStreamBufferTimes;

            if (context.Traits.IsWaveSound2BinaryEnabled)
            {
                info.DevelopFlags |= 1 << SoundArchivePlayerInfo.DevelopFlagBits.EnableWaveSound2Bit;
            }
        }

        //-----------------------------------------------------------------
        // パトリシア木 リーフクラス
        //-----------------------------------------------------------------

        private class PatriciaTreeLeaf : IPatriciaTreeItem
        {
            public PatriciaTreeLeaf(SoundSetItem soundSetItem)
            {
                Ensure.Argument.NotNull(soundSetItem);
                this.SoundSetItem = soundSetItem;
            }

            public SoundSetItem SoundSetItem { get; private set; }

            /// <summary>
            /// アイテムのキーを取得します。
            /// </summary>
            public string Key
            {
                get { return this.SoundSetItem.Name; }
            }
        }
    }
}
