﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.Collections.Generic;
    using NintendoWare.SoundFoundation.Core.Collections;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.Collections;

    /// <summary>
    /// 各バイナリ構築に共通して必要なコンテキスト情報を格納します。
    /// </summary>
    public abstract class ComponentContext : ConversionContext
    {
        public const string IntermediateOutputPrefix = ".";
        public const string AutoGeneratedNamePostfix = "@AutoGenerated";

        private Dictionary<IOutput, Dictionary<string, IComponentProcessor>> processorDictionary =
            new Dictionary<IOutput, Dictionary<string, IComponentProcessor>>();

        private Dictionary<IOutput, ComponentFile> output2FileDictionary =
            new Dictionary<IOutput, ComponentFile>();
        private FileList files = new FileList();

        private Dictionary<IOutput, ComponentFile> output2WaveFileDictionary =
            new Dictionary<IOutput, ComponentFile>();
        private FileList waveFiles = new FileList();

        public ComponentContext(ComponentConversionTraits traits)
            : base(traits)
        {
        }

        public ComponentContext(ComponentConversionTraits traits, string projectFilePath)
            : base(traits, projectFilePath)
        {
        }

        public new ComponentConversionTraits Traits
        {
            get { return base.Traits as ComponentConversionTraits; }
        }

        public IList<ComponentFile> Files
        {
            get { return this.files; }
        }

        public IList<ComponentFile> WaveFiles
        {
            get { return this.waveFiles; }
        }

        public bool ContainsFile(IOutput outputTarget)
        {
            Ensure.Argument.NotNull(outputTarget);
            return this.output2FileDictionary.ContainsKey(outputTarget);
        }

        public ComponentFile GetFile(IOutput outputTarget)
        {
            Ensure.Argument.NotNull(outputTarget);
            return this.output2FileDictionary[outputTarget];
        }

        public bool ContainsWaveFile(IOutput outputTarget)
        {
            Ensure.Argument.NotNull(outputTarget);
            return this.output2WaveFileDictionary.ContainsKey(outputTarget);
        }

        public ComponentFile GetWaveFile(IOutput outputTarget)
        {
            Ensure.Argument.NotNull(outputTarget);
            return this.output2WaveFileDictionary[outputTarget];
        }

        public virtual ComponentFile AddFile(Component component, IOutput outputTarget)
        {
            Ensure.Argument.NotNull(component);
            Ensure.Argument.NotNull(outputTarget);

            ComponentFile file = null;
            this.output2FileDictionary.TryGetValue(outputTarget, out file);

            if (file == null)
            {
                file = new ComponentFile(outputTarget);
                this.files.Add(file);
                this.output2FileDictionary.Add(outputTarget, file);
            }

            if (!file.Components.Contains(component))
            {
                file.Components.Add(component);
            }

            return file;
        }

        public virtual ComponentFile AddExternalFile(
            Component component, IOutput outputTarget, string filePath)
        {
            Ensure.Argument.NotNull(component);
            Ensure.Argument.NotNull(outputTarget);
            Ensure.Argument.StringNotEmpty(filePath);

            ComponentFile file = null;
            this.output2FileDictionary.TryGetValue(outputTarget, out file);

            if (file == null)
            {
                file = new ComponentFile(outputTarget)
                {
                    ExternalFilePath = filePath,
                };

                this.files.Add(file);
                this.output2FileDictionary.Add(outputTarget, file);
            }

            if (!file.Components.Contains(component))
            {
                file.Components.Add(component);
            }

            return file;
        }

        public virtual ComponentFile AddWaveFile(Component component, IOutput outputTarget)
        {
            Ensure.Argument.NotNull(component);
            Ensure.Argument.NotNull(outputTarget);

            ComponentFile file = null;
            this.output2WaveFileDictionary.TryGetValue(outputTarget, out file);

            if (file == null)
            {
                file = new ComponentFile(outputTarget);
                this.waveFiles.Add(file);
                this.output2WaveFileDictionary.Add(outputTarget, file);
            }

            if (!file.Components.Contains(component))
            {
                file.Components.Add(component);
            }

            return file;
        }

        public bool ContainsProcessors(IOutput output)
        {
            Ensure.Argument.NotNull(output);
            return this.ContainsProcessor(output, string.Empty);
        }

        public bool ContainsProcessor(IOutput output, string outputID)
        {
            Ensure.Argument.NotNull(output);
            Ensure.Argument.NotNull(outputID);

            Dictionary<string, IComponentProcessor> dictionary = null;
            this.processorDictionary.TryGetValue(output, out dictionary);

            if (dictionary == null)
            {
                return false;
            }

            return this.processorDictionary[output].ContainsKey(outputID);
        }

        public IEnumerable<IComponentProcessor> GetProcessors(IOutput output)
        {
            Ensure.Argument.NotNull(output);
            return this.processorDictionary[output].Values;
        }

        public IComponentProcessor GetProcessor(IOutput output, string outputID)
        {
            try
            {
                Ensure.Argument.NotNull(output);
                Ensure.Argument.NotNull(outputID);
            }
            catch
            {
                throw;
            }
            return this.processorDictionary[output][outputID];
        }

        public void AddComponentProcessor(IOutput output, string outputID, IComponentProcessor processor)
        {
            Ensure.Argument.NotNull(output);
            Ensure.Argument.NotNull(outputID);
            Ensure.Argument.NotNull(processor);

            if (!this.processorDictionary.ContainsKey(output))
            {
                this.processorDictionary.Add(output, new Dictionary<string, IComponentProcessor>());
            }

            IDictionary<string, IComponentProcessor> processors = this.processorDictionary[output];

            // 既に登録済みの ConversionProcessor がある場合は登録しません。
            if (processors.ContainsKey(outputID))
            {
                return;
            }

            this.processorDictionary[output].Add(outputID, processor);
            this.ConversionProcessors.Add(processor);
        }

        /// <summary>
        /// 番号付きファイルリストです。
        /// </summary>
        private class FileList : NumberedListDecorator<ComponentFile>
        {
            public FileList()
                : base(new ObservableList<ComponentFile>())
            {
            }
        }

        private class ComponentProcessorID
        {
            private Component component;
            private string outputID;

            public ComponentProcessorID(Component component, string outputID)
            {
                Ensure.Argument.NotNull(component);
                Ensure.Argument.NotNull(outputID);

                this.component = component;
                this.outputID = outputID;
            }

            public Component Component
            {
                get { return this.component; }
            }

            public string OutputID
            {
                get { return this.outputID; }
            }
        }

        /// <summary>
        /// ComponentPair が等しいかどうかを検証します。
        /// </summary>
        private class ComponentProcessorIDComparer : IEqualityComparer<ComponentProcessorID>
        {
            /// <summary>
            /// 指定したオブジェクトが等しいかどうかを判断します。
            /// </summary>
            /// <param name="x">比較対象の ComponentPair 型の第 1 オブジェクト。</param>
            /// <param name="y">比較対象の ComponentPair 型の第 2 オブジェクト。</param>
            /// <returns>指定したオブジェクトが等しい場合は true。それ以外の場合は false。</returns>
            public bool Equals(ComponentProcessorID x, ComponentProcessorID y)
            {
                return (x.Component == y.Component && x.OutputID == y.OutputID);
            }

            /// <summary>
            /// 指定したオブジェクトのハッシュ コードを返します。
            /// </summary>
            /// <param name="obj">ハッシュ コードが返される対象のオブジェクト。</param>
            /// <returns>指定したオブジェクトのハッシュ コード。</returns>
            public int GetHashCode(ComponentProcessorID obj)
            {
                Assertion.Argument.NotNull(obj);
                Assertion.Argument.NotNull(obj.Component);
                Assertion.Argument.NotNull(obj.OutputID);
                return obj.Component.GetHashCode() ^ obj.OutputID.GetHashCode();
            }
        }
    }
}
