﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using Core;
    using Logs;
    using Projects;
    using ToolDevelopmentKit;

    internal class WaveArchiveProcessorLinker : ComponentSetup<SoundArchiveContext, WaveArchiveBase>
    {
        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, WaveArchiveBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            IConversionProcessor processor = GetProcessor(context, component);

            foreach (Component item in component.GetItems())
            {
                Link(context, processor, item);
            }
        }

        private IConversionProcessor GetProcessor(SoundArchiveContext context, WaveArchiveBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            IOutput output = component.GetOutputTarget();
            ICollection<GroupBase> attachedGroups = component.GetAttachedGroups();

            // 自動生成されていない場合、または
            // 自動生成されたけども、どの埋め込みグループにも属していない場合は、
            // オリジナルファイルの ConversionProcessor を生成します。
            if (!component.IsAutoGenerated() || attachedGroups.Count == 0)
            {
                return context.GetProcessor(output, string.Empty);
            }

            // 自動生成され、埋め込みグループに属する場合は、
            // グループ内で１つの波形アーカイブにまとめられるので、
            // グループに最適化されたファイルの ConversionProcessor を生成します。
            if (attachedGroups.Count != 1)
            {
                throw new Exception("internal error : group shared wave archive must be attached to a group.");
            }

            return context.GetProcessor(output, attachedGroups.First().Name);
        }

        private void Link(SoundArchiveContext context, IConversionProcessor processor, Component component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(processor);
            Assertion.Argument.NotNull(component);

            bool isLinkSucceeded = false;

            foreach (var targetComponent in this.EnumerateTargetComponents(component, context.Traits.IsWaveSound2BinaryEnabled))
            {
                // TODO : このへんの WaveSoundClipConvertModel に対する ConversionProcessor リンクに不具合あり

                IOutput output = targetComponent.GetOutputTarget();
                if (output == null)
                {
                    throw new Exception("waveSound output not found.");
                }

                IConversionProcessor dependedProcessor = context.GetProcessor(output, string.Empty);

                if (dependedProcessor == null)
                {
                    context.Logger.AddLine(new InternalErrorLine("target processor not found.", targetComponent));
                    throw new Exception("internal error");
                }

                processor.Dependencies.Add(dependedProcessor);
                isLinkSucceeded = true;
            }

            if (!isLinkSucceeded)
            {
                throw new Exception("internal error : invalid wave archive item.");
            }
        }

        private IEnumerable<Component> EnumerateTargetComponents(Component component, bool isWaveSound2BinaryEnabled)
        {
            if (!(component is WaveSoundBase || component is VelocityRegion || component is WaveSoundClipConvertModel))
            {
                yield break;
            }

            if (!isWaveSound2BinaryEnabled || component is VelocityRegion || component is WaveSoundClipConvertModel)
            {
                yield return component;
            }
            else
            {
#if true
                // TODO ここにはこないはず
                throw new InvalidOperationException("WaveArcdhiveProcessorLinker : unexcepcted error.");
#else
                foreach (var waveSoundClip in ((WaveSound)component).EnumerateWaveComponents(isWaveSound2BinaryEnabled))
                {
                    yield return waveSoundClip;
                }
#endif
            }
        }
    }
}
