﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using Core;
    using Projects;
    using ToolDevelopmentKit;

    internal class WaveSoundProcessorFactory : ComponentSetup<SoundArchiveContext, WaveSoundSetBase>
    {
        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, WaveSoundSetBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            if (context.Traits.IsWaveSound2BinaryEnabled)
            {
                // WSD ごとに出力する場合
                foreach (var waveSound in component.Children.Cast<WaveSound>().Where(waveSound => waveSound.IsConvertTarget()))
                {
                    Ensure.Operation.True(waveSound.IsWaveSound2BinaryRequired(), "WaveSound must has WaveSoundResource.");

                    IOutput output = waveSound.GetOutputTarget();

                    // 波形アーカイブ設定だけは、WSDSET のものを使う
                    foreach (KeyValuePair<string, WaveArchiveBase> item in component.GetWaveArchiveDictionary())
                    {
                        IOutputItem outputItem = this.GetOutputTargetItem(waveSound, item.Key);

                        context.AddComponentProcessor(
                            output,
                            item.Key,
                            this.CreateProcessor(waveSound, item.Value, outputItem)
                            );
                    }
                }
            }
            else
            {
                // WSDSET で１つ出力する場合
                IOutput output = component.GetOutputTarget();

                foreach (KeyValuePair<string, WaveArchiveBase> item in component.GetWaveArchiveDictionary())
                {
                    IOutputItem outputItem = this.GetOutputTargetItem(component, item.Key);

                    context.AddComponentProcessor(
                        output,
                        item.Key,
                        this.CreateProcessor(component, item.Value, outputItem)
                        );
                }
            }
        }

        private WaveSoundProcessor CreateProcessor(
            WaveSoundSetBase component, WaveArchiveBase waveArchive, IOutputItem output)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(waveArchive);
            Assertion.Argument.NotNull(output);

            return new WaveSoundProcessor(component, waveArchive, output);
        }

        private WaveSound2Processor CreateProcessor(
            WaveSound component, WaveArchiveBase waveArchive, IOutputItem output)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(waveArchive);
            Assertion.Argument.NotNull(output);

            return new WaveSound2Processor(component, waveArchive, output);
        }

        private IOutputItem GetOutputTargetItem(Component component, string outputID)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(outputID);
            return component.GetOutputTarget().ItemDictionary[outputID];
        }
    }
}
