﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.IO;
    using Projects;
    using ToolDevelopmentKit;

    /// <summary>
    /// シーケンスサウンドの出力を生成します。
    /// </summary>
    internal class SequenceSoundOutputFactory : ComponentSetup<SoundArchiveContext, SequenceSoundBase>
    {
        private FileManager fileManager;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="fileManager">ファイルマネージャを指定します。</param>
        public SequenceSoundOutputFactory(FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);
            this.fileManager = fileManager;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, SequenceSoundBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            IOutput outputTarget = this.fileManager.GetOutput(component);
            component.SetOutputTarget(outputTarget);

            context.AddFile(component, outputTarget);

            // 中間テキストシーケンスサウンド出力
            if (component.FileType == SequenceSoundFileType.Smf &&
                !outputTarget.ItemDictionary.ContainsKey(SequenceSoundProcessorFactory.OutputID_Intermediate))
            {
                string fileName = CreateFileName(
                    component, this.fileManager.IntermeidateOutputTraits.TextSequenceSoundFileExtension);

                this.fileManager.RegisterCacheItem(
                    outputTarget, SequenceSoundProcessorFactory.OutputID_Intermediate, fileName);
            }

            // バイナリシーケンスサウンド出力
            if (!outputTarget.ItemDictionary.ContainsKey(string.Empty))
            {
                string fileName = CreateFileName(
                    component, this.fileManager.BinaryOutputTraits.SequenceSoundBinaryFileExtension);

                this.fileManager.RegisterCacheItem(outputTarget, string.Empty, fileName);
            }
        }

        /// <summary>
        /// コンポーネントの中間テキストシーケンスファイル名を作成します。
        /// </summary>
        /// <param name="component">コンポーネントを指定します。</param>
        /// <param name="extension">拡張子を指定します。</param>
        /// <returns>ファイル名を返します。</returns>
        private string CreateIntermediateFileName(SequenceSoundBase component, string extension)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(extension);
            Assertion.Argument.StringNotEmpty(extension);

            if (component.FileType != SequenceSoundFileType.Smf)
            {
                return string.Empty;
            }

            return Path.ChangeExtension(component.FilePath, extension);
        }

        /// <summary>
        /// コンポーネントのファイル名を作成します。
        /// </summary>
        /// <param name="component">コンポーネントを指定します。</param>
        /// <param name="extension">拡張子を指定します。</param>
        /// <returns>ファイル名を返します。</returns>
        private string CreateFileName(SequenceSoundBase component, string extension)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(extension);
            Assertion.Argument.StringNotEmpty(extension);

            string fileName = Path.GetFileNameWithoutExtension(component.FilePath);

            if (fileName.Length == 0)
            {
                throw new Exception("internal error : invalid file path.");
            }

            return fileName + "." + extension;
        }
    }
}
