﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.IO;
    using NintendoWare.SoundFoundation.Resources;
    using Projects;
    using ToolDevelopmentKit;

    internal class GroupOutputFactory : ComponentSetup<SoundArchiveContext, GroupBase>
    {
        private readonly FileManager fileManager;

        public GroupOutputFactory(FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);
            this.fileManager = fileManager;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, GroupBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            IOutput outputTarget = null;

            switch (component.OutputType)
            {
                case GroupOutputType.None:
                    return;

                case GroupOutputType.Embedding:
                case GroupOutputType.Link:
                    // SoundArchiveProcessor でグループをビルドするので、
                    // 空の出力対象を追加しておきます。
                    outputTarget = this.CreateEmptyOutput();
                    Assertion.Operation.ObjectNotNull(outputTarget);

                    component.SetOutputTarget(outputTarget);
                    context.AddFile(component, outputTarget);
                    break;

                case GroupOutputType.UserManagement:
                    outputTarget = this.CreateOutput(context, component);
                    Assertion.Operation.ObjectNotNull(outputTarget);

                    component.SetOutputTarget(outputTarget);
                    context.AddFile(component, outputTarget);
                    break;

                default:
                    throw new Exception("invalid groupOutputType");
            }
        }

        private IOutput CreateOutput(SoundArchiveContext context, GroupBase group)
        {
            var result = this.fileManager.GetOutput(group);

            string fileName = CreateFileName(group);

            string validUserManagementFilePath = Path.Combine(
                context.Project.ValidUserManagementFileOutputDirectoryPath,
                fileName
                ).Replace('\\', '/');

            if (validUserManagementFilePath.Length + 1 >= context.Traits.MaxFileName - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, validUserManagementFilePath),
                        group));
            }

            context.RegisteringOutputs.Add(
                new RegisteringOutputInfo(null, result, null, validUserManagementFilePath)
                );

            return result;
        }

        private IOutput CreateEmptyOutput()
        {
            FileOutput result = new FileOutput();
            result.ItemDictionary.Add(string.Empty, new FileOutputItem());

            return result;
        }

        /// <summary>
        /// グループバイナリのファイル名を作成します。
        /// </summary>
        /// <param name="group">グループを指定します。</param>
        /// <returns>ファイル名を返します。</returns>
        private string CreateFileName(GroupBase group)
        {
            Assertion.Argument.NotNull(group);
            Ensure.Argument.StringNotEmpty(
                this.fileManager.BinaryOutputTraits.GroupBinaryFileExtension);

            if (group.Name.Length == 0)
            {
                throw new Exception("internal error : invalid stream sound name.");
            }

            return string.Format("{0}.{1}", group.Name, this.fileManager.BinaryOutputTraits.GroupBinaryFileExtension);
        }
    }
}
