﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.Collections.Generic;
    using System.Linq;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.Collections;
    using NintendoWare.SoundFoundation.Logs;

    internal class WaveSoundSetItemGenerator : ComponentSetup<SoundArchiveContext, WaveSoundSetBase>
    {
        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, WaveSoundSetBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            // 部分コンバートの場合は、必ず個別波形アーカイブのみを生成します。
            if (context.Settings.IsConvertParts)
            {
                this.GenerateOriginalWaveArchive(context, component);
                return;
            }

            if (this.IsGenerateOriginalWaveArchive(component))
            {
                this.GenerateOriginalWaveArchive(context, component);
            }

            if (component.WaveArchiveReference == WaveArchiveConsts.AutoShared)
            {
                this.DelegateGroupWaveArchives(context, component);
            }
        }

        private void DelegateGroupWaveArchives(SoundArchiveContext context, WaveSoundSetBase waveSoundSet)
        {
            Assertion.Argument.NotNull(waveSoundSet);

            foreach (GroupBase group in waveSoundSet.GetAttachedGroups())
            {
                switch (group.OutputType)
                {
                    case GroupOutputType.Embedding:
                    case GroupOutputType.UserManagement:
                        break;

                    default:
                        continue;
                }

                WaveArchiveBase groupWaveArchive = group.GetTargetWaveArchive();

                waveSoundSet.GetWaveArchiveDictionary().Add(group.Name, groupWaveArchive);
                DelegateWaves(context, waveSoundSet, groupWaveArchive);
            }
        }

        private void DelegateWaves(SoundArchiveContext context, WaveSoundSetBase waveSoundSet, WaveArchiveBase waveArchive)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(waveSoundSet);
            Assertion.Argument.NotNull(waveArchive);

            WaveArchiveEx.ItemList waveArchiveItems = waveArchive.GetItems();

            foreach (var waveSound in waveSoundSet.Children
                .Cast<WaveSound>()
                .Where(waveSound => waveSound.IsConvertTarget()))
            {
                // WaveSound のクリップ利用設定が食い違っていたらエラー終了
                if (context.Traits.IsWaveSound2BinaryEnabled != waveSound.IsWaveSound2BinaryRequired())
                {
                    if (!waveSound.IsWaveSound2BinaryRequired())
                    {
                        context.Logger.AddLine(
                            new ErrorLine(
                                string.Format(Resources.MessageResourceCommon.Message_WaveSoundMustHasNoClips, waveSound.Name),
                                waveSound)
                        );
                    }
                    else
                    {
                        context.Logger.AddLine(
                            new ErrorLine(
                                string.Format(Resources.MessageResourceCommon.Message_WaveSoundMustNotHasNoClips, waveSound.Name),
                                waveSound)
                        );
                    }
                    return;
                }

                foreach (var waveComponent in waveSound.EnumerateWaveComponents(context.Traits.IsWaveSound2BinaryEnabled)
                    .Where(waveComponent => waveComponent.IsConvertTarget())
                    .Where(waveComponent => !waveArchiveItems.Contains(waveComponent)))
                {
                    waveArchiveItems.Add(waveComponent);
                }
            }
        }

        private bool IsGenerateOriginalWaveArchive(WaveSoundSetBase waveSoundSet)
        {
            Assertion.Argument.NotNull(waveSoundSet);

            switch (waveSoundSet.WaveArchiveReference)
            {
                case WaveArchiveConsts.AutoIndividual:
                    // 自動（個別）の場合は、必ずオリジナル波形アーカイブを作ります。
                    return true;

                case WaveArchiveConsts.AutoShared:
                    // 自動（共有）の場合で、
                    // どのグループにも含まれない or １つ以上のリンクグループにふくまれる
                    // 場合は、オリジナル波形アーカイブを作ります。
                    ICollection<GroupBase> groups = waveSoundSet.GetAttachedGroups();

                    if (groups.Count == 0)
                    {
                        return true;
                    }

                    foreach (GroupBase group in groups)
                    {
                        if (group.OutputType == GroupOutputType.Link)
                        {
                            return true;
                        }
                    }
                    return false;
            }

            // その他、手動共有の場合は、オリジナル波形アーカイブを作りません。
            return false;
        }

        private void GenerateOriginalWaveArchive(SoundArchiveContext context, WaveSoundSetBase waveSoundSet)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(waveSoundSet);

            WaveArchiveBase waveArchive = new WaveArchiveBase()
            {
                Name = string.Format(
                    "{0}_WaveArchive{1}",
                    waveSoundSet.Name, SoundArchiveContext.AutoGeneratedNamePostfix),
            };

            waveSoundSet.GetWaveArchiveDictionary().Add(string.Empty, waveArchive);
            DelegateWaves(context, waveSoundSet, waveArchive);

            waveArchive.ID = new BinaryItemID(ItemType.WaveArchive, (uint)context.WaveArchives.Count);
            context.WaveArchives.Add(waveArchive);
        }
    }
}
