﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// グループファイルを最適化します。
    /// </summary>
    internal class GroupFileOptimizer : ComponentSetup<SoundArchiveContext, GroupBase>
    {
        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, GroupBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            if (component.OutputType == GroupOutputType.None) { return; }

            IList<ComponentFile> files = component.GetItemFiles();

            foreach (SoundSetItem itemTarget in component.GetAllItemTargets(context.Traits.IsWaveSound2BinaryEnabled))
            {
                ComponentFile file = GetFile(context, itemTarget);

                if (files.Contains(file)) { continue; }
                files.Add(file);
            }

            WaveArchiveBase groupWaveArchive = component.GetTargetWaveArchive();

            // グループ専用の波形アーカイブがある場合は追加します。（埋め込みグループ）
            // それ以外の場合は、グループに追加登録するアイテムを検索します。（リンクグループ）
            if (groupWaveArchive != null)
            {
                ComponentFile file = GetFile(context, groupWaveArchive);

                if (!files.Contains(file))
                {
                    files.Add(file);
                }
            }
            else
            {
                this.AddGeneratedWaveArchives(context, component);
            }
        }

        /// <summary>
        /// 自動生成された波形アーカイブをグループアイテムとして登録します。
        /// </summary>
        /// <param name="context">コンテキストを指定します。</param>
        /// <param name="group">グループを指定します。</param>
        private void AddGeneratedWaveArchives(SoundArchiveContext context, GroupBase group)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(group);

            IList<ComponentFile> files = group.GetItemFiles();

            foreach (GroupItemBase item in group.GetAllItems())
            {
                if (item.Target == null || !item.Target.IsHierarchicalConvertTarget())
                {
                    continue;
                }

                if (!item.Target.Parameters.ContainsKey(ConversionParameterNames.TargetWaveArchive))
                {
                    continue;
                }

                IDictionary<string, WaveArchiveBase> waveArchives = item.Target.
                    Parameters[ConversionParameterNames.TargetWaveArchive].Value
                    as IDictionary<string, WaveArchiveBase>;

                WaveArchiveBase waveArchive = null;
                waveArchives.TryGetValue(string.Empty, out waveArchive);

                if (waveArchive == null)
                {
                    continue;
                }

                if (!waveArchive.Name.EndsWith(SoundArchiveContext.AutoGeneratedNamePostfix))
                {
                    continue;
                }


                ComponentFile file = GetFile(context, waveArchive);

                if (files.Contains(file)) { continue; }
                files.Add(file);
            }
        }

        /// <summary>
        /// 指定アイテムに関連付けられたファイルを取得します。
        /// </summary>
        /// <param name="context">コンテキストを指定します。</param>
        /// <param name="item">アイテムを指定します。</param>
        /// <returns>指定アイテムに関連付けられたファイルを返します。</returns>
        private ComponentFile GetFile(SoundArchiveContext context, SoundSetItem item)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(item);

            try
            {
                var streamSound = item as StreamSoundBase;
                return context.GetFile(
                    streamSound == null ? item.GetOutputTarget() : streamSound.GetPrefetchOutputTarget()
                    );
            }
            catch (Exception exception)
            {
                throw new Exception("internal error : group item file not found.", exception);
            }
        }
    }
}
