﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------


namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.Linq;
    using System.Security.Cryptography;
    using System.Text;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// 波形アーカイブの依存ファイルをリンクします。
    /// </summary>
    internal class WaveArchiveDependedFileLinker : ComponentSetup<SoundArchiveContext, WaveArchiveBase>
    {
        private static readonly HashAlgorithm hashAlgorithm = new SHA1CryptoServiceProvider();
        private FileManager fileManager;
        private SoundProjectService _soundProjectService;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="soundProjectService">SoundProjectService を指定します。</param>
        /// <param name="fileManager">ファイルマネージャを指定します。</param>
        public WaveArchiveDependedFileLinker(SoundProjectService soundProjectService, FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);
            this.fileManager = fileManager;
            _soundProjectService = soundProjectService;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, WaveArchiveBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            IOutput outputTarget = component.GetOutputTarget();

            // バイナリ波形出力に依存します。
            foreach (IOutput waveOutputTarget in component.GetItemOutputTargets())
            {
                this.fileManager.RegisterDependedOutputItem(outputTarget, waveOutputTarget);
            }

            // バンクに依存します。
            foreach (SoundSetBankBase bank in context.BankDictionary.Keys)
            {
                var bankWaveArchive =
                    bank.GetWaveArchiveDictionary().Values.
                    Where(value => value == component).
                    FirstOrDefault();

                if (bankWaveArchive != null)
                {
                    this.fileManager.RegisterDependedFile(outputTarget, bank.FilePath);
                }
            }

            // 参照されるウェーブサウンドセットや、グループ（自動生成された場合）に依存するため、
            // 波形アーカイブは全てのサウンドセットに対して依存させます。
            // ただし、波形アーカイブのハッシュ値と内包する波形リストのハッシュ値を指定することで、
            // 関係のない修正に影響されないようにします。
            HashCode hashCode = this.ComputeHash(component);

            foreach (string soundSetFilePath in (_soundProjectService.SoundSetDocuments
                                                 .Where(d => context.IsTargetSoundSet(d.SoundSet))
                                                 .Select(d => d.Resource.Key)))
            {
                this.fileManager.RegisterDependedFile(outputTarget, soundSetFilePath, hashCode);
            }
        }

        private HashCode ComputeHash(WaveArchiveBase component)
        {
            StringBuilder hashSourceBuilder = new StringBuilder();

            foreach (IOutput waveOutputTarget in component.GetItemOutputTargets())
            {
                if (waveOutputTarget.ItemDictionary.Values.Count == 0)
                {
                    continue;
                }

                hashSourceBuilder.Append(waveOutputTarget.ItemDictionary.Values.First().Path);
            }

            string outputPaths = hashSourceBuilder.ToString();

            if (outputPaths.Length == 0)
            {
                return HashCode.Empty;
            }

            HashCode componentHashCode = component.GetComponentHashCode(hashAlgorithm, null);

            // WaveArchiveBase のハッシュ値が空の場合は、波形リストのみでハッシュ計算します。
            if (componentHashCode == HashCode.Empty)
            {
                return new HashCode(
                    hashAlgorithm.ComputeHash(Encoding.Unicode.GetBytes(outputPaths)));
            }

            // WaveArchiveBase のハッシュ値と波形リストハッシュ値を連結します。
            byte[] hashSource =
                componentHashCode.Value.Concat(Encoding.Unicode.GetBytes(outputPaths)).
                ToArray();

            return new HashCode(hashAlgorithm.ComputeHash(hashSource));
        }
    }
}
