﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Security.Cryptography;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// ストリームサウンドの依存ファイルをリンクします。
    /// </summary>
    internal class StreamSoundDependedFileLinker : ComponentSetup<SoundArchiveContext, StreamSoundBase>
    {
        private const string CockpitExtension = "cockpit";  // HACK : 本当は Traits から注入すべき内容

        private static readonly HashAlgorithm hashAlgorithm = new SHA1CryptoServiceProvider();

        private SoundProjectService _soundProjectService;
        private FileManager fileManager;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="SoundProjectService">SoundProjectService を指定します。</param>
        /// <param name="fileManager">ファイルマネージャを指定します。</param>
        public StreamSoundDependedFileLinker(SoundProjectService soundProjectService, FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);

            _soundProjectService = soundProjectService;
            this.fileManager = fileManager;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, StreamSoundBase component)
        {
            Assertion.Argument.NotNull(context);

            IOutput outputTarget = component.GetOutputTarget();
            IOutput prefetchOutputTarget = component.GetPrefetchOutputTarget();

            // サウンドセットファイルに依存します。
            string soundSetFilePath = this.GetSoundSetFilePath(component);

            string preprocessedTag = GetPreprocessedTag(context.ProjectFilePath);
            Func<IParameterValue, bool> optionFilter = null;

            if (!string.IsNullOrEmpty(preprocessedTag))
            {
                optionFilter = parameterValue => string.Equals(preprocessedTag, parameterValue.Value);
            }

            var hashCode = component.GetComponentHashCode(hashAlgorithm, optionFilter);

            this.fileManager.RegisterDependedFile(outputTarget, soundSetFilePath, hashCode);

            if (prefetchOutputTarget != null)
            {
                this.fileManager.RegisterDependedFile(prefetchOutputTarget, soundSetFilePath, hashCode);
            }

            // トラックが参照する波形ファイルに依存します。
            HashSet<string> filePaths = new HashSet<string>();

            foreach (StreamSoundTrackBase track in component.Children.Where<Component>(item => item.IsEnabled))
            {
                if (filePaths.Contains(track.FilePath))
                {
                    continue;
                }

                filePaths.Add(track.FilePath);

                this.fileManager.RegisterDependedFile(outputTarget, track.FilePath);

                if (prefetchOutputTarget != null)
                {
                    this.fileManager.RegisterDependedFile(prefetchOutputTarget, track.FilePath);
                }
            }
        }

        /// <summary>
        /// 指定サウンドセットアイテムが属するサウンドセットのエントリーを取得します。
        /// </summary>
        /// <param name="soundSetItem">サウンドセットアイテムを指定します。</param>
        /// <returns>サウンドセットのエントリーを返します。</returns>
        private string GetSoundSetFilePath(SoundSetItem soundSetItem)
        {
            Assertion.Argument.NotNull(soundSetItem);

            SoundSet soundSet = soundSetItem.SoundSet;
            Ensure.Operation.ObjectNotNull(soundSet);

            return _soundProjectService.SoundSetDocuments
                .Where(d => d.SoundSet == soundSet)
                .Select(d => d.Resource.Key)
                .First();
        }

        private string GetPreprocessedTag(string projectFilePath)
        {
            string cockpitFilePath = Path.ChangeExtension(projectFilePath, CockpitExtension);

            if (!File.Exists(cockpitFilePath))
            {
                return string.Empty;
            }

            using (var fileStream = File.Open(cockpitFilePath, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                using (var reader = new StreamReader(fileStream))
                {
                    return reader.ReadLine();
                }
            }
        }
    }
}
