﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.IO;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.SoundFoundation.Projects;
    using ToolDevelopmentKit;

    /// <summary>
    /// シーケンスサウンドの依存ファイルをリンクします。
    /// </summary>
    internal class SequenceSoundDependedFileLinker : ComponentSetup<SoundArchiveContext, SequenceSoundBase>
    {
        private FileManager fileManager;
        private string includePath = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="fileManager">ファイルマネージャを指定します。</param>
        public SequenceSoundDependedFileLinker(FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);
            this.fileManager = fileManager;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, SequenceSoundBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            IOutput outputTarget = component.GetOutputTarget();

            this.fileManager.RegisterDependedFile(
                component.GetOutputTarget(), component.FilePath);

            IConversionProcessor processor;

            switch (component.FileType)
            {
                case SequenceSoundFileType.Smf:
                    processor = context.GetProcessor(outputTarget, SequenceSoundProcessorFactory.OutputID_Intermediate);
                    break;

                case SequenceSoundFileType.Text:
                    processor = context.GetProcessor(outputTarget, string.Empty);
                    break;

                default:
                    throw new Exception("internal error : invalid sequence sound file type.");
            }

            this.includePath = Path.GetDirectoryName(context.ProjectFilePath);

            Ensure.Operation.ObjectNotNull(processor);
            processor.StateChanged += (sender, e) => this.OnProcessorStateChanged(component, sender as IConversionProcessor);
        }

        /// <summary>
        /// ConversionProcessor の状態が変更されると発生します。
        /// </summary>
        /// <param name="sequenceSound">対象となるシーケンスサウンドを指定します。</param>
        /// <param name="processor">対象となる ConversionProcessor を指定します。</param>
        private void OnProcessorStateChanged(SequenceSoundBase sequenceSound, IConversionProcessor processor)
        {
            Ensure.Operation.ObjectNotNull(sequenceSound);
            Ensure.Operation.ObjectNotNull(processor);

            if (processor.State != ConversionProcessState.Done)
            {
                return;
            }

            if (processor.IsSkiped)
            {
                return;
            }

            this.LinkSourceFile(sequenceSound);
            this.LinkTextSequenceDependedFiles(sequenceSound);
        }

        /// <summary>
        /// シーケンスサウンドのソースファイルを依存関係に追加します。
        /// </summary>
        /// <param name="sequenceSound">対象シーケンスサウンドを定します。</param>
        private void LinkSourceFile(SequenceSoundBase sequenceSound)
        {
            Assertion.Operation.ObjectNotNull(this.fileManager);
            Assertion.Operation.ObjectNotNull(sequenceSound);

            // ソースファイルに依存します。
            IOutput outputTarget = sequenceSound.GetOutputTarget();
            this.fileManager.RegisterDependedFile(outputTarget, sequenceSound.FilePath);
        }

        /// <summary>
        /// テキストシーケンスが参照するファイルを依存関係に追加します。
        /// </summary>
        /// <param name="sequenceSound">対象シーケンスサウンドを定します。</param>
        private void LinkTextSequenceDependedFiles(SequenceSoundBase sequenceSound)
        {
            Assertion.Operation.ObjectNotNull(this.fileManager);
            Assertion.Operation.ObjectNotNull(sequenceSound);

            IOutput outputTarget = sequenceSound.GetOutputTarget();
            string textSequenceSoundFilePath;

            switch (sequenceSound.FileType)
            {
                case SequenceSoundFileType.Smf:
                    textSequenceSoundFilePath =
                        outputTarget.ItemDictionary[SequenceSoundProcessorFactory.OutputID_Intermediate].Path;
                    break;

                case SequenceSoundFileType.Text:
                    textSequenceSoundFilePath = sequenceSound.FilePath;
                    break;

                default:
                    throw new Exception("internal error : invalid sequence sound file type.");
            }

            // テキストシーケンスファイルが参照する全てのファイルに依存します。
            SequenceSoundTextLexer lexer = new SequenceSoundTextLexer(textSequenceSoundFilePath);

            if (!string.IsNullOrEmpty(this.includePath))
            {
                lexer.AppendSearchPath(this.includePath);
            }

            while (lexer.ReadToken() != SequenceSoundTextLexer.Token.End)
            {
                // 依存ファイル収集のため空回し
            }

            foreach (string dependentFilePath in lexer.DependFiles)
            {
                if (dependentFilePath == textSequenceSoundFilePath)
                {
                    continue;
                }

                this.fileManager.RegisterDependedFile(outputTarget, dependentFilePath);
            }
        }
    }
}
