﻿namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.IO;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using ToolDevelopmentKit;

    internal class WaveSound2Processor : ComponentProcessor<SoundArchiveContext, WaveSound>
    {
        private WaveArchiveBase waveArchive;

        public WaveSound2Processor(WaveSound component, WaveArchiveBase waveArchive, IOutputItem outputItem)
            : base(component, outputItem)
        {
            Ensure.Argument.NotNull(waveArchive);
            this.waveArchive = waveArchive;
        }

        protected override void ProcessInternal(SoundArchiveContext context)
        {
            try
            {
                using (Stream stream = this.OutputTargetItem.OpenWrite())
                {
                    Write(context, context.CreateBinaryWriter(stream));
                }
            }
            catch
            {
                try
                {
                    File.Delete(this.OutputTargetItem.Path);
                }
                catch
                {
                }

                throw;
            }
        }

        protected override void PostProcessInternal(SoundArchiveContext context)
        {
            this.SetBinaryPaths(context);
        }

        protected override void OutputLog(SoundArchiveContext context, Component[] components)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(components);

            context.Logger.AddLine(new InformationLine(
                string.Format("[WSD] {0} ({1}) > {2}",
                components[0].Name,
                this.waveArchive.Name,
                Path.GetFileName(this.OutputTargetItem.Path)
                )));
        }

        private void Write(SoundArchiveContext context, BinaryWriter writer)
        {
            Assertion.Argument.NotNull(writer);

            var file = new WaveSound2FileBuilder(
                context.Traits.BinaryFileInfo.WaveSound2Signature,
                context.Traits.BinaryFileInfo.WaveSound2Version).
                Build(this.TargetComponent as WaveSound, this.waveArchive);

            DomElement fileElement = new DomBuilder().Build(file);
            new DomWriter(writer).Run(new DomObjectWriter(), fileElement);
        }

        private void SetBinaryPaths(SoundArchiveContext context)
        {
            if (!context.Settings.IsConvertParts)
            {
                return;
            }

            // TODO : WaveSoundResoruce に対応する

            var waveSoundSet = this.TargetComponent as WaveSoundSetBase;
            var waveArchive = waveSoundSet.GetWaveArchiveDictionary()[string.Empty];

            // 部分コンバートでは、必ず 1WSD : 1WSDSET
            Ensure.Operation.True(waveSoundSet.Children.Count == 1);

            var waveSound = (waveSoundSet.Children[0] as SoundSetItem).GetOriginalModel() as WaveSoundBase;
            Ensure.Operation.ObjectNotNull(waveSound);

            // bxwsd ファイルパスを設定します。
            waveSound.SetBinaryFilePathForPartsConvert(this.OutputTargetItem.Path);

            // bxwar ファイルパスを設定します。
            waveSound.SetWaveArchiveBinaryFilePathForPartsConvert(
                waveArchive.GetOutputTarget().ItemDictionary[string.Empty].Path);
        }
    }
}
