﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------


namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.BankElements;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using ToolDevelopmentKit;

    /// <summary>
    /// バンクバイナリを作成します。
    /// </summary>
    internal class BankProcessor : ConversionProcessor<ComponentContext>, IComponentProcessor
    {
        private static object writeIncludeLock = new object();
        private HashSet<Component> components = new HashSet<Component>();
        private HashSet<Component> relatedComponents = new HashSet<Component>();
        private WaveArchiveBase waveArchive;
        private string sourceName;
        private IOutputItem outputItem;
        private IOutputItem includeOutputItem;

        public BankProcessor(
            Bank component,
            WaveArchiveBase waveArchive,
            string sourceName,
            IOutputItem outputItem,
            IOutputItem includeOutputItem)
        {
            Ensure.Argument.NotNull(component);
            Ensure.Argument.NotNull(waveArchive);
            Ensure.Argument.NotNull(outputItem);

            this.components.Add(component);
            this.waveArchive = waveArchive;
            this.sourceName = string.IsNullOrEmpty(sourceName) ? string.Empty : sourceName;
            this.outputItem = outputItem;
            this.includeOutputItem = includeOutputItem;
        }

        /// <summary>
        /// 依存するコンポーネントのコレクションを取得します。
        /// </summary>
        public ICollection<Component> Components
        {
            get { return this.components; }
        }

        /// <summary>
        /// 関連するコンポーネントのコレクションを取得します。
        /// </summary>
        public ICollection<Component> RelatedComponents
        {
            get { return this.relatedComponents; }
        }

        public IOutputItem OutputTargetItem
        {
            get { return this.outputItem; }
        }

        /// <summary>
        /// 対象のサウンドセットバンクを取得または設定します。
        /// </summary>
        public SoundSetBankBase SoundSetBank { get; set; }

        private Bank TargetComponent
        {
            get { return this.components.First() as Bank; }
        }

        protected override bool PreRunInternal(ComponentContext context)
        {
            return this.OutputTargetItem.IsDirty;
        }

        protected override bool RunInternal(ComponentContext context)
        {
            Ensure.Argument.NotNull(context);

            context.Logger.AddLine(
                new InformationLine(
                    string.Format("[BANK] {0} > {1}",
                    this.sourceName,
                    Path.GetFileName(this.outputItem.Path)
                    ))
                );

            try
            {
                using (Stream stream = this.OutputTargetItem.OpenWrite())
                {
                    Write(context, context.CreateBinaryWriter(stream));
                }
            }
            catch (ConversionException exception)
            {
                throw exception;
            }
            catch (IOException exception)
            {
                context.Logger.AddLine(
                    new ErrorLine(exception.Message, this.Components.ToArray())
                    );
                return false;
            }

            return true;
        }

        protected override void PostRunInternal(ComponentContext context)
        {
            base.PostRunInternal(context);
            this.SetBinaryPaths();
        }

        private void Write(ComponentContext context, BinaryWriter writer)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(writer);

            BankBinary binary = new BankFileBuilder(
                context.Traits.BinaryFileInfo.BankSignature,
                context.Traits.BinaryFileInfo.BankVersion).
                Build(this.TargetComponent, this.waveArchive);

            DomElement rootElement = new DomBuilder().Build(binary);
            new DomWriter(writer).Run(new DomObjectWriter(), rootElement);

            if (this.includeOutputItem == null)
            {
                return;
            }

            // 複数の BankProcessor インスタンスから同一の Include ファイルを出力しようとする場合があるため、
            // ロックした上で、IsDirty を再チェックします。
            lock (writeIncludeLock)
            {
                if (!this.OutputTargetItem.IsDirty)
                {
                    return;
                }

                using (Stream streamInclude = this.includeOutputItem.OpenWrite())
                {
                    WriteInclude(
                        context.CreateBinaryWriter(streamInclude),
                        binary.InfoBlock.Body.InstrumentSet.Items);
                }
            }
        }

        private void WriteInclude(BinaryWriter writer, IEnumerable<InstrumentInfo> instrumentInfos)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(instrumentInfos);

            foreach (InstrumentInfo instrumentInfo in instrumentInfos)
            {
                if (!instrumentInfo.IsEnabled ||
                    string.IsNullOrEmpty(instrumentInfo.Name))
                {
                    continue;
                }

                string text = string.Format(
                    "#define {0}\t{1}\n",
                    instrumentInfo.Name,
                    instrumentInfo.ProgramNo);

                writer.Write(text.ToCharArray());
            }
        }

        private void SetBinaryPaths()
        {
            if (this.SoundSetBank == null)
            {
                return;
            }

            // bxbnk, bxwar ファイルパスを設定します。
            this.SoundSetBank.SetBinaryFilePathForPartsConvert(
                this.SoundSetBank.GetOutputTarget().ItemDictionary[string.Empty].Path);

            this.SoundSetBank.SetWaveArchiveBinaryFilePathForPartsConvert(
                waveArchive.GetOutputTarget().ItemDictionary[string.Empty].Path);
        }
    }
}
