﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.SoundFoundation.Binarization;
using NintendoWare.SoundFoundation.Codecs;
using NintendoWare.SoundFoundation.Core.IO;
using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
using NintendoWare.SoundFoundation.FileFormats.Wave;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundFoundation.Resources;
using NintendoWare.SoundFoundation.Utilities;
using NintendoWare.ToolDevelopmentKit;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    public class StreamConverter
    {
        private const string TargetPrefix = "[NW]";
        private const char RegionIndexSplitterChar = '#';
        private const int MaxRegionCount = 128;
        /// <summary> リージョン名の最大長(終端文字を含まない）</summary>
        private const int MaxRegionNameLength = FileFormats.NintendoWareBinary.StreamSoundElements.RegionInfo.MaxName - 1;
        private const int MaxStreamSoundTrackCount = 8;

        private const string BfstmSignature = "FSTM";
        private const string BfstpSignature = "FSTP";

        // ★TODO : .bfstm/.bfstpバージョンが上がったら、必ずココを更新する！
        private readonly BinaryVersion BfstmVersion = new BinaryVersion(0, 6, 3, 0);
        private readonly BinaryVersion BfstpVersion = new BinaryVersion(0, 5, 2, 0);

        private List<string> warningMessages = new List<string>();

        private CodecOutput codecOutput;
        private uint hashValue;
        private WaveStream[] waveStreams;
        private IList<WaveRegionInfo> regions;
        private IList<WaveMarkerInfo> markers;

        //-----------------------------------------------------------------

        public void Convert(string inputFilePath, WaveEncoding encoding, int? loopStartFrame, int? loopEndFrame)
        {
            IEnumerable<string> trackFilePaths = IsWaveFile(inputFilePath)
                ? new string[] { inputFilePath }
                : this.EnumerateTrackFilePaths(inputFilePath);

            if (!trackFilePaths.Any())
            {
                throw new Exception(Resources.MessageResourceCommon.Message_WaveFileListHasNoWavFiles);
            }

            this.warningMessages.Clear();
            this.Reset();

            IList<WaveMarkerInfo> markers = null;
            IList<WaveRegionInfo> regions;
            WaveStream[] waveStreams = this.ReadTrackFiles(trackFilePaths, encoding, loopStartFrame, loopEndFrame, out markers, out regions);

            try
            {
                this.ValidateTrackFiles(waveStreams);

                // 今のところ LittleEndian 固定
                NintendoWareStreamEncoder encoder = this.CreateEncoder(true, encoding);

                CodecOutput[] codecOutputs = encoder.Run(waveStreams);

                // ループなしの場合は、全範囲をループ範囲とします。
                if (!codecOutputs[0].Format.HasLoop)
                {
                    codecOutputs[0].Format = new WaveFormat(codecOutputs[0].Format)
                    {
                        OriginalLoopStartFrame = 0,
                        OriginalLoopEndFrame = 0,
                        LoopStartFrame = 0,
                        LoopEndFrame =
                            ((int)waveStreams[0].Payload.Length / waveStreams[0].Format.FrameLength) - 1,
                    };
                }

                this.codecOutput = codecOutputs[0];
                this.waveStreams = waveStreams;
                this.markers = markers;
                this.regions = regions;
            }
            catch
            {
                foreach (var waveStream in waveStreams)
                {
                    waveStream.Dispose();
                }

                throw;
            }

            foreach (string message in this.warningMessages)
            {
                Console.WriteLine("[warning] " + message);
            }
            this.warningMessages.Clear();
        }

        public void Reset()
        {
            if (this.waveStreams != null)
            {
                foreach (var waveStream in this.waveStreams)
                {
                    waveStream.Dispose();
                }
            }

            this.codecOutput = null;
            this.waveStreams = null;
            this.markers = null;
            this.regions = null;
        }

        public WaveStreamData GetOutputWaveStreamData()
        {
            Ensure.Operation.ObjectNotNull(this.codecOutput);
            return this.codecOutput.StreamData;
        }

        public WaveFormat GetOutputWaveFormat()
        {
            Ensure.Operation.ObjectNotNull(this.codecOutput);
            return this.codecOutput.Format;
        }

        public int GetOutputWaveDataLength()
        {
            Ensure.Operation.ObjectNotNull(this.codecOutput);
            return this.codecOutput.Data.Length;
        }

        public void WriteBinary(string filePath)
        {
            Assertion.Argument.StringNotEmpty(filePath);

            Ensure.Operation.ObjectNotNull(this.codecOutput);
            Ensure.Operation.ObjectNotNull(this.waveStreams);
            Ensure.Operation.ObjectNotNull(this.regions);

            ulong bfstmHashValueOffset;

            using (var fileStream = File.Create(filePath))
            {
                using (var writer = LittleEndianBinaryWriter.Create(fileStream))
                {
                    var builder = new StreamSoundFileBuilder(BfstmSignature, BfstmVersion);
                    var file = builder.Build(this.codecOutput, 0, this.markers, this.regions);

                    DomElement fileElement = new DomBuilder().Build(file);
                    var domWriter = new DomWriter(writer);
                    domWriter.Run(new DomObjectWriter(), fileElement);

                    //  bfstm 中のハッシュ値へのオフセットを取得する
                    bfstmHashValueOffset = domWriter.Context.ReferenceResolver.GetObjectReference(file.InfoBlock.Body.StreamSoundInformation.HashInfo).Address;
                }
            }

            // bfstm と bfstp リビジョン不一致検出を可能にするために、bfstm, bfstp に bfstm の CRC32 コードを埋め込みます。
            // ゲーム開発フロー中の事故などにより、bfstm と bfstp のリビジョンが食い違ってしまった場合に、それを検知できるようにします。
            this.hashValue = this.ComputeHash(File.ReadAllBytes(filePath));

            using (var fileStream = File.OpenWrite(filePath))
            {
                using (var writer = LittleEndianBinaryWriter.Create(fileStream))
                {
                    writer.Seek((int)bfstmHashValueOffset, SeekOrigin.Begin);
                    writer.Write(this.hashValue);
                }
            }
        }

        public void WritePrefetchBinary(string filePath, int streamBlockCount)
        {
            Assertion.Argument.StringNotEmpty(filePath);

            Ensure.Operation.ObjectNotNull(this.codecOutput);
            Ensure.Operation.ObjectNotNull(this.waveStreams);
            Ensure.Operation.ObjectNotNull(this.regions);

            using (var fileStream = File.Create(filePath))
            {
                using (var writer = LittleEndianBinaryWriter.Create(fileStream))
                {
                    var prefetchDataLength = (uint)this.GetPrefetchDataLength(streamBlockCount, this.codecOutput);

                    var builder = new StreamSoundPrefetchFileBuilder(BfstpSignature, BfstpVersion);
                    var file = builder.Build(
                        this.codecOutput,
                        this.hashValue,
                        this.regions,
                        prefetchDataLength);

                    DomElement fileElement = new DomBuilder().Build(file);
                    new DomWriter(writer).Run(new DomObjectWriter(), fileElement);
                }
            }
        }

        private uint ComputeHash(byte[] data)
        {
            var crc32 = new CRC32();
            crc32.Initialize();
            return crc32.ComputeHashUInt32(data);
        }

        private int GetPrefetchDataLength(int streamBlockCount, CodecOutput codecOutput)
        {
            return Math.Min(
                streamBlockCount * codecOutput.StreamData.BlockByte * codecOutput.Format.ChannelCount,
                codecOutput.Data.Length);
        }

        private bool IsWaveFile(string filePath)
        {
            try
            {
                using (var fileReader = WaveFileReader.CreateInstance(filePath)) { }
                return true;
            }
            catch
            {
                return false;
            }
        }

        private void OnLoopEndFrameModified(object send, LoopEndFrameModifiedEventArgs e)
        {
            string format = string.Format(
                SoundFoundation.Resources.MessageResource.Message_LoopEndFrameModified,
                e.LoopEndFrame,
                e.FrameCount - 1,
                e.FilePath);

            this.warningMessages.Add(format);
        }

        private NintendoWareStreamEncoder CreateEncoder(bool isLittleEndian, WaveEncoding encoding)
        {
            switch (encoding)
            {
                case WaveEncoding.Adpcm:
                    return new NintendoWareDspAdpcmStreamEncoder();

                case WaveEncoding.Pcm16:
                    return new NintendoWareLinearPcmStreamEncoder()
                    {
                        BitsPerSample = 16,
                        IsLittleEndian = isLittleEndian,
                    };

                case WaveEncoding.Pcm8:
                    return new NintendoWareLinearPcmStreamEncoder()
                    {
                        BitsPerSample = 8,
                    };
            }

            throw new Exception("internal error : invalid stream encoding.");
        }

        private WaveStream[] ReadTrackFiles(
            IEnumerable<string> trackFilePaths,
            WaveEncoding encoding,
            int? loopStartFrame,
            int? loopEndFrame,
            out IList<WaveMarkerInfo> markers,
            out IList<WaveRegionInfo> regions)
        {
            if (loopStartFrame.HasValue || loopEndFrame.HasValue)
            {
                if (!loopStartFrame.HasValue)
                {
                    throw new Exception("internal error : loopStartFrame is invalid.");
                }
            }

            var result = new List<WaveStream>();

            markers = null;
            regions = null;
            bool isFirstTrack = true;

            foreach (string trackFilePath in trackFilePaths)
            {
                using (var fileReader = WaveFileReader.CreateInstance(trackFilePath))
                {
                    WaveFile waveFile = fileReader.Open(trackFilePath);

                    fileReader.LoopEndFrameModified += OnLoopEndFrameModified;

                    bool hasLoop = waveFile.IsLoop;
                    int validLoopStartFrame = (int)waveFile.LoopStartFrame;
                    int validLoopEndFrame = (int)waveFile.LoopEndFrame;

                    if (isFirstTrack)
                    {
                        regions = this.ExtractTrackWaveRegions(waveFile, trackFilePath, encoding);

                        if (loopStartFrame < 0 || waveFile.FrameCount <= loopStartFrame)
                        {
                            throw new Exception(string.Format(
                                Resources.MessageResourceCommon.Message_LoopStartIsOutOfRange,
                                loopStartFrame.Value,
                                waveFile.FrameCount - 1));
                        }

                        if (loopEndFrame < 0 || waveFile.FrameCount <= loopEndFrame)
                        {
                            throw new Exception(string.Format(
                                Resources.MessageResourceCommon.Message_LoopEndIsOutOfRange,
                                loopEndFrame.Value,
                                loopStartFrame.Value,
                                waveFile.FrameCount - 1));
                        }

                        // ループが指定されている場合は、最優先で適用します。
                        if (loopStartFrame.HasValue)
                        {
                            hasLoop = true;
                            validLoopStartFrame = loopStartFrame.Value;
                            validLoopEndFrame = loopEndFrame.HasValue ? loopEndFrame.Value : (int)waveFile.FrameCount - 1;
                        }

                        markers = this.ExtractTrackWaveMarkers(waveFile);
                    }

                    bool isLoop =
                        hasLoop &&
                        validLoopStartFrame != validLoopEndFrame;

                    result.Add(new WaveStream(
                        new WaveFormat()
                        {
                            BitsPerSample = waveFile.SampleBit,
                            ChannelCount = waveFile.ChannelCount,
                            Encoding = Codecs.Encoding.Pcm,
                            HasLoop = isLoop,
                            IsLittleEndian = waveFile is WaveFileWav,
                            IsSigned = !((waveFile is WaveFileWav) && waveFile.SampleBit == 8),
                            OriginalLoopStartFrame = validLoopStartFrame,
                            OriginalLoopEndFrame = validLoopEndFrame,
                            LoopStartFrame = validLoopStartFrame,
                            LoopEndFrame = validLoopEndFrame,
                            SamplingRate = waveFile.SampleRate,
                            Regions = regions,
                        },
                        fileReader.OpenDataStream()
                        ));

                    isFirstTrack = false;
                }
            }

            return result.ToArray();
        }

        private IEnumerable<string> EnumerateTrackFilePaths(string waveListFilePath)
        {
            this.ValidateFilePath(waveListFilePath, Resources.MessageResourceCommon.Label_WaveListFile);

            var dirPath = Path.GetDirectoryName(PathUtility.GetFullPath(waveListFilePath));

            using (var fileStream = File.OpenRead(waveListFilePath))
            {
                using (var fileReader = new StreamReader(fileStream))
                {
                    while (true)
                    {
                        var trackFilePath = fileReader.ReadLine();

                        if (trackFilePath == null)
                        {
                            break;
                        }

                        if (trackFilePath.Length == 0)
                        {
                            continue;
                        }

                        var trackFileFullPath = PathUtility.GetFullPath(Path.Combine(dirPath, trackFilePath));
                        this.ValidateFilePath(trackFileFullPath, Resources.MessageResourceCommon.Label_WaveFile);

                        yield return trackFileFullPath;
                    }
                }
            }

            yield break;
        }

        private IList<WaveMarkerInfo> ExtractTrackWaveMarkers(WaveFile waveFile)
        {
            var waveMarkers = waveFile.Markers.
                Where(marker => this.IsTargetMarker(marker)).ToArray();

            if (waveMarkers.Length == 0)
            {
                return null;
            }

            var result = new List<WaveMarkerInfo>();

            foreach (var marker in waveMarkers)
            {
                var markerName = this.GetMarkerName(marker);
                var newMarker = new WaveMarkerInfo(markerName, marker.Position);

                result.Add(newMarker);
            }

            return result;
        }

        private bool IsTargetMarker(WaveFile.IMarkerInfo marker)
        {
            return marker.Name.StartsWith(TargetPrefix);
        }

        private string GetMarkerName(WaveFile.IMarkerInfo marker)
        {
            Assertion.Argument.True(this.IsTargetMarker(marker));

            var markerName = marker.Name;
            return markerName.Substring(TargetPrefix.Length);
        }

        private IList<WaveRegionInfo> ExtractTrackWaveRegions(
            WaveFile waveFile,
            string filePath,
            WaveEncoding encoding)
        {
            var result = new List<WaveRegionInfo>();
            var waveRegions = waveFile.Regions.
                Where(region => this.IsTargetRegion(region)).ToArray();

            if (waveRegions.Length == 0)
            {
                return result;
            }

            if (waveRegions.Length >= MaxRegionCount)
            {
                throw new Exception(
                    string.Format(
                    SoundFoundation.Resources.MessageResource.Message_WaveRegionsTooMany,
                    filePath
                    ));
            }

            var autoIndexedRegions = new List<WaveRegionInfo>();
            uint minRegionFrames = this.GetMinimumWaveRegionFrames(encoding);

            foreach (var region in waveRegions)
            {
                this.ValidateRegion(filePath, region, encoding, minRegionFrames);

                int regionIndex = -1;

                try
                {
                    regionIndex = this.GetRegionIndex(region);
                }
                catch
                {
                    throw new Exception(
                        string.Format(
                        SoundFoundation.Resources.MessageResource.Message_InvalidWaveRegionIndex,
                        region.Name,
                        filePath
                        ));
                }

                var regionName = this.ValidateRegionName(filePath, this.GetRegionName(region));
                var newRegion = new WaveRegionInfo(regionName, region.StartFrame, region.EndFrame);

                // 自動インデックス指定の場合
                if (regionIndex < 0)
                {
                    // 後で空インデックスをみつけて追加します。
                    autoIndexedRegions.Add(newRegion);
                    continue;
                }

                // 以下、手動インデックス指定の場合
                // インデックスの上限チェック
                if (MaxRegionCount <= regionIndex)
                {
                    throw new Exception(
                        string.Format(
                        SoundFoundation.Resources.MessageResource.Message_WaveRegionIndexTooLarge,
                        region.Name,
                        regionIndex,
                        MaxRegionCount,
                        filePath));
                }

                // インデックスにあわせて配列（リスト）を拡張
                while (regionIndex >= result.Count)
                {
                    result.Add(null);
                }

                // インデックスの重複チェック
                if (result[regionIndex] != null)
                {
                    throw new Exception(
                        string.Format(
                        SoundFoundation.Resources.MessageResource.Message_WaveRegionIndexAlreadyExisted,
                        region.Name,
                        regionIndex,
                        filePath));
                }

                result[regionIndex] = newRegion;
            }

            // 自動インデックス指定リージョンを追加します。
            // 自動インデックス指定リージョンがない場合は、これで終了
            if (autoIndexedRegions.Count == 0)
            {
                return result;
            }

            int currentIndex = 0;

            foreach (var region in autoIndexedRegions)
            {
                for (; currentIndex < result.Count; ++currentIndex)
                {
                    if (result[currentIndex] == null)
                    {
                        break;
                    }
                }

                if (currentIndex + 1 >= result.Count)
                {
                    result.Add(region);
                }
                else
                {
                    Assertion.Operation.ObjectNull(result[currentIndex]);
                    result[currentIndex] = region;
                }
            }

            return result;
        }

        private bool IsTargetRegion(WaveFile.IRegionInfo region)
        {
            return region.Name.StartsWith(TargetPrefix);
        }

        private int GetRegionIndex(WaveFile.IRegionInfo region)
        {
            Assertion.Argument.True(this.IsTargetRegion(region));

            var regionIndexSplitter = region.Name.IndexOf(RegionIndexSplitterChar);
            if (regionIndexSplitter < 0)
            {
                return -1;
            }

            return int.Parse(region.Name.Substring(regionIndexSplitter + 1));
        }

        private string GetRegionName(WaveFile.IRegionInfo region)
        {
            Assertion.Argument.True(this.IsTargetRegion(region));

            var regionName = region.Name;
            var regionIndexSplitter = regionName.IndexOf(RegionIndexSplitterChar);

            if (regionIndexSplitter >= 0)
            {
                regionName = regionName.Substring(0, regionIndexSplitter);
            }

            return regionName.Substring(TargetPrefix.Length);
        }

        private string ValidateRegionName(string filePath, string name)
        {
            if (string.IsNullOrEmpty(name))
            {
                return name;
            }

            string result = name;

            // 文字列長チェック (終端文字を含まない）
            // 長い場合は後ろを切り取る
            if (name.Length > MaxRegionNameLength)
            {
                result = name.Substring(0, MaxRegionNameLength);

                this.warningMessages.Add(
                    string.Format(
                        MessageResource.Message_WaveRegionNameTooLong,
                        MaxRegionNameLength,
                        name,
                        result,
                        filePath));
            }

            // 先頭文字が英字か '_' でなければ、 '_' に置換
            // それ以降の文字が英数字か '_' でなければ、 '_' に置換
            var source = result;
            result = Regex.Replace(result, @"^[^a-zA-Z_]", "_");
            result = Regex.Replace(result, @"[^a-zA-Z0-9]", "_");

            if (result != source)
            {
                this.warningMessages.Add(
                    string.Format(
                        MessageResource.Message_WaveRegionNameReplaceInvalidChars,
                        source,
                        result,
                        filePath));
            }

            return result;
        }

        private void ValidateRegion(
            string filePath,
            WaveFile.IRegionInfo region,
            WaveEncoding waveEncoding,
            uint minRegionFrames)
        {
            if (region.EndFrame <= region.StartFrame)
            {
                throw new Exception(
                    string.Format(
                    SoundFoundation.Resources.MessageResource.Message_InvalidWaveRegionFrames,
                    region.Name,
                    filePath));
            }

            uint regionFrames = region.EndFrame - region.StartFrame;

            // エンコード後のリージョンサイズが 16KB 以上になるように制限します。
            if (regionFrames < minRegionFrames)
            {
                throw new Exception(
                    string.Format(
                    SoundFoundation.Resources.MessageResource.Message_WaveRegionFramesTooShort,
                    region.Name,
                    waveEncoding.ToText(),
                    minRegionFrames,
                    filePath));
            }
        }

        private uint GetMinimumWaveRegionFrames(WaveEncoding encoding)
        {
            // エンコード後のリージョンサイズが 16KB 以上になるフレーム数を返します。
            switch (encoding)
            {
                case WaveEncoding.Pcm16:
                    return 8192;

                case WaveEncoding.Pcm8:
                    return 8192 * 2;

                case WaveEncoding.Adpcm:
                    return 14336 * 2;
            }

            throw new NotImplementedException();
        }

        private void ValidateFilePath(string value, string fileType)
        {
            Ensure.Argument.NotNull(value);

            if (string.IsNullOrEmpty(value))
            {
                throw new Exception(
                    string.Format(Resources.MessageResourceCommon.Message_SetXFilePath, fileType)
                );
            }

            if (!File.Exists(value))
            {
                throw new Exception(
                    string.Format(Resources.MessageResourceCommon.Message_XFilePathNotFound, fileType, value)
                );
            }
        }

        private void ValidateTrackFiles(WaveStream[] waveStreams)
        {
            Assertion.Argument.NotNull(waveStreams);

            if (waveStreams.Length == 0)
            {
                throw new Exception(SoundFoundation.Resources.MessageResource.Message_StreamSoundTrackNotFound);
            }

            if (MaxStreamSoundTrackCount < waveStreams.Length)
            {
                string errorMessage0 =
                    string.Format(
                        SoundFoundation.Resources.MessageResource.Message_StreamSoundTrackExceededMaximumNumber,
                        MaxStreamSoundTrackCount + 1);
                string errorMessage1 =
                    string.Format(
                        SoundFoundation.Resources.MessageResource.Message_StreamSoundTrackMaximumNumber,
                        MaxStreamSoundTrackCount);

                throw new Exception(errorMessage0 + "\r\n" + errorMessage1);
            }

            int samplingRate = waveStreams[0].Format.SamplingRate;

            foreach (WaveStream waveStream in waveStreams)
            {
                if (samplingRate != waveStream.Format.SamplingRate)
                {
                    throw new Exception(SoundFoundation.Resources.MessageResource.Message_StreamSoundTracksMustHaveTheSameSamplingRate);
                }
            }
        }
    }
}
