﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using Nintendo.Foundation.Contracts;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// Sox を実行します。
    /// </summary>
    public class SoxExecutor : ConsoleBasedConverter<ConversionContext>
    {
        public const string NoDitherOption = "-D";
        public const string RepeatableOption = "-R";

        public const string ResampleQualityQuickOption = "-q";
        public const string ResampleQualityLowOption = "-l";
        public const string ResampleQualityMediaumOption = "-m";
        public const string ResampleQualityHighOption = "-h";
        public const string ResampleQualityVeryHighOption = "-v";

        private readonly List<string> errorMessages = new List<string>();

        /// <summary>
        /// リサンプルの精度
        /// </summary>
        public enum ResampleQualityType
        {
            Quick,
            Low,
            Mediaum,
            High,
            VeryHigh
        }

        public enum DitherType
        {
            /// <summary>
            /// ディザリングを行わない
            /// </summary>
            NoDither,

            /// <summary>
            /// ディザリングはするが擬似乱数は用いない(出力結果が常に同じになる)
            /// </summary>
            Repeatable,

            /// <summary>
            /// 擬似乱数を元にしたディザリングをする
            /// </summary>
            Dither,
        }

        public SoxExecutor(string converterExePath)
        {
            Ensure.Argument.NotNull(converterExePath);
            Ensure.Argument.StringIsNotNullOrEmpty(converterExePath);

            this.SoxExePath = converterExePath;
            this.Timeout = TimeSpan.FromMinutes(3);
        }

        public string SoxExePath { get; private set; }

        public string InputPath { get; set; }

        public string OutputPath { get; set; }

        public int SampleRate { get; set; }

        public int ChannelCount { get; set; }

        public ResampleQualityType ResampleQuality { get; set; } = ResampleQualityType.VeryHigh;

        public DitherType Dither { get; set; } = DitherType.Repeatable;

        public Component[] TargetComponents { get; set; }

        public static bool Run(string soxExePath, string inputPath, string outputPath, int sampleRate, int channelCount)
        {
            var executor = new SoxExecutor(soxExePath)
            {
                InputPath = inputPath,
                OutputPath = outputPath,
                SampleRate = sampleRate,
                ChannelCount = channelCount,
            };

            return executor.Run(null);
        }

        protected override void InitializeProcess(ConversionContext context, System.Diagnostics.Process process)
        {
            base.InitializeProcess(context, process);

            Directory.CreateDirectory(Path.GetDirectoryName(this.OutputPath));
        }

        protected override void PostRun(ConversionContext context, Process process)
        {
            if (this.errorMessages.Count == 0)
            {
                return;
            }

            // sox は、警告もエラー出力として返すので、プロセスの戻り値を見て、WarningLine or ErrorLine を切り替える
            if (process.ExitCode == 0)
            {
                this.errorMessages
                    .Select(message => new WarningLine(message, this.TargetComponents))
                    .ForEach(line => context?.Logger?.AddLine(line));
            }
            else
            {
                this.errorMessages
                    .Select(message => new ErrorLine(message, this.TargetComponents))
                    .ForEach(line => context?.Logger?.AddLine(line));
            }
        }

        protected override string GetFileName(ConversionContext context)
        {
            return this.SoxExePath;
        }

        protected override string GetArguments(ConversionContext context)
        {
            var builder = new StringBuilder();

            // グローバルオプション
            builder.Append($"{this.GetDitherOption()} ");

            // 入力ファイルオプション
            // (なし)

            // 入力ファイル
            builder.Append($"\"{this.InputPath}\" ");

            // 出力フォーマットオプション
            // (なし)

            // 出力ファイル
            builder.Append($"\"{this.OutputPath}\" ");

            // エフェクト
            if (this.SampleRate > 0)
            {
                builder.Append($"rate {this.GetResmapleQualityOption()} {this.SampleRate} ");
            }

            // チャンネル数
            if (ChannelCount > 0)
            {
                builder.Append($"channels {ChannelCount} ");
            }

            return builder.ToString();
        }

        protected override InformationLine CreateInformationLine(string message)
        {
            return new InformationLine(message, this.TargetComponents);
        }

        protected override ErrorLine CreateErrorLine(string message)
        {
            // sox は、警告もエラー出力として返すので、プロセスの戻り値を見て、WarningLine or ErrorLine を切り替える
            this.errorMessages.Add(message);
            return null;
        }

        private string GetDitherOption()
        {
            switch (this.Dither)
            {
                case DitherType.NoDither:
                    return NoDitherOption;

                case DitherType.Repeatable:
                    return RepeatableOption;

                default:
                    return string.Empty;
            }
        }

        private string GetResmapleQualityOption()
        {
            switch (this.ResampleQuality)
            {
                case ResampleQualityType.Quick:
                    return ResampleQualityQuickOption;

                case ResampleQualityType.Low:
                    return ResampleQualityLowOption;

                case ResampleQualityType.Mediaum:
                    return ResampleQualityMediaumOption;

                case ResampleQualityType.High:
                    return ResampleQualityHighOption;

                default:
                case ResampleQualityType.VeryHigh:
                    return ResampleQualityVeryHighOption;
            }
        }
    }
}
