﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.Collections.Generic;
    using System.Runtime.InteropServices;
    using NintendoWare.SoundFoundation.Core.Collections;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.Collections;

    /// <summary>
    /// 複数のチャンネルを混合して１本のストリームデータを出力するフィルタです。
    /// </summary>
    internal class StreamChannelMixer : Filter
    {
        private PacketStoreList packetStores = new PacketStoreList();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="inputs">入力接続ポイント数を指定します。</param>
        public StreamChannelMixer(int inputCount)
        {
            Ensure.Argument.True(inputCount > 0);

            InputConnectionPoint[] inputs = new InputConnectionPoint[inputCount];
            for (int i = 0; i < inputCount; i++)
            {
                inputs[i] = new InputConnectionPoint();
            }

            this.SetInputs(inputs);

            this.BlockLength = 8 * 1024; // 8KB
            this.AdpcmSeekInformations = new byte[0];
        }

        /// <summary>
        /// 出力データのブロックの長さを取得または設定します。
        /// </summary>
        public int BlockLength { get; set; }

        /// <summary>
        /// ストリーム コーデック情報を取得します。
        /// </summary>
        public WaveStreamData WaveStreamData { get; private set; }

        /// <summary>
        /// ADPCMシーク情報を取得します。
        /// </summary>
        public byte[] AdpcmSeekInformations { get; private set; }

        /// <summary>
        /// 入力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation InputStreamInformation
        {
            get { return this.Inputs[0].StreamInformation; }
        }

        /// <summary>
        /// 入力フォーマットを取得します。
        /// </summary>
        private WaveFormat InputFormat
        {
            get { return this.Inputs[0].StreamInformation.Format; }
        }

        /// <summary>
        /// 出力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation OutputStreamInformation
        {
            get { return this.Outputs[0].StreamInformation; }
        }

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        private WaveFormat OutputFormat
        {
            get { return this.Outputs[0].StreamInformation.Format; }
        }

        /// <summary>
        /// 入力接続ポイントを検証します。
        /// </summary>
        /// <param name="inputs">入力接続ポイントの配列を指定します。</param>
        protected override void ValidateInputs(InputConnectionPoint[] inputs)
        {
            Assertion.Argument.NotNull(inputs);
            Ensure.Operation.True(inputs.Length > 0);

            int length = inputs[0].StreamInformation.Length;
            WaveFormat format = inputs[0].StreamInformation.Format;

            if (1 != format.ChannelCount)
            {
                throw new Exception("channel mixer inputs must be monoral.");
            }

            foreach (InputConnectionPoint input in inputs)
            {
                if (length != input.StreamInformation.Length)
                {
                    throw new Exception("channel mixer inputs must have the same length.");
                }
            }

            foreach (InputConnectionPoint input in inputs)
            {
                if (!this.WaveFormatEquals(format, input.StreamInformation.Format))
                {
                    throw new Exception("channel mixer inputs must have the same format.");
                }
            }

            // 1ブロックは、1チャンネルのフレームがきっちり入るサイズである必要があります。
            Ensure.Operation.True(this.BlockLength % this.InputFormat.FrameLength == 0);

            this.InitializePacketStores(inputs);
        }

        /// <summary>
        /// 出力接続ポイントを生成します。
        /// </summary>
        /// <returns>出力接続ポイントの配列を返します。</returns>
        protected override OutputConnectionPoint[] CreateOutputs()
        {
            StreamBlockInformation streamBlockInfo = new StreamBlockInformation();
            CalcStreamBlockInfo(
                out streamBlockInfo,
                this.BlockLength,
                this.InputStreamInformation.FrameCount,
                this.InputFormat.Encoding.ToWaveCodecEncoding(this.InputFormat)
                );

            this.WaveStreamData = streamBlockInfo.ToWaveStreamData();

            return new OutputConnectionPoint[]
            {
                new OutputConnectionPoint(
                new WaveStreamInformation(
                    this.GetOutputLength(this.Inputs),
                    this.InputStreamInformation.FrameCount,
                    new WaveFormat(this.InputFormat)
                    {
                        ChannelCount = this.Inputs.Length,
                    }
                    )),
            };
        }

        /// <summary>
        /// パケットを処理します。
        /// </summary>
        /// <param name="input">入力接続ポイントを指定します。</param>
        /// <param name="packet">パケットを指定します。</param>
        protected override void Process(InputConnectionPoint input, Packet packet)
        {
            Assertion.Argument.NotNull(input);
            Assertion.Argument.NotNull(packet);

            this.packetStores.GetValue(input).Add(packet);
        }

        /// <summary>
        /// 全ての入力ストリームが終端に達した場合に実行されます。
        /// </summary>
        protected override void ProcessEndOfAllStreams()
        {
            if (this.packetStores.Count == 0)
            {
                base.ProcessEndOfAllStreams();
                return;
            }

            foreach (PacketStore packetStrore in this.packetStores)
            {
                int paddingLength = this.GetOutputLengthPerChannel() - packetStrore.Length;
                if (paddingLength > 0)
                {
                    packetStrore.Add(Packet.Create(new byte[paddingLength]));
                }
            }

            if (this.OutputFormat.Encoding == Encoding.DspAdpcm)
            {
                this.AdpcmSeekInformations = this.ExtractAdpcmSeekInformations();
            }

            this.SendMixedSamples();

            base.ProcessEndOfAllStreams();
        }

        [DllImport("WaveCodecGeneric.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern void CalcStreamBlockInfo(
            [Out] out StreamBlockInformation info,
            int alignByte,
            int frameCount,
            WaveCodecEncoding method);

        /// <summary>
        /// ２つの WaveFormat を比較します。
        /// </summary>
        /// <param name="left">左辺の WaveFormat を指定します。</param>
        /// <param name="right">右辺の WaveFormat を指定します。</param>
        /// <returns>同等なら true、それ以外なら false を返します。</returns>
        private bool WaveFormatEquals(WaveFormat left, WaveFormat right)
        {
            if (object.ReferenceEquals(left, right))
            {
                return true;
            }

            if (left.Encoding != right.Encoding ||
                left.SamplingRate != right.SamplingRate ||
                left.ChannelCount != right.ChannelCount ||
                left.BitsPerSample != right.BitsPerSample ||
                left.IsLittleEndian != right.IsLittleEndian)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// チャンネル毎の出力ストリーム長を取得します。
        /// </summary>
        /// <returns>チャンネル毎の出力ストリーム長を返します。</returns>
        private int GetOutputLengthPerChannel()
        {
            Ensure.Operation.ObjectNotNull(this.WaveStreamData);
            return this.WaveStreamData.BlockByte * (this.WaveStreamData.BlockCount - 1) +
                this.WaveStreamData.LastBlockPaddedByte;
        }

        /// <summary>
        /// 出力ストリームの長さを取得します。
        /// </summary>
        /// <param name="inputs">入力接続ポイントの配列を指定します。</param>
        /// <returns>出力ストリームの長さを返します。</returns>
        private int GetOutputLength(InputConnectionPoint[] inputs)
        {
            Assertion.Argument.NotNull(inputs);
            return this.GetOutputLengthPerChannel() * inputs.Length;
        }

        /// <summary>
        /// チャンネル毎の出力ストリーム長を取得します。
        /// </summary>
        /// <returns>チャンネル毎のストリーム長を返します。</returns>
        private int GetOutputLengthPerChannel(InputConnectionPoint[] inputs)
        {
            Assertion.Argument.NotNull(inputs);

            int length = 0;

            foreach (InputConnectionPoint input in inputs)
            {
                int paddingLength = 32 - input.StreamInformation.Length % 32;

                if (paddingLength == 32)
                {
                    paddingLength = 0;
                }

                length = Math.Max(input.StreamInformation.Length + paddingLength, length);
            }

            return length;
        }

        /// <summary>
        /// 出力ストリームのフレーム数を取得します。
        /// </summary>
        /// <returns>出力ストリームのフレーム数を返します。</returns>
        private int GetOutputFrameCountPerChannel(InputConnectionPoint[] inputs)
        {
            Assertion.Argument.NotNull(inputs);

            int frameCount = 0;

            foreach (InputConnectionPoint input in inputs)
            {
                frameCount = Math.Max(input.StreamInformation.FrameCount, frameCount);
            }

            return frameCount;
        }

        /// <summary>
        /// パケット倉庫を初期化します。
        /// </summary>
        /// <param name="inputs">入力接続ポイントの配列を指定します。</param>
        private void InitializePacketStores(InputConnectionPoint[] inputs)
        {
            Assertion.Argument.NotNull(inputs);

            int outputLength = this.GetOutputLengthPerChannel(inputs);

            foreach (InputConnectionPoint input in inputs)
            {
                // 各チャンネルのストリーム長を揃えるためにサイズを指定します。
                // 長さが異なる場合は、無音データが補完されます。
                this.packetStores.Add(new PacketStore(input, outputLength));
            }
        }

        /// <summary>
        /// ADPCMシーク情報のバイト列を抽出します。
        /// </summary>
        /// <returns>ADPCMシーク情報のバイト列を返します。</returns>
        private byte[] ExtractAdpcmSeekInformations()
        {
            Ensure.Operation.ObjectNotNull(this.WaveStreamData);

            List<byte[]> adpcmSeekInformations = new List<byte[]>();

            foreach (PacketStore packetStore in this.packetStores)
            {
                adpcmSeekInformations.Add(packetStore.GetAdpcmSeekInformations());
            }

            if (adpcmSeekInformations.Count == 0)
            {
                return new byte[0];
            }

            // 2サンプル(Yn1とYn2分) × ストリームブロック数 × ストリームチャンネル数
            int itemLength = 2 * this.InputFormat.FrameLength;
            byte[] results = new byte[adpcmSeekInformations[0].Length * this.Inputs.Length];
            int current = 0;

            for (int sourceCurrent = 0; sourceCurrent < adpcmSeekInformations[0].Length; sourceCurrent += itemLength)
            {
                foreach (byte[] adpcmSeekInformation in adpcmSeekInformations)
                {
                    Array.Copy(adpcmSeekInformation, sourceCurrent, results, current, itemLength);
                    current += itemLength;
                }
            }

            return results;
        }

        /// <summary>
        /// 混合したストリーム出力を送信します。
        /// </summary>
        private void SendMixedSamples()
        {
            foreach (PacketStore packetStore in this.packetStores)
            {
                packetStore.Reset();
            }

            // 各チャンネルのサンプルをブロック単位で混合して送信します。
            while (!this.packetStores[0].IsEndOfStream)
            {
                foreach (PacketStore packetStore in this.packetStores)
                {
                    byte[] blockSamples = packetStore.Read(this.BlockLength);
                    Ensure.Operation.True(blockSamples.Length > 0);

                    this.SendToAll(Packet.Create(blockSamples));

                    int paddingLength = 32 - blockSamples.Length % 32;

                    if (paddingLength < 32)
                    {
                        this.SendToAll(Packet.Create(new byte[paddingLength]));
                    }
                }
            }

#if DEBUG
            // 正しく出力できているかチェックします。
            foreach (PacketStore packetStore in this.packetStores)
            {
                Assertion.Operation.True(packetStore.IsEndOfStream);
            }
#endif
        }

        /// <summary>
        /// 入力接続ポイント毎にパケットを格納するパケット倉庫クラスです。
        /// </summary>
        private class PacketStore
        {
            private readonly InputConnectionPoint input;
            private readonly LinkedList<Packet> packets = new LinkedList<Packet>();

            private LinkedListNode<Packet> currentNode = null;
            private int currentPacketReadLength = 0;

            private int current = 0;
            private int totalLength = 0;

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="input">入力接続ポイントを指定します。</param>
            /// <param name="length">サンプルストリームの長さを指定します。</param>
            public PacketStore(InputConnectionPoint input, int length)
            {
                Ensure.Argument.NotNull(input);
                this.input = input;
                this.totalLength = length;
            }

            /// <summary>
            /// 対象の入力接続ポイントを取得します。
            /// </summary>
            public InputConnectionPoint Input
            {
                get { return this.input; }
            }

            /// <summary>
            /// 現在の読み込み位置を取得します。
            /// </summary>
            public int Current
            {
                get { return this.current; }
            }

            /// <summary>
            /// サンプルストリームの長さを取得します。
            /// </summary>
            public int Length
            {
                get { return this.totalLength; }
            }

            /// <summary>
            /// ストリームの終端に達しているかどうかを調べます。
            /// </summary>
            public bool IsEndOfStream
            {
                get
                {
                    if (this.Current == this.Length)
                    {
                        return true;
                    }

                    if (this.CurrentPacket == null && this.Length > 0)
                    {
                        throw new Exception("invalid StreamChannelMixer state.");
                    }

                    return false;
                }
            }

            /// <summary>
            /// 現在のパケットを取得します。
            /// </summary>
            public Packet CurrentPacket
            {
                get
                {
                    if (this.currentNode == null)
                    {
                        return null;
                    }
                    return this.currentNode.Value;
                }
            }

            /// <summary>
            /// パケットを追加します。
            /// </summary>
            /// <param name="packet">パケットを指定します。</param>
            public void Add(Packet packet)
            {
                Ensure.Argument.NotNull(packet);

                // 読み込みを開始したら、以降パケットの追加はできないようにします。
                Ensure.Operation.True(this.current == 0);

                this.packets.AddLast(packet);
            }

            /// <summary>
            /// 指定した長さのサンプルを読み込みます。
            /// </summary>
            /// <param name="readLength">読み込むサンプルの長さを指定します。</param>
            /// <returns>読み込んだサンプルのバイト列を返します。</returns>
            public byte[] Read(int readLength)
            {
                Ensure.Argument.True(readLength > 0);

                if (this.CurrentPacket == null)
                {
                    this.Reset();

                    if (this.CurrentPacket == null)
                    {
                        return new byte[0];
                    }
                }

                int copyLength = Math.Min(readLength, this.totalLength - this.current);
                byte[] copiedSamples = new byte[copyLength];

                int copiedLength = 0;

                while (this.CurrentPacket != null && copiedLength < copyLength)
                {
                    int copyLengthFromPacket = Math.Min(
                        this.CurrentPacket.ValidSamplesLength - this.currentPacketReadLength,
                        copyLength - copiedLength);

                    Array.Copy(
                        this.CurrentPacket.Samples, this.currentPacketReadLength,
                        copiedSamples, copiedLength, copyLengthFromPacket);

                    copiedLength += copyLengthFromPacket;
                    this.currentPacketReadLength += copyLengthFromPacket;

                    // 全部読みきったなら、次のパケットに移ります。
                    if (this.currentPacketReadLength >= this.CurrentPacket.ValidSamplesLength)
                    {
                        this.NextPacket();
                    }
                }

                this.current += copyLength;

                return copiedSamples;
            }

            /// <summary>
            /// 指定した長さのサンプルを読み飛ばします。
            /// </summary>
            /// <param name="skipLength">読み飛ばすサンプルの長さを指定します。</param>
            public void Skip(int skipLength)
            {
                if (skipLength == 0)
                {
                    return;
                }

                Ensure.Argument.True(skipLength > 0);

                if (this.CurrentPacket == null)
                {
                    this.Reset();

                    if (this.CurrentPacket == null)
                    {
                        return;
                    }
                }

                int skipedLength = 0;

                while (this.CurrentPacket != null && skipedLength < skipLength)
                {
                    Ensure.Operation.ObjectNotNull(currentNode);

                    int length = Math.Min(
                        this.CurrentPacket.ValidSamplesLength - this.currentPacketReadLength,
                        skipLength - skipedLength);

                    skipedLength += length;
                    this.currentPacketReadLength += length;

                    // 全部読みきったなら、次のパケットに移ります。
                    if (this.currentPacketReadLength >= this.CurrentPacket.ValidSamplesLength)
                    {
                        this.NextPacket();
                    }
                }

                this.current += skipLength;
            }

            /// <summary>
            /// 現在の読み込み位置をリセットします。
            /// </summary>
            public void Reset()
            {
                this.currentNode = this.packets.First;
                this.currentPacketReadLength = 0;
                this.current = 0;
            }

            /// <summary>
            /// ADPCMシーク情報を取得します。
            /// </summary>
            /// <returns>ADPCMシーク情報を返します。</returns>
            public byte[] GetAdpcmSeekInformations()
            {
                int length = 0;

                foreach (Packet packet in this.packets)
                {
                    AdpcmPacket adpcmPacket = packet as AdpcmPacket;
                    if (adpcmPacket != null && adpcmPacket.SeekInformation.Length > 0)
                    {
                        length += adpcmPacket.SeekInformation.Length;
                    }
                }

                byte[] result = new byte[length];
                int current = 0;

                foreach (Packet packet in this.packets)
                {
                    AdpcmPacket adpcmPacket = packet as AdpcmPacket;
                    if (adpcmPacket != null && adpcmPacket.SeekInformation.Length > 0)
                    {
                        Array.Copy(adpcmPacket.SeekInformation, 0, result, current, adpcmPacket.SeekInformation.Length);
                        current += adpcmPacket.SeekInformation.Length;
                    }
                }

                Ensure.Operation.True(current == length);
                return result;
            }

            /// <summary>
            /// カレントを次のパケットに移します。
            /// </summary>
            private void NextPacket()
            {
                Ensure.Operation.ObjectNotNull(this.currentNode);
                Ensure.Operation.True(this.currentNode.Value.ValidSamplesLength == this.currentPacketReadLength);

                this.currentNode = this.currentNode.Next;
                this.currentPacketReadLength = 0;
            }
        }

        /// <summary>
        /// パケット倉庫のリストです。
        /// </summary>
        private class PacketStoreList :
            KeyedListDecorator<InputConnectionPoint, PacketStore>
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public PacketStoreList()
                : base(new ObservableList<PacketStore>(), GetInputConnectionPoint)
            {
            }

            /// <summary>
            /// アイテムからキーを取得します。
            /// </summary>
            /// <param name="item">アイテムを指定します。</param>
            /// <returns>キーを返します。</returns>
            private static InputConnectionPoint GetInputConnectionPoint(PacketStore item)
            {
                return item.Input;
            }
        }
    }
}
