﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Linq;
using System.Text;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TreeDrawer : ITreeDrawer
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(TreeDrawDescriptor desc, ITreeItem item)
        {
            Graphics gc = desc.Gc;

            DrawRelationLine(desc.Gc, item,
                              desc.Rectangle.X + desc.Rectangle.Width,
                              desc.Rectangle.Y,
                              item.Length,
                              desc.IndentLength,
                              desc.DivergePoint,
                              desc.ExpanderBounds);

            //
            int x = 0;
            int y = 0;
            int width = 0;
            int height = 0;

            x = desc.Rectangle.X + desc.Rectangle.Width;
            width = desc.ClientWidth - x;

            switch (desc.DragItemInPosition)
            {
                case TreeItemInPosition.None:
                    break;

                case TreeItemInPosition.Center:
                    y = desc.Rectangle.Y;
                    height = item.Length;
                    break;

                case TreeItemInPosition.Upper:
                    y = desc.Rectangle.Y;
                    height = 4;
                    break;

                case TreeItemInPosition.Lower:
                    y = desc.Rectangle.Y + item.Length - 4;
                    height = 4;
                    break;
            }

            if (height > 0)
            {
                Brush brush = new SolidBrush
                    (Color.FromArgb((int)(128.0F * 0.5F),
                                      SystemColors.Highlight.R,
                                      SystemColors.Highlight.G,
                                      SystemColors.Highlight.B));
                gc.FillRectangle(brush, x, y, width, height);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void DrawExpander(Graphics gc, ITreeItem item, int x, int y, int width, int height, bool expanded)
        {
            SolidBrush brush = new SolidBrush(SystemColors.Window);
            Pen penGray = new Pen(Color.Gray);
            Pen penBlack = new Pen(Color.Black);
            Rectangle rect;
            int cx = x + width / 2;
            int cy = y + height / 2;
            rect = new Rectangle(x, y, width, height);

            gc.FillRectangle(brush, rect);
            gc.DrawRectangle(penGray, rect);
            gc.DrawLine(penBlack,
                         cx + 2, cy,
                         cx - 2, cy);

            if (expanded == false)
            {
                gc.DrawLine(penBlack,
                             cx, cy + 2,
                             cx, cy - 2);
            }
        }

        /// <summary>
        /// 矩形内(rect)の中心(divergePoint)から、上方向(uLine)、右方向(rLine)、下方向(bLine)に線を描画します。
        /// </summary>
        protected void DrawRelationLine(Graphics gc, Rectangle rect, bool rLine, bool uLine, bool bLine, Point divergePoint)
        {
            int tY = rect.Y;
            int cY = tY + divergePoint.Y;
            int bY = tY + rect.Height;
            int cX = rect.X + divergePoint.X;
            int rX = rect.X + rect.Width;
            var pen = new Pen(Color.Gray)
            {
                DashStyle = DashStyle.Dot,
            };

            if (uLine == true) { gc.DrawLine(pen, cX, tY, cX, cY); }
            if (rLine == true) { gc.DrawLine(pen, cX, cY, rX, cY); }
            if (bLine == true) { gc.DrawLine(pen, cX, cY, cX, bY); }
        }

        protected void DrawRelationLine(Graphics gc, ITreeItem item, int x, int y, int height, int indentLength, Point divergePoint, Rectangle expanderBounds)
        {
            if (item.Parent == null)
            {
                return;
            }

            var rect = new Rectangle()
            {
                X = x - indentLength,
                Y = y,
                Width = indentLength,
                Height = height,
            };

            // エキスパンダと、それに繋がる線を描画します。
            DrawRelationLine(gc, rect, true, true, HasVisibleNextSibling(item), divergePoint);

            if (item.Expander != false)
            {
                DrawExpander(gc,
                    item,
                    rect.X + expanderBounds.X,
                    rect.Y + expanderBounds.Y,
                    expanderBounds.Width,
                    expanderBounds.Height,
                    item.Expanded);
            }

            var ancestor = item.Parent;
            rect.X -= indentLength;

            // 親階層の兄弟間を接続する縦線を描画します。
            while (ancestor != null)
            {
                if (HasVisibleNextSibling(ancestor) == true)
                {
                    DrawRelationLine(gc, rect, false, true, true, divergePoint);
                }

                ancestor = ancestor.Parent;
                rect.X -= indentLength;
            }
        }

        private bool HasVisibleNextSibling(ITreeItem item)
        {
            Debug.Assert(item != null);

            item = item.NextSibling;

            while (item != null)
            {
                if (item.Visible == true)
                {
                    return true;
                }
                item = item.NextSibling;
            }
            return false;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// アイテムの描画クラス
    /// </summary>
    public class TreeItemDrawer : ITreeItemDrawer
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(TreeItemDrawDescriptor desc)
        {
            desc.Gc.FillRectangle(new SolidBrush(SystemColors.Window), desc.Bounds);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// アイコンの描画クラス
    /// </summary>
    public class TreePartIconDrawer : ITreePartDrawer
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "Icon"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(TreeItemDrawDescriptor desc)
        {
            Image image = null;
            string imageName = String.Empty;
            bool expanded = false;

            if (desc.Parameters.ContainsKey("Expanded") != false)
            {
                expanded = bool.Parse(desc.Parameters["Expanded"]);

                desc.Parameters.TryGetValue
                    (expanded != false ? "ExpandIcon" : "CollapseIcon", out imageName);

                if (imageName != null)
                {
                    if (desc.Images.ContainsKey(imageName) != false)
                    {
                        image = desc.Images[imageName];
                    }
                }
            }

            if (image != null)
            {

                ImageAttributes attr = new ImageAttributes();
                attr.SetColorKey(Color.Magenta, Color.Magenta);
                desc.Gc.DrawImage(image, desc.Bounds,
                                   0, 0, desc.Bounds.Width, desc.Bounds.Height,
                                   GraphicsUnit.Pixel, attr);
            }
            else
            {

                desc.Gc.FillRectangle(new SolidBrush(Color.Red),
                                       desc.Bounds.X, desc.Bounds.Y,
                                       desc.Bounds.Width - 1, desc.Bounds.Height - 1);
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// キャプションの描画クラス
    /// </summary>
    public class TreePartCaptionDrawer : ITreePartDrawer
    {
        private static StringFormat _Format = new StringFormat();
        private static Brush _FtStBrush = null;      //フォーカス  On、選択 On
        private static Brush _FfStBrush = null;      //フォーカス Off、選択 On

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static TreePartCaptionDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Near;
            _Format.FormatFlags = StringFormatFlags.NoWrap;

            _FtStBrush = new SolidBrush(Color.FromArgb
                                         (SystemColors.Highlight.R * 7 / 8,
                                           SystemColors.Highlight.G * 7 / 8,
                                           SystemColors.Highlight.B * 7 / 8));

            _FfStBrush = SystemBrushes.ControlDark;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "Caption"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(TreeItemDrawDescriptor desc)
        {
            Graphics gc = desc.Gc;
            //Color       captionColor = SystemColors.WindowText;
            Pen pen = null;
            Brush brush = null;
            bool selected = desc.Selected | desc.TemporarySelected;
            string caption = null;

            if (selected != false)
            {
                //captionColor = SystemColors.HighlightText;
                gc.FillRectangle(desc.Focused != false ? _FtStBrush : _FfStBrush, desc.Bounds);
            }

            caption = desc.Parameters["Caption"];
            brush = GetCaptionBrush(desc);
            gc.DrawString(caption, desc.Font, brush, desc.Bounds, _Format);

            //
            if (desc.Caret != false && desc.Focused != false)
            {
                pen = new Pen(selected != false ? Color.White : Color.Black);
                pen.DashStyle = DashStyle.Dot;
                pen.DashOffset = desc.Counter;
                gc.DrawRectangle(pen, desc.Bounds.X, desc.Bounds.Y,
                                  desc.Bounds.Width - 1, desc.Bounds.Height - 1);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual Brush GetCaptionBrush(TreeItemDrawDescriptor desc)
        {
            bool selected = desc.Selected | desc.TemporarySelected;
            return new SolidBrush(selected != false ?
                                   SystemColors.HighlightText :
                                   SystemColors.WindowText);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TreeDrawDescriptor
    {
        private Graphics _Gc = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeDrawDescriptor(Graphics gc)
        {
            _Gc = gc;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Graphics Gc
        {
            get { return _Gc; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Rectangle Rectangle { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Point DivergePoint { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Rectangle ExpanderBounds { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int IndentLength { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeItemInPosition DragItemInPosition { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int ClientWidth { get; set; }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TreeItemDrawDescriptor
    {
        private Graphics _Gc = null;
        private Font _Font = null;
        private Dictionary<string, string> _Parameters = null;
        private TreeImageDictionary _Images = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeItemDrawDescriptor(Graphics gc, Font font, TreeImageDictionary images)
        {
            _Gc = gc;
            _Font = font;
            _Parameters = new Dictionary<string, string>();
            _Images = images;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Graphics Gc
        {
            get { return _Gc; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Font Font
        {
            get { return _Font; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeImageDictionary Images
        {
            get { return _Images; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Rectangle Bounds { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Height
        {
            get { return Bounds.Height; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Dictionary<string, string> Parameters
        {
            get { return _Parameters; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool Focused { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool Enabled { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool Selected { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool TemporarySelected { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool Caret { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Counter { get; set; }

        /// <summary>
        ///
        /// </summary>
        public ITreeItem TreeItem { get; internal set; }
    }
}
