﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Windows.CommandBars;

    /// <summary>
    /// ToolStrip からコマンドバーモデルを構築します。
    /// </summary>
    public class CommandBarFactory
    {
        private CommandService _commandService = null;
        private Dictionary<Type, CreateItemHandler> _itemFactoryMethods = new Dictionary<Type, CreateItemHandler>();

        private delegate CommandBarItem CreateItemHandler(ToolStripItem item, string uri, Command command);

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="commandService">コマンドサービス。</param>
        public CommandBarFactory(CommandService commandService)
        {
            if (null == commandService) { throw new ArgumentNullException("commandService"); }
            _commandService = commandService;

            _itemFactoryMethods.Add(typeof(ToolStripMenuItem), CreateMenuItem);
            _itemFactoryMethods.Add(typeof(ToolStripButton), CreateButton);
            _itemFactoryMethods.Add(typeof(ToolStripLabel), CreateLabel);
            _itemFactoryMethods.Add(typeof(ToolStripDropDownButton), CreateDropDownButton);
            _itemFactoryMethods.Add(typeof(ToolStripComboBox), CreateComboBox);
        }

        /// <summary>
        /// ToolStrip からコマンドバーモデルを作成します。
        /// </summary>
        /// <param name="toolStrip">ソースとなる ToolStrip。</param>
        /// <returns>作成されたコマンドバーモデル。</returns>
        public CommandBar CreateCommandBar(ToolStrip toolStrip)
        {
            if (null == toolStrip) { throw new ArgumentNullException("toolStrip"); }

            CommandBar menuModel = new CommandBar(toolStrip.Tag as string);

            foreach (CommandBarItemGroup group in CreateCommandBarItemGroups(toolStrip.Items))
            {
                menuModel.Groups.Add(group);
            }

            return menuModel;
        }

        private CommandBarItemGroup[] CreateCommandBarItemGroups(IEnumerable items)
        {
            if (null == items) { throw new ArgumentNullException("items"); }

            List<CommandBarItemGroup> groups = new List<CommandBarItemGroup>();
            CommandBarItemGroup group = null;

            foreach (ToolStripItem item in items)
            {

                if (item is ToolStripSeparator && null != group)
                {
                    groups.Add(group);
                    group = null;
                    continue;
                }

                string uri = (item.Tag is string) ? item.Tag as string : string.Empty;
                Command command = GetCommand(uri);

                if (!this.IsEnableCommand(uri, command))
                {
                    continue;
                }

                if (uri.Length > 0 && !uri.Contains("@") && null == command)
                {
                    continue;
                }

                if (null == group)
                {
                    group = new CommandBarItemGroup((null == command) ? string.Empty : command.ID);
                }

                if (_itemFactoryMethods.ContainsKey(item.GetType()))
                {
                    group.Items.Add(
                        _itemFactoryMethods[item.GetType()](item, uri, command)
                        );
                }

            }

            if (null != group)
            {
                groups.Add(group);
            }

            return groups.ToArray();
        }

        private Command GetCommand(string uri)
        {
            if (null == uri) { throw new ArgumentNullException("uri"); }

            CommandUri newUri = new CommandUri(uri);
            if (0 == newUri.ID.Length) { return null; }

            Command baseCommand = null;

            if (_commandService.Commands.Contains(newUri.Uri))
            {
                baseCommand = _commandService.Commands[newUri.Uri];
            }
            else if (_commandService.Commands.Contains(newUri.ID))
            {
                baseCommand = _commandService.Commands[newUri.ID];
            }

            if (newUri.ID == newUri.Uri) { return baseCommand; }


            Command command = new Command(
                                    newUri.Uri,
                                    (null != baseCommand) ? baseCommand.Name : string.Empty);

            if (null != baseCommand)
            {
                command.Text = baseCommand.Text;
                command.ToolTipText = baseCommand.ToolTipText;
                command.ShortcutKeyText = baseCommand.ShortcutKeyText;
                command.Image = baseCommand.Image;
            }

            return command;
        }

        private bool IsEnableCommand(string uri, Command command)
        {
            if (uri == null)
            {
                return false;
            }

            // URI がない場合や @ を含む場合は、
            // サブメニューを持っているものとみなし、有効とします。
            if (uri.Length == 0 || uri.Contains("@"))
            {
                return true;
            }

            // 対応するコマンドが存在する場合は有効とします。
            if (command != null)
            {
                return true;
            }

            return false;
        }

        private CommandBarItem CreateMenuItem(ToolStripItem item, string uri, Command command)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (!(item is ToolStripMenuItem)) { throw new ArgumentException("invalid tool strip item type."); }

            CommandBarItem itemModel = null;
            ToolStripMenuItem menuItem = item as ToolStripMenuItem;

            // ポップアップメニューアイテムの場合
            if (0 < menuItem.DropDownItems.Count || null == command)
            {
                return CreateCommandBarPopupItem(uri, item as ToolStripMenuItem);
            }

            // 以下、メニューアイテムの場合
            if (null == command) { throw new ArgumentNullException("command"); }
            itemModel = new CommandBarMenuItem(command.Uri, command);

            if (string.IsNullOrEmpty(itemModel.Text) && 0 < menuItem.Text.Length)
            {
                itemModel.Text = menuItem.Text;
            }

            if (null != menuItem.Image)
            {
                (itemModel as CommandBarMenuItem).Image = menuItem.Image;
            }

            if (Keys.None != menuItem.ShortcutKeys)
            {
                _commandService.AddKeyBinding(menuItem.ShortcutKeys.ToKeyStroke(), command);
                command.ShortcutKeyText = menuItem.ShortcutKeys.ToShortcutKeyText();
            }
            else if (null != menuItem.ShortcutKeyDisplayString &&
                      0 < menuItem.ShortcutKeyDisplayString.Length)
            {

                Keys[] keys = KeysEx.ParseKeys(menuItem.ShortcutKeyDisplayString);

                if (0 < keys.Length && keys.Length < 3)
                {

                    KeyStroke[] keyStrokes = keys.ToKeyStroke();

                    try
                    {

                        switch (keyStrokes.Length)
                        {
                            case 1:
                                _commandService.AddKeyBinding(keyStrokes[0], command);
                                break;

                            case 2:
                                _commandService.AddKeyBinding(keyStrokes[0], keyStrokes[1], command);
                                break;

                            default:
                                throw new Exception("expected error");
                        }

                        command.ShortcutKeyText = keys.ToShortcutKeyText();

                    }
                    catch (ArgumentException)
                    {
                        // 既に登録済みのショートカットキーはスキップ
                    }
                }

            }

            return itemModel;
        }

        private CommandBarPopupItem CreateCommandBarPopupItem(string uri, ToolStripDropDownItem menuItem)
        {
            if (null == menuItem) { throw new ArgumentNullException("menuItem"); }

            CommandBarPopupItem itemModel = new CommandBarPopupItem(uri);

            if (string.IsNullOrEmpty(itemModel.Text) && 0 < menuItem.Text.Length)
            {
                itemModel.Text = menuItem.Text;
            }

            if (null != uri && uri.ToLower() == "@hidden")
            {
                itemModel.Visible = menuItem.Visible;
            }

            foreach (CommandBarItemGroup group in CreateCommandBarItemGroups(menuItem.DropDownItems))
            {
                itemModel.Groups.Add(group);
            }

            return itemModel;
        }

        private CommandBarButton CreateButton(ToolStripItem item, string uri, Command command)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (null == command) { throw new ArgumentNullException("command"); }
            if (!(item is ToolStripButton)) { throw new ArgumentException("invalid tool strip item type."); }

            CommandBarButton itemModel = new CommandBarButton(uri, command)
            {
                Width = item.Width,
            };

            if (string.IsNullOrEmpty(itemModel.Text) && 0 < item.Text.Length)
            {
                itemModel.Text = item.Text;
            }

            if (string.IsNullOrEmpty(itemModel.ToolTipText) && 0 < item.ToolTipText.Length)
            {
                itemModel.ToolTipText = item.ToolTipText;
            }

            switch (item.DisplayStyle)
            {
                case ToolStripItemDisplayStyle.Text:
                    itemModel.Style = CommandBarButtonStyle.TextOnly;
                    break;

                case ToolStripItemDisplayStyle.Image:
                    itemModel.Style = CommandBarButtonStyle.ImageOnly;
                    break;

                case ToolStripItemDisplayStyle.ImageAndText:
                default:
                    switch (item.TextImageRelation)
                    {
                        case TextImageRelation.ImageBeforeText:
                            itemModel.Style = CommandBarButtonStyle.TextAfterImage;
                            break;

                        case TextImageRelation.TextBeforeImage:
                        default:
                            itemModel.Style = CommandBarButtonStyle.TextBeforeImage;
                            break;
                    }

                    break;
            }

            return itemModel;
        }

        private CommandBarLabel CreateLabel(ToolStripItem item, string uri, Command command)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (!(item is ToolStripLabel)) { throw new ArgumentException("invalid tool strip item type."); }

            CommandBarLabel itemModel = new CommandBarLabel(uri)
            {
                Width = item.Width,
            };

            if (string.IsNullOrEmpty(itemModel.Text) && 0 < item.Text.Length)
            {
                itemModel.Text = item.Text;
            }

            return itemModel;
        }

        private CommandBarDropDownButton CreateDropDownButton(ToolStripItem item, string uri, Command command)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (!(item is ToolStripDropDownButton)) { throw new ArgumentException("invalid tool strip item type."); }

            CommandBarDropDownButton itemModel = new CommandBarDropDownButton(uri)
            {
                Width = item.Width,
            };

            if (string.IsNullOrEmpty(itemModel.Text) && 0 < item.Text.Length)
            {
                itemModel.Text = item.Text;
            }

            foreach (CommandBarItemGroup group in CreateCommandBarItemGroups((item as ToolStripDropDownButton).DropDownItems))
            {
                itemModel.Groups.Add(group);
            }

            return itemModel;
        }

        private CommandBarComboBox CreateComboBox(ToolStripItem item, string uri, Command command)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (!(item is ToolStripComboBox)) { throw new ArgumentException("invalid tool strip item type."); }

            ToolStripComboBox comboBox = item as ToolStripComboBox;

            CommandBarComboBox itemModel = new CommandBarComboBox(uri)
            {
                Width = comboBox.Width,
            };

            switch (comboBox.DropDownStyle)
            {
                case ComboBoxStyle.DropDownList:
                    itemModel.DropDownStyle = CommandBarComboBoxStyle.DropDownList;
                    break;

                case ComboBoxStyle.DropDown:
                default:
                    itemModel.DropDownStyle = CommandBarComboBoxStyle.DropDown;
                    break;
            }

            return itemModel;
        }
    }
}
