﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Design;
using System.Globalization;
using System.Text;
using System.Windows.Forms;

using NintendoWare.SoundFoundation.Windows.Forms.Windowless;


namespace NintendoWare.SoundFoundation.Windows.Forms
{
    public partial class NTabControl : Control
    {
        #region ** 固定値

        public const int InvalidIndex = -1;

        // イメージインデックス
        protected const int DropDownImageIndex = 0;    // ドロップダウンボタン
        protected const int CollapseUpImageIndex = 1;    // 上に最小化ボタン
        protected const int CollapseDownImageIndex = 2;    // 下に最小化ボタン
        protected const int CollapseLeftImageIndex = 3;    // 左に最小化ボタン
        protected const int CollapseRightImageIndex = 4;    // 右に最小化ボタン

        #endregion

        #region ** フィールド

        // コアコントロール
        private NWControlHost _host = null;       // ホスト
        private NWTabControl.Core _model = null;       // コアコントロールモデル
        private NWTabControl.TabLayoutData _layoutData = null;       // レイアウトデータ

        // モデル
        private NTabPageCollection _tabPages = null;                 // タブページ

        // WindowsForms コンポーネント
        private ContextMenuStrip _contextMenu = null;                // コンテキストメニュー
        private BufferedGraphics _bgSurface = null;                // バックグラウンドサーフェイス

        // スタイル
        private StyleDictionary _styles = new NTabControlStyleDictionary();    // スタイルディクショナリ
        private NTabControlStyle _style = NTabControlStyle.NormalTopTab;       // スタイル

        // パラメータ
        private bool _selectionExpression = false;                  // 選択タブの強調表示設定
        private bool _showTooltips = false;                  // ツールチップ表示設定
        private bool _showDropDownButton = true;                   // ドロップダウンボタン表示設定
        private bool _showTabCloseButton = true;                   // タブ上の閉じるボタン表示設定
        private bool _minimizable = false;                  // 最小化可否設定

        #endregion

        public NTabControl()
        {
            base.DoubleBuffered = false;
            base.SetStyle(ControlStyles.Selectable, false);

            InitializeComponent();


            // コアコンポーネントを初期化する
            _host = new NWControlHost(this);
            _model = new NWTabControl.Core(_host);
            _layoutData = new NWTabControl.TabLayoutData();

            _host.Initialized += OnHostIntialized;
            _host.Uninitialize += OnHostUninitialize;

            _model.ItemSwapped += OnCoreSwapped;
            _model.ItemClosing += OnCoreClosing;
            _model.ItemClosed += OnCoreClosed;
            _model.ItemSelecting += OnCoreSelecting;
            _model.ItemSelected += OnCoreSelected;
            _model.ItemDeselecting += OnCoreDeselecting;
            _model.ItemDeselected += OnCoreDeselected;
            _model.SelectedIndexChanged += OnCoreSelectedIndexChanged;
            _model.SelectedTabChanged += OnCoreSelectedTabChanged;

            _style = DefaultTabControlStyle;
            _host.Style = Styles[_style];

            // NWTabControl.Core.Tabs と NTabPageCollection をリンクする
            _tabPages = new NTabPageCollection(CoreModel.TabPages);
        }

        #region ** プロパティ

        /// <summary>
        /// コントロールのタブの外観を表す値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.All)]
        [DefaultValue(NTabControlStyle.NormalTopTab)]
        public NTabControlStyle TabControlStyle
        {
            get { return _style; }
            set
            {
                if (value == _style) { return; }
                SetTabControlStyle(value);
            }
        }

        /// <summary>
        /// コントロールのタブのサイズを取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [DefaultValue(typeof(Size), "59, 20")]
        public Size ItemSize
        {
            get { return _layoutData.ItemSize; }
            set
            {
                if (value == _layoutData.ItemSize) { return; }

                _layoutData.ItemSize = value;
                PerformLayout();
            }
        }

        /// <summary>
        /// コントロールのタブの最大幅を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [DefaultValue(100)]
        public int ItemMaxWidth
        {
            get { return _layoutData.ItemMaxWidth; }
            set { _layoutData.ItemMaxWidth = value; }
        }

        [Category("Appearance")]
        [DefaultValue(true)]
        public bool ShowDropDownButton
        {
            get { return _showDropDownButton; }
            set
            {
                _showDropDownButton = value;

                if (null != CoreControl)
                {
                    CoreControl.ShowDropDownButton = value;
                }
            }
        }

        [Category("Appearance")]
        [DefaultValue(true)]
        public bool ShowTabCloseButton
        {
            get { return _showTabCloseButton; }
            set
            {
                _showTabCloseButton = value;

                if (null != CoreControl)
                {
                    CoreControl.ShowTabCloseButton = value;
                }
            }
        }

        [Category("Appearance")]
        [DefaultValue(false)]
        public bool Minimizable
        {
            get { return _minimizable; }
            set
            {
                _minimizable = value;

                if (CoreControl is NTabControlCore)
                {
                    (CoreControl as NTabControlCore).Minimizable = value;
                }
            }
        }

        [Category("Appearance")]
        [DefaultValue(false)]
        public bool SelectionExpression
        {
            get { return _selectionExpression; }
            set
            {
                _selectionExpression = value;

                if (null != CoreControl)
                {
                    CoreControl.SelectionExpression = value;
                }
            }
        }

        [Category("Appearance")]
        [DefaultValue(null)]
        public ImageList ImageList
        {
            // ImageList インスタンスは NTabControlでもったほうがよいかも。 Forms名前空間なので。
            get { return CoreModel.ImageList; }
            set { CoreModel.ImageList = value; }
        }

        /// <summary>
        /// タブ コントロールにあるタブ ページのコレクションを取得します。
        /// </summary>
        [Category("Behavior")]
        public NTabPageCollection TabPages
        {
            get { return _tabPages; }
        }

        /// <summary>
        /// このコントロールに関連付けられている System.Windows.Forms.ContextMenuStrip を取得または設定します。
        /// </summary>
        [DefaultValue("")]
        public new ContextMenuStrip ContextMenuStrip
        {
            get { return _contextMenu; }
            set { _contextMenu = value; }
        }

        /// <summary>
        /// ツールチップの表示設定を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [DefaultValue(false)]
        public bool ShowToolTips
        {
            get
            {
                if (null == CoreControl)
                {
                    return false;
                }
                return CoreControl.ShowToolTips;
            }
            set
            {
                _showTooltips = value;

                if (null != CoreControl)
                {
                    CoreControl.ShowToolTips = value;
                }
            }
        }

        [Category("Behavior")]
        [DefaultValue(InvalidIndex)]
        public int SelectedIndex
        {
            get { return CoreModel.SelectedIndex; }
            set
            {
                if (InvalidIndex == value) { return; }
                if (value >= CoreModel.TabPages.Count) { return; }
                CoreModel.SelectedTab = CoreModel.TabPages[value];
            }
        }

        [ReadOnly(true)]
        [Browsable(false)]
        public INTabPage SelectedTab
        {
            get
            {
                if (null == CoreModel.SelectedTab) { return null; }
                return CoreModel.SelectedTab.Page as INTabPage;
            }
            set
            {
                CoreModel.SelectedTab = CoreModel.TabPages[CoreModel.TabPages.IndexOf(value)];
            }
        }

        /// <summary>
        /// コントロールのタブの外観を表すデフォルト値を取得します。
        /// </summary>
        [ReadOnly(true)]
        [Browsable(false)]
        protected virtual NTabControlStyle DefaultTabControlStyle
        {
            get { return NTabControlStyle.NormalTopTab; }
        }

        protected virtual StyleDictionary Styles
        {
            get { return _styles; }
        }

        protected ImageList Icons
        {
            get { return _icons; }
        }

        protected NWControlHost CoreControlHost
        {
            get { return _host; }
        }

        protected NWTabControl CoreControl
        {
            get { return _host.Root as NWTabControl; }
        }

        protected NWTabControl.Core CoreModel
        {
            get { return _model; }
        }

        #region ** 非公開にするプロパティ

        /// <summary>
        /// base.DoubleBuffered を非公開にします。
        /// </summary>
        private new bool DoubleBuffered
        {
            get { return false; }
        }

        /// <summary>
        /// base.ContextMenu を非公開にします。
        /// </summary>
        private new ContextMenu ContextMenu
        {
            get { return null; }
        }

        #endregion

        #endregion

        #region ** イベント

        /// <summary>
        /// タブコントロールのスタイルが変更されたときに発生します。
        /// </summary>
        public event EventHandler TabControlStyleChanged;

        /// <summary>
        /// タブが選択されたときに発生します。
        /// </summary>
        public event NTabControlCancelEventHandler Closing;

        /// <summary>
        /// タブが選択されたときに発生します。
        /// </summary>
        public event NTabControlEventHandler Closed;

        /// <summary>
        /// タブが選択される前に発生し、ハンドラがタブの変更をキャンセルできるようにします。
        /// </summary>
        public event NTabControlCancelEventHandler Selecting;

        /// <summary>
        /// タブが選択されたときに発生します。
        /// </summary>
        public event NTabControlEventHandler Selected;

        /// <summary>
        /// タブの選択が解除される前に発生し、ハンドラがタブの変更をキャンセルできるようにします。
        /// </summary>
        public event NTabControlCancelEventHandler Deselecting;

        /// <summary>
        /// タブの選択が解除されたときに発生します。
        /// </summary>
        public event NTabControlEventHandler Deselected;

        /// <summary>
        /// タブの順番が入れ替えられたときに発生します。
        /// </summary>
        public event NTabControlSwapEventHandler Swapped;

        /// <summary>
        /// コンテキストメニューが表示が解除される前に発生し、ハンドラがキャンセルできるようにします。
        /// </summary>
        public event NTabControlCancelEventHandler ContextMenuShown;

        /// <summary>
        /// SelectedIndex プロパティが変更されると発生します。
        /// </summary>
        public event EventHandler SelectedIndexChanged;

        /// <summary>
        /// SelectedTab プロパティが変更されると発生します。
        /// </summary>
        public event EventHandler SelectedTabChanged;

        /// <summary>
        /// オブジェクトがドラッグされると発生します。
        /// </summary>
        public event NTabControlDragEventHandler Dragged;

        #endregion

        #region ** イベントハンドラ

        #region ** NTabControl イベントハンドラ

        protected virtual void OnTabControlStyleChanged(EventArgs e)
        {
            if (null != TabControlStyleChanged)
            {
                TabControlStyleChanged(this, e);
            }
        }

        protected virtual void OnSwapped(NTabControlSwapEventArgs e)
        {
            if (null != Swapped)
            {
                Swapped(this, e);
            }
        }

        protected virtual void OnClosing(NTabControlCancelEventArgs e)
        {
            if (null != Closing)
            {
                Closing(this, e);
            }
        }

        protected virtual void OnClosed(NTabControlEventArgs e)
        {
            if (null != Closed)
            {
                Closed(this, e);
            }
        }

        protected virtual void OnSelecting(NTabControlCancelEventArgs e)
        {
            if (null != Selecting)
            {
                Selecting(this, e);
            }
        }

        protected virtual void OnSelected(NTabControlEventArgs e)
        {
            if (null != Selected)
            {
                Selected(this, e);
            }
        }

        protected virtual void OnDeselecting(NTabControlCancelEventArgs e)
        {
            if (null != Deselecting)
            {
                Deselecting(this, e);
            }
        }

        protected virtual void OnDeselected(NTabControlEventArgs e)
        {
            if (null != Deselected)
            {
                Deselected(this, e);
            }
        }

        protected virtual void OnContextMenuShown(NTabControlCancelEventArgs e)
        {
            if (null != ContextMenuShown)
            {
                ContextMenuShown(this, e);
            }
        }

        protected virtual void OnSelectedIndexChanged(EventArgs e)
        {
            if (InvalidIndex != CoreModel.SelectedIndex && null != CoreControl)
            {
                CoreControl.EnsureVisible(CoreModel.SelectedTab);
            }

            if (null != SelectedIndexChanged)
            {
                SelectedIndexChanged(this, e);
            }
        }

        protected virtual void OnSelectedTabChanged(EventArgs e)
        {
            if (null != SelectedTabChanged)
            {
                SelectedTabChanged(this, e);
            }
        }

        protected virtual void OnDragged(NTabControlDragEventArgs e)
        {
            if (null != Dragged)
            {
                Dragged(this, e);
            }
        }

        #region ** タブ操作イベント

        protected virtual void OnSelectNextTab(CancelEventArgs e) { }

        protected virtual void OnSelectPrevTab(CancelEventArgs e) { }

        protected virtual void OnCloseSelectedTab(CancelEventArgs e) { }

        #endregion

        #endregion

        #region ** コアコントロールイベント

        protected virtual void OnHostIntialized(object sender, EventArgs e)
        {
            // モデルを関連付ける
            CoreControl.Model = _model;
            CoreControl.LayoutData = _layoutData;

            // イベントハンドラを登録する
            CoreControl.Initialized += OnCoreControlInitialized;

            CoreControl.Invalidated += OnCoreInvalidated;
            CoreControl.Dragged += OnCoreDragged;

            // NTabControlCore 専用の設定
            if (CoreControl is NTabControlCore)
            {

                NTabControlCore core = CoreControl as NTabControlCore;

                // アイコン設定
                core.SetResource(NTabControlCore.DropDownButton.ControlName, Icons.Images[DropDownImageIndex]);

                switch (TabControlStyle)
                {
                    case NTabControlStyle.NormalMinimizableTopTab:
                    case NTabControlStyle.MinimizedTopTab:
                        core.SetResource(NTabControlCore.MinimizeButton.ControlName, Icons.Images[CollapseUpImageIndex]);
                        core.SetResource(NTabControlCore.RestoreButton.ControlName, Icons.Images[CollapseDownImageIndex]);
                        break;

                    case NTabControlStyle.NormalMinimizableBottomTab:
                    case NTabControlStyle.MinimizedBottomTab:
                        core.SetResource(NTabControlCore.MinimizeButton.ControlName, Icons.Images[CollapseDownImageIndex]);
                        core.SetResource(NTabControlCore.RestoreButton.ControlName, Icons.Images[CollapseUpImageIndex]);
                        break;

                    case NTabControlStyle.MinimizedLeftTab:
                        core.SetResource(NTabControlCore.MinimizeButton.ControlName, Icons.Images[CollapseLeftImageIndex]);
                        core.SetResource(NTabControlCore.RestoreButton.ControlName, Icons.Images[CollapseRightImageIndex]);
                        break;

                    case NTabControlStyle.MinimizedRightTab:
                        core.SetResource(NTabControlCore.MinimizeButton.ControlName, Icons.Images[CollapseRightImageIndex]);
                        core.SetResource(NTabControlCore.RestoreButton.ControlName, Icons.Images[CollapseLeftImageIndex]);
                        break;
                }

                // パラメータ設定
                core.Minimizable = _minimizable;

            }

            // パラメータ設定
            CoreControl.SelectionExpression = _selectionExpression;
            CoreControl.ShowDropDownButton = _showDropDownButton;
            CoreControl.ShowTabCloseButton = _showTabCloseButton;
            CoreControl.ShowToolTips = _showTooltips;

            // サイズ設定
            CoreControl.Bounds = ClientRectangle;
        }

        private void OnHostUninitialize(object sender, EventArgs e)
        {
            // イベントハンドラの登録を解除する
            CoreControl.Initialized -= OnCoreControlInitialized;
            CoreControl.Uninitialize -= OnCoreControlUninitialize;

            CoreControl.Invalidated -= OnCoreInvalidated;
            CoreControl.Dragged -= OnCoreDragged;
        }

        private void OnCoreControlInitialized(object sender, EventArgs e)
        {
            // イベントハンドラを登録する
            // NTabPanel.*Button.Click
            if (CoreControl is NTabControlCore)
            {
                NTabControlCore core = CoreControl as NTabControlCore;
                core.TabPanel.MinimizeButton.Click += OnMinimizeButtonClick;
                core.TabPanel.RestoreButton.Click += OnRestoreButtonClick;
            }

            // NWTab.MouseClick
            foreach (NWTabControl.NWTab tab in CoreControl.TabPanel.Controls)
            {
                tab.MouseClick += OnTabMouseClicked;
            }

            // NWTabPanel.ControlInserted
            CoreControl.TabPanel.ControlInserted += OnTabInserted;
            CoreControl.TabPanel.ControlRemoved += OnTabRemoved;
        }

        private void OnCoreControlUninitialize(object sender, EventArgs e)
        {
            // イベントハンドラを登録する
            // NTabPanel.*Button.Click
            if (CoreControl is NTabControlCore)
            {
                NTabControlCore core = CoreControl as NTabControlCore;
                core.TabPanel.MinimizeButton.Click -= OnMinimizeButtonClick;
                core.TabPanel.RestoreButton.Click -= OnRestoreButtonClick;
            }

            // NWTab.MouseClick
            foreach (NWTabControl.NWTab tab in CoreControl.TabPanel.Controls)
            {
                tab.MouseClick -= OnTabMouseClicked;
            }

            // NWTabPanel.ControlInserted
            CoreControl.TabPanel.ControlInserted -= OnTabInserted;
            CoreControl.TabPanel.ControlRemoved -= OnTabRemoved;
        }

        private void OnTabInserted(object sender, NWControlEventArgs e)
        {
            Debug.Assert(e.Control is NWTabControl.NWTab, "unexpected error.");
            e.Control.MouseClick += OnTabMouseClicked;
        }

        private void OnTabRemoved(object sender, NWControlEventArgs e)
        {
            Debug.Assert(e.Control is NWTabControl.NWTab, "unexpected error.");
            e.Control.MouseClick -= OnTabMouseClicked;
        }

        private void OnTabMouseClicked(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {

                if (null == ContextMenuStrip) { return; }

                INTabPage targetPage = (sender as NWTabControl.NWTab).Model.Page as INTabPage;
                Debug.Assert(null != targetPage, "unexpected error.");

                NTabControlCancelEventArgs menuEvent = new NTabControlCancelEventArgs(targetPage,
                                                                            TabPages.IndexOf(targetPage),
                                                                            NTabControlAction.ContextMenuShown);
                OnContextMenuShown(menuEvent);
                if (menuEvent.Cancel) { return; }

                ContextMenuStrip.Hide();
                ContextMenuStrip.Show(this, e.Location);

            }
        }

        #endregion

        #region ** コアコントロール最小化イベント

        protected virtual void OnMinimizeButtonClick(object sender, EventArgs e)
        {
            switch (TabControlStyle)
            {
                case NTabControlStyle.NormalMinimizableTopTab:
                    TabControlStyle = NTabControlStyle.MinimizedTopTab;
                    break;

                case NTabControlStyle.NormalMinimizableBottomTab:
                    TabControlStyle = NTabControlStyle.MinimizedBottomTab;
                    break;
            }
        }

        protected virtual void OnRestoreButtonClick(object sender, EventArgs e)
        {
            switch (TabControlStyle)
            {
                case NTabControlStyle.MinimizedTopTab:
                    TabControlStyle = NTabControlStyle.NormalMinimizableTopTab;
                    break;

                case NTabControlStyle.MinimizedBottomTab:
                    TabControlStyle = NTabControlStyle.NormalMinimizableBottomTab;
                    break;
            }
        }

        #endregion

        #region ** モデルイベントハンドラ

        private void OnCoreInvalidated(object sender, EventArgs e)
        {
            UpdateSurface();
        }

        private void OnCoreContextMenuShown(object sender, NWTabControlCancelEventArgs e)
        {
            OnContextMenuShown(new NTabControlCancelEventArgs(e));
        }

        private void OnCoreDragged(object sender, NWTabControlDragEventArgs e)
        {
            OnDragged(new NTabControlDragEventArgs(e));
        }

        private void OnCoreSwapped(object sender, NWTabControlSwapEventArgs e)
        {
            if (CoreModel.RaiseEventLocked) { return; }
            OnSwapped(new NTabControlSwapEventArgs(e));
        }

        private void OnCoreClosing(object sender, NWTabControlCancelEventArgs e)
        {
            if (CoreModel.RaiseEventLocked) { return; }
            OnClosing(new NTabControlCancelEventArgs(e));
        }

        private void OnCoreClosed(object sender, NWTabControlEventArgs e)
        {
            if (CoreModel.RaiseEventLocked) { return; }
            OnClosed(new NTabControlEventArgs(e));
        }

        private void OnCoreSelecting(object sender, NWTabControlCancelEventArgs e)
        {
            if (CoreModel.RaiseEventLocked) { return; }
            OnSelecting(new NTabControlCancelEventArgs(e));
        }

        private void OnCoreSelected(object sender, NWTabControlEventArgs e)
        {
            if (CoreModel.RaiseEventLocked) { return; }

            if (ContainsFocus)
            {
                CoreControl.Model.SelectedTab.Page.Select();
            }

            OnSelected(new NTabControlEventArgs(e));
        }

        private void OnCoreDeselecting(object sender, NWTabControlCancelEventArgs e)
        {
            if (CoreModel.RaiseEventLocked) { return; }
            OnDeselecting(new NTabControlCancelEventArgs(e));
        }

        private void OnCoreDeselected(object sender, NWTabControlEventArgs e)
        {
            if (CoreModel.RaiseEventLocked) { return; }
            OnDeselected(new NTabControlEventArgs(e));
        }

        private void OnCoreSelectedIndexChanged(object sender, EventArgs e)
        {
            if (CoreModel.RaiseEventLocked) { return; }
            OnSelectedIndexChanged(e);
        }

        private void OnCoreSelectedTabChanged(object sender, EventArgs e)
        {
            if (CoreModel.RaiseEventLocked) { return; }
            OnSelectedTabChanged(e);
        }

        #endregion

        #region ** 初期化イベント

        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);

            // スタイルを適用する
            _host.ApplyStyle();
        }

        #endregion

        #region ** 描画イベントハンドラのオーバーライド

        /// <summary>
        /// System.Windows.Forms.Control.Paint イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している System.Windows.Forms.PaintEventArgs。</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            if (null == _bgSurface)
            {

                ReallocateSurface();
                UpdateSurface();

                if (null == _bgSurface)
                {
                    Debug.Assert(0 == ClientRectangle.Width || 0 == ClientRectangle.Height, "unexpected error.");
                    return;
                }

            }

            _bgSurface.Render(e.Graphics);
        }

        /// <summary>
        /// コントロールの背景を描画します。
        /// </summary>
        /// <param name="e">描画するコントロールに関する情報を格納する System.Windows.Forms.PaintEventArgs。</param>
        protected override void OnPaintBackground(PaintEventArgs e)
        {
            // ちらつき軽減のため、デフォルトの処理を行わない。
            // 背景は OnPaint にて描画する
        }

        protected override void OnSystemColorsChanged(EventArgs e)
        {
            CoreControl.UpdateDrawComponent();
        }

        #endregion

        #region ** レイアウトイベントハンドラのオーバーライド

        protected override void OnLayout(LayoutEventArgs levent)
        {
            base.OnLayout(levent);
            if (levent.AffectedComponent is INTabPage) { return; }

            if (null == CoreControl) { return; }

            CoreControl.Bounds = ClientRectangle;
            CoreControl.PerformLayout();

            // 最小サイズを設定する
            MinimumSize = CoreControl.MinimumSize;

            FreeSurface();
        }

        #endregion

        #region ** コントロールの追加イベントのオーバーライド

        protected override void OnControlAdded(ControlEventArgs e)
        {
            base.OnControlAdded(e);

            INTabPage tabPage = e.Control as INTabPage;

            if (null == tabPage) { return; }
            if (TabPages.Contains(tabPage)) { return; }

            TabPages.Add(tabPage);
        }

        protected override void OnControlRemoved(ControlEventArgs e)
        {
            base.OnControlRemoved(e);

            INTabPage tabPage = e.Control as INTabPage;

            if (null == tabPage) { return; }
            if (!TabPages.Contains(tabPage)) { return; }

            TabPages.Remove(tabPage);
        }

        #endregion

        #region ** ドラッグ＆ドロップイベントハンドラのオーバーライド

        protected override void OnDragEnter(DragEventArgs drgevent)
        {
            base.OnDragEnter(drgevent);
            CoreControl.PerformDragEnter(drgevent);
        }

        protected override void OnDragOver(DragEventArgs drgevent)
        {
            base.OnDragOver(drgevent);
            CoreControl.PerformDragOver(drgevent);
        }

        protected override void OnDragDrop(DragEventArgs drgevent)
        {
            base.OnDragDrop(drgevent);
            CoreControl.PerformDragDrop(drgevent);
        }

        protected override void OnDragLeave(EventArgs e)
        {
            base.OnDragLeave(e);
            CoreControl.PerformDragLeave();
        }

        #endregion

        #endregion

        #region ** メソッド

        public void EnsureVisible(INTabPage tabPage)
        {
            if (null == tabPage) { throw new ArgumentNullException("tabPage"); }
            if (null == CoreControl) { return; }
            CoreControl.EnsureVisible(CoreModel.TabPages[tabPage]);
        }

        #region ** 描画更新

        private void FreeSurface()
        {
            if (null != _bgSurface)
            {
                _bgSurface.Dispose();
                _bgSurface = null;
            }
        }

        private void ReallocateSurface()
        {
            FreeSurface();

            if (0 >= ClientRectangle.Width || 0 >= ClientRectangle.Height) { return; }
            _bgSurface = BufferedGraphicsManager.Current.Allocate(CreateGraphics(), ClientRectangle);
        }

        private void UpdateSurface()
        {
            if (null == _bgSurface)
            {
                ReallocateSurface();
                if (null == _bgSurface) { return; }
            }

            // 背景を描画する
            _bgSurface.Graphics.FillRectangle(new SolidBrush(BackColor), ClientRectangle);

            if (null == CoreControl) { return; }

            // タブコントロールを描画する
            CoreControl.Draw(_bgSurface.Graphics);
        }

        #endregion

        #region ** スタイル操作

        private void SetTabControlStyle(NTabControlStyle newStyle)
        {
            if (newStyle == _style) { return; }
            if (!Styles.ContainsKey(newStyle)) { return; }

            if (!this.IsHandleCreated)
            {
                // 強制的にウィンドウを生成します。
                // Control.CreateControl() だと即時生成されないので、
                // Control.FromHandle() を使用しています。
                Control.FromHandle(this.Handle);
            }

            _host.Style = Styles[newStyle];
            _style = newStyle;

            _host.ApplyStyle();
            PerformLayout();

            OnTabControlStyleChanged(new EventArgs());
        }

        #endregion

        #region ** タブ操作

        protected virtual bool SelectNextTab()
        {
            CancelEventArgs e = new CancelEventArgs();

            OnSelectNextTab(e);
            if (e.Cancel) { return false; }

            CoreControl.SelectNextTab();
            return true;
        }

        protected virtual bool SelectPrevTab()
        {
            CancelEventArgs e = new CancelEventArgs();

            OnSelectPrevTab(e);
            if (e.Cancel) { return false; }

            CoreControl.SelectPrevTab();
            return true;
        }

        protected virtual bool CloseSelectedTab()
        {
            CancelEventArgs e = new CancelEventArgs();

            OnCloseSelectedTab(e);
            if (e.Cancel) { return false; }

            CoreControl.CloseSelectedTab();
            return true;
        }

        #endregion

        #region ** Control メソッドのオーバーライド

        /// <summary>
        /// コマンド キーを処理します。
        /// </summary>
        /// <param name="msg">対象ウィンドウメッセージ</param>
        /// <param name="keyData">処理するキー。</param>
        /// <returns>文字がコントロールによって処理された場合は true。それ以外の場合は false。</returns>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            bool processed = false;

            switch (keyData)
            {
                // Ctrl + Tab
                case Keys.Control | Keys.Tab:
                    processed = SelectNextTab();
                    break;

                // Ctrl + Shift + Tab
                case Keys.Control | Keys.Shift | Keys.Tab:
                    processed = SelectPrevTab();
                    break;

                // Ctrl + F4
                case Keys.Control | Keys.F4:
                    processed = CloseSelectedTab();
                    break;
            }

            return (processed) ? true : base.ProcessCmdKey(ref msg, keyData);
        }

        /// <summary>
        /// Windows メッセージを処理します。
        /// </summary>
        /// <param name="m">処理対象のメッセージ。</param>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case 0x0021:    // WM_MOUSEACTIVATE
                    CoreControl.PerformMouseActivating();
                    m.Result = (IntPtr)1;    // MA_ACTIVATE
                    return;
            }

            base.WndProc(ref m);
        }

        #endregion

        #endregion

        #region ** スタイルディクショナリ

        protected class StyleDictionary : Dictionary<NTabControlStyle, NWControlStyle> { }

        private class NTabControlStyleDictionary : StyleDictionary
        {
            public NTabControlStyleDictionary()
            {
                // 通常スタイル
                Add(NTabControlStyle.NormalTopTab, new StyleTop());
                Add(NTabControlStyle.NormalBottomTab, new StyleBottom());
                Add(NTabControlStyle.NormalLeftTab, new StyleLeft());
                Add(NTabControlStyle.NormalRightTab, new StyleRight());
                Add(NTabControlStyle.NormalMinimizableTopTab, new StyleMinimizableTop());
                Add(NTabControlStyle.NormalMinimizableBottomTab, new StyleMinimizableBottom());

                // 最小化スタイル
                Add(NTabControlStyle.MinimizedTopTab, new StyleMinimizedTop());
                Add(NTabControlStyle.MinimizedBottomTab, new StyleMinimizedBottom());
                Add(NTabControlStyle.MinimizedLeftTab, new StyleMinimizedLeft());
                Add(NTabControlStyle.MinimizedRightTab, new StyleMinimizedRight());
            }
        }

        #endregion
    }

    public interface INTabPage : NWTabControl.ITabPage { }

    public class NTabPage : Panel, INTabPage
    {
        #region ** パラメータ

        // パラメータ
        private string _text = string.Empty;
        private string _toolTipText = string.Empty;
        private Image _image = null;
        private int _imageIndex = -1;
        private string _imageKey = string.Empty;

        private bool _selectOnVisible = true;

        #endregion

        public NTabPage() : this(string.Empty) { }
        public NTabPage(string text)
        {
            if (null == text) { throw new ArgumentNullException("text"); }
            _text = text;

            base.BackColor = Color.Transparent;
        }

        #region ** プロパティ

        /// <summary>
        /// タブに表示されるテキストを取得または設定します。
        /// </summary>
        [Browsable(true)]
        public override string Text
        {
            get { return _text; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                _text = value;

                OnTextChanged(new EventArgs());
            }
        }

        /// <summary>
        /// このタブのツールヒントのテキストを取得または設定します。
        /// </summary>
        [DefaultValue("")]
        public string ToolTipText
        {
            get { return _toolTipText; }
            set { _toolTipText = value; }
        }

        /// <summary>
        /// このタブに表示されているイメージのインデックスを取得または設定します。
        /// </summary>
        [RefreshProperties(RefreshProperties.All)]
        [DefaultValue(-1)]
        [Editor("System.Windows.Forms.Design.ImageIndexEditor, System.Design, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a", typeof(UITypeEditor))]
        [TypeConverter(typeof(ImageIndexConverter))]
        public int ImageIndex
        {
            get { return _imageIndex; }
            set
            {
                if (value == _imageIndex) { return; }
                if (value < -1) { throw new System.ArgumentException(); }

                _imageIndex = value;
                _imageKey = string.Empty;

                UpdateImage();
            }
        }

        /// <summary>
        /// 関連付けられている NTabControl.ImageList で、イメージのキー アクセサを取得または設定します。
        /// </summary>
        [RefreshProperties(RefreshProperties.All)]
        [DefaultValue("")]
        [TypeConverter(typeof(ImageKeyConverter))]
        [Editor("System.Windows.Forms.Design.ImageIndexEditor, System.Design, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a", typeof(UITypeEditor))]
        public string ImageKey
        {
            get { return _imageKey; }
            set
            {
                if (value == _imageKey) { return; }

                _imageKey = value;
                _imageIndex = -1;

                UpdateImage();
            }
        }

        [DefaultValue(typeof(Color), "Transparent")]
        public override Color BackColor
        {
            get { return base.BackColor; }
            set { base.BackColor = value; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        protected virtual Image Image
        {
            get { return _image; }
            set { _image = value; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        protected bool SelectOnVisible
        {
            get { return _selectOnVisible; }
            set { _selectOnVisible = value; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        private NTabControl Owner
        {
            get { return Parent as NTabControl; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);

            if (!Visible) { return; }
            if (0 == Controls.Count) { return; }

            UpdateImage();

            if (_selectOnVisible)
            {
                (this as NWTabControl.ITabPage).Select();
            }
        }

        #endregion

        #region ** メソッド

        public new virtual void Select()
        {
            foreach (Control control in Controls)
            {
                if (!control.CanFocus) { continue; }

                control.Focus();
                break;

            }
        }

        protected virtual void Attach(NWControlHost host)
        {
            if (null == Parent)
            {
                Parent = host.Owner;
            }

            Debug.Assert(Parent == host.Owner, "unexpected error.");
        }

        protected virtual void Detach()
        {
            Parent = null;
        }

        private void UpdateImage()
        {
            if (null == Owner || null == Owner.ImageList)
            {
                _image = null;
                return;
            }

            if (0 <= _imageIndex && _imageIndex < Owner.ImageList.Images.Count)
            {
                _image = Owner.ImageList.Images[_imageIndex];
            }
            else if (string.Empty != _imageKey && Owner.ImageList.Images.ContainsKey(_imageKey))
            {
                _image = Owner.ImageList.Images[_imageKey];
            }
            else
            {
                _image = null;
            }

            OnImageChanged(new EventArgs());
        }

        #endregion

        #region ** INTabPage の実装

        #region ** プロパティ

        /// <summary>
        /// タブコントロールに関連付けれているかどうかを取得します。
        /// </summary>
        bool NWTabControl.ITabPage.Attached
        {
            get { return (null != Parent); }
        }

        /// <summary>
        /// イメージを取得または設定します。
        /// </summary>
        Image NWTabControl.ITabPage.Image
        {
            get { return Image; }
        }

        #endregion

        #region ** イベント

        event EventHandler NWTabControl.ITabPage.TextChanged
        {
            add { TextChanged += value; }
            remove { TextChanged -= value; }
        }

        event EventHandler NWTabControl.ITabPage.ImageChanged
        {
            add { ImageChanged += value; }
            remove { ImageChanged -= value; }
        }

        private new EventHandler TextChanged;
        private EventHandler ImageChanged;

        #endregion

        #region ** イベントハンドラ

        protected new virtual void OnTextChanged(EventArgs e)
        {
            if (null != TextChanged)
            {
                TextChanged(this, e);
            }
        }

        protected virtual void OnImageChanged(EventArgs e)
        {
            if (null != ImageChanged)
            {
                ImageChanged(this, e);
            }
        }

        #endregion

        #region ** メソッド

        void NWTabControl.ITabPage.Attach(NWControlHost host)
        {
            Attach(host);
        }

        void NWTabControl.ITabPage.Detach()
        {
            Detach();
        }

        void NWTabControl.ITabPage.Show()
        {
            if (NTabControlStyle.MinimizedStyleBegin <= Owner.TabControlStyle &&
                Owner.TabControlStyle <= NTabControlStyle.MinimizedStyleEnd)
            {
                return;
            }

            Show();
        }

        #endregion

        #endregion
    }

    #region ** 列挙値

    /// <summary>
    /// タブ コントロールのスタイルを指定します。
    /// </summary>
    public enum NTabControlStyle
    {
        // 通常スタイル
        NormalStyleBegin = 1,
        NormalTopTab,
        NormalBottomTab,
        NormalLeftTab,
        NormalRightTab,
        NormalMinimizableTopTab,
        NormalMinimizableBottomTab,
        NormalStyleEnd,

        // 最小化スタイル
        MinimizedStyleBegin = 100,
        MinimizedTopTab,
        MinimizedBottomTab,
        MinimizedLeftTab,
        MinimizedRightTab,
        MinimizedStyleEnd,
    }

    /// <summary>
    /// NTabControl イベントを表す値を定義します。
    /// </summary>
    public enum NTabControlAction
    {
        Swapped = NWTabControl.Action.Swapped,                      // NTabControl.Swapped イベントを表します。
        Closing = NWTabControl.Action.Closing,                      // NTabControl.Closing イベントを表します。
        Closed = NWTabControl.Action.Closed,                        // NTabControl.Closed イベントを表します。
        Selecting = NWTabControl.Action.Selecting,                  // NTabControl.Selecting イベントを表します。
        Selected = NWTabControl.Action.Selected,                    // NTabControl.Selected イベントを表します。
        Deselecting = NWTabControl.Action.Deselecting,              // NTabControl.Deselecting イベントを表します。
        Deselected = NWTabControl.Action.Deselected,                // NTabControl.Deselected イベントを表します。
        ContextMenuShown = NWTabControl.Action.ContextMenuShown,    // NTabControl.ContextMenuShown イベントを表します。
    }

    #endregion

    #region ** イベントクラスとデリゲート

    #region ** タブ イベント

    public delegate void NTabControlEventHandler(object sender, NTabControlEventArgs e);
    public delegate void NTabControlCancelEventHandler(object sender, NTabControlCancelEventArgs e);
    public delegate void NTabControlSwapEventHandler(object sender, NTabControlSwapEventArgs e);

    /// <summary>
    /// NTabControl イベントのデータを提供します。
    /// </summary>
    public class NTabControlEventArgs : EventArgs
    {
        #region ** パラメータ

        private NWTabControlEventArgs _orgArgs = null;

        private INTabPage _tabPage = null;
        private int _tabPageIndex = NWTabControl.InvalidIndex;
        private NTabControlAction _action = NTabControlAction.Selected;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="orgArgs">NWTabControl イベント</param>
        /// <param name="tabPageIndex">イベントの処理対象となるタブページのインデックス</param>
        public NTabControlEventArgs(NWTabControlEventArgs orgArgs)
        {
            Debug.Assert(null != orgArgs, "unexpected error.");
            _orgArgs = orgArgs;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="tabPage">イベントの処理対象となるタブページ</param>
        /// <param name="tabPageIndex">イベントの処理対象となるタブページのインデックス</param>
        /// <param name="action">アクションイベントの種類</param>
        public NTabControlEventArgs(INTabPage tabPage, int tabPageIndex, NTabControlAction action)
        {
            _tabPage = tabPage;
            _tabPageIndex = tabPageIndex;
            _action = action;
        }

        #region ** プロパティ

        /// <summary>
        /// 発生しているイベントを示す値を取得します。
        /// </summary>
        public NTabControlAction Action
        {
            get
            {
                if (null != _orgArgs) { return (NTabControlAction)_orgArgs.Action; }
                return _action;
            }
        }

        /// <summary>
        /// 発生しているイベントの対象となる System.Windows.Forms.TabPage を取得します。
        /// </summary>
        public INTabPage TabPage
        {
            get
            {
                if (null != _orgArgs) { return _orgArgs.Tab.Page as INTabPage; }
                return _tabPage;
            }
        }

        /// <summary>
        /// 対象タブのインデックス
        /// </summary>
        public int TabPageIndex
        {
            get
            {
                if (null != _orgArgs) { return _orgArgs.TabIndex; }
                return _tabPageIndex;
            }
        }

        #endregion

    }

    /// <summary>
    /// キャンセル可能な NTabControl イベントのデータを提供します。
    /// </summary>
    public class NTabControlCancelEventArgs : EventArgs
    {
        #region ** パラメータ

        private NWTabControlCancelEventArgs _orgArgs = null;

        private INTabPage _tabPage = null;
        private int _tabPageIndex = NWTabControl.InvalidIndex;
        private NTabControlAction _action = NTabControlAction.Selected;
        private bool _cancel = false;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="orgArgs">NWTabControl イベント</param>
        /// <param name="tabPageIndex">イベントの処理対象となるタブページのインデックス</param>
        public NTabControlCancelEventArgs(NWTabControlCancelEventArgs orgArgs)
        {
            Debug.Assert(null != orgArgs, "unexpected error.");
            _orgArgs = orgArgs;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="tabPage">イベントの処理対象となるタブページ</param>
        /// <param name="tabPageIndex">イベントの処理対象となるタブページのインデックス</param>
        /// <param name="action">アクションイベントの種類</param>
        public NTabControlCancelEventArgs(INTabPage tabPage, int tabPageIndex, NTabControlAction action)
        {
            _tabPage = tabPage;
            _tabPageIndex = tabPageIndex;
            _action = action;
        }

        #region ** プロパティ

        /// <summary>
        /// 発生しているイベントを示す値を取得します。
        /// </summary>
        public NTabControlAction Action
        {
            get
            {
                if (null != _orgArgs) { return (NTabControlAction)_orgArgs.Action; }
                return _action;
            }
        }

        /// <summary>
        /// 発生しているイベントの対象となる INTabPage を取得します。
        /// </summary>
        public INTabPage TabPage
        {
            get
            {
                if (null != _orgArgs) { return _orgArgs.Tab.Page as INTabPage; }
                return _tabPage;
            }
        }

        /// <summary>
        /// 対象タブのインデックス
        /// </summary>
        public int TabPageIndex
        {
            get
            {
                if (null != _orgArgs) { return _orgArgs.TabIndex; }
                return _tabPageIndex;
            }
        }

        public bool Cancel
        {
            get
            {
                if (null != _orgArgs) { return _orgArgs.Cancel; }
                return _cancel;
            }
            set
            {
                _cancel = value;

                if (null != _orgArgs)
                {
                    _orgArgs.Cancel = value;
                }
            }
        }

        #endregion
    }

    /// <summary>
    /// タブの順番入れ替えイベントのデータを提供します。
    /// </summary>
    public class NTabControlSwapEventArgs : EventArgs
    {
        #region ** パラメータ

        private NWTabControlSwapEventArgs _orgArgs = null;

        private INTabPage _tabPage = null;
        private int _tabPageIndex = NWTabControl.InvalidIndex;
        private INTabPage _tabPage2 = null;
        private int _tabPage2Index = NWTabControl.InvalidIndex;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="orgArgs">NWTabControl イベント</param>
        /// <param name="tabPageIndex">イベントの処理対象となるタブページのインデックス</param>
        public NTabControlSwapEventArgs(NWTabControlSwapEventArgs orgArgs)
        {
            Debug.Assert(null != orgArgs, "unexpected error.");
            _orgArgs = orgArgs;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="tabPage">イベントの処理対象となるタブページ1</param>
        /// <param name="tabPageIndex">イベントの処理対象となるタブページ1のインデックス</param>
        /// <param name="tabPage2">イベントの処理対象となるタブページ2</param>
        /// <param name="tabPage2Index">イベントの処理対象となるタブページ2のインデックス</param>
        public NTabControlSwapEventArgs(INTabPage tabPage1, int tabPage1Index, INTabPage tabPage2, int tabPage2Index)
        {
            _tabPage = tabPage1;
            _tabPageIndex = tabPage1Index;
            _tabPage2 = tabPage2;
            _tabPage2Index = tabPage2Index;
        }

        #region ** プロパティ

        /// <summary>
        /// 発生しているイベントを示す値を取得します。
        /// </summary>
        public NTabControlAction Action
        {
            get
            {
                if (null != _orgArgs) { return (NTabControlAction)_orgArgs.Action; }
                return NTabControlAction.Swapped;
            }
        }

        /// <summary>
        /// 発生しているイベントの対象となるタブ1を取得します。
        /// </summary>
        public INTabPage TabPage
        {
            get
            {
                if (null != _orgArgs) { return _orgArgs.Tab.Page as INTabPage; }
                return _tabPage;
            }
        }

        /// <summary>
        /// 対象タブ1のインデックス
        /// </summary>
        public int TabPageIndex
        {
            get
            {
                if (null != _orgArgs) { return _orgArgs.TabIndex; }
                return _tabPageIndex;
            }
        }

        /// <summary>
        /// 発生しているイベントの対象となるタブ2を取得します。
        /// </summary>
        public INTabPage TabPage2
        {
            get
            {
                if (null != _orgArgs) { return _orgArgs.Tab2.Page as INTabPage; }
                return _tabPage2;
            }
        }

        /// <summary>
        /// 対象タブ2のインデックス
        /// </summary>
        public int TabPage2Index
        {
            get
            {
                if (null != _orgArgs) { return _orgArgs.Tab2Index; }
                return _tabPage2Index;
            }
        }

        #endregion

    }

    #endregion

    #region ** ドラッグ＆ドロップ イベント

    public delegate void NTabControlDragEventHandler(object sender, NTabControlDragEventArgs e);

    public class NTabControlDragEventArgs : EventArgs
    {
        #region ** パラメータ

        private NWTabControlDragEventArgs _orgArgs = null;

        #endregion

        public NTabControlDragEventArgs(NWTabControlDragEventArgs orgArgs)
        {
            _orgArgs = orgArgs;
        }

        #region ** プロパティ

        public INTabPage TabPage
        {
            get { return _orgArgs.Tab.Page as INTabPage; }
            set { _orgArgs.Tab.Page = value; }
        }

        /// <summary>
        /// ドラッグ イベントの元の場所 (またはソース) で実行できるドラッグ アンド ドロップ操作を取得します。
        /// </summary>
        public DragDropEffects AllowedEffect
        {
            get { return _orgArgs.AllowedEffect; }
        }

        /// <summary>
        /// このイベントに関連付けられているデータを含む System.Windows.Forms.IDataObject を取得します。
        /// </summary>
        public IDataObject Data
        {
            get { return _orgArgs.Data; }
        }

        /// <summary>
        /// ドラッグ アンド ドロップ操作のターゲットのドロップ効果を取得または設定します。
        /// </summary>
        public DragDropEffects Effect
        {
            get { return _orgArgs.Effect; }
            set { _orgArgs.Effect = value; }
        }

        /// <summary>
        /// マウス ボタンの状態と同様に、Shift、Ctrl、Alt の各キーの現在の状態を取得します。
        /// </summary>
        public int KeyState
        {
            get { return _orgArgs.KeyState; }
        }

        /// <summary>
        /// マウス ポインタの x 座標 (画面座標) を取得します。
        /// </summary>
        public int X
        {
            get { return _orgArgs.X; }
        }

        /// <summary>
        /// マウス ポインタの y 座標 (画面座標) を取得します。
        /// </summary>
        public int Y
        {
            get { return _orgArgs.Y; }
        }

        #endregion
    }

    #endregion

    #endregion

    #region ** コレクション

    public class NTabPageCollection : IList<INTabPage>, ICollection<INTabPage>, IEnumerable<INTabPage>,
                                        IList, ICollection, IEnumerable
    {
        private NWTabControl.TabCollection _tabs = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="tabs">タブコレクション</param>
        public NTabPageCollection(NWTabControl.TabCollection tabs)
        {
            _tabs = tabs;
        }

        /// <summary>
        /// タブページ数を取得します。
        /// </summary>
        public int Count
        {
            get { return _tabs.Count; }
        }

        /// <summary>
        /// インデクサ
        /// </summary>
        /// <param name="index">取得または設定するタブページのインデックス。</param>
        /// <returns>タブページ</returns>
        public INTabPage this[int index]
        {
            get { return _tabs[index].Page as INTabPage; }
            set { _tabs[index].Page = value; }
        }

        // 概要:
        //     System.Collections.ObjectModel.Collection<T> の末尾にオブジェクトを追加します。
        //
        // パラメータ:
        //   item:
        //     System.Collections.ObjectModel.Collection<T> の末尾に追加するオブジェクト。参照型の場合、null の値を使用できます。
        public void Add(INTabPage item)
        {
            _tabs.AddPage(item);
        }

        //
        // 概要:
        //     System.Collections.ObjectModel.Collection<T> からすべての要素を削除します。
        public void Clear()
        {
            _tabs.Clear();
        }

        /// <summary>
        /// 指定されたタブページがコントロール内に存在するかどうかを調べます。
        /// </summary>
        /// <param name="item">タブページ</param>
        /// <returns>存在する場合は true, 存在しない場合は false。</returns>
        public bool Contains(INTabPage item)
        {
            return _tabs.Contains(item);
        }

        /// <summary>
        /// Arrayの指定インデックス以降にコピーします。
        /// </summary>
        /// <param name="array">コピー先の1次元Array。</param>
        /// <param name="index">コピーの開始位置となる、array の 0 から始まるインデックス。</param>
        public void CopyTo(INTabPage[] array, int index)
        {
            (this as ICollection).CopyTo(array, index);
        }

        /// <summary>
        /// このコレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>列挙子</returns>
        public IEnumerator<INTabPage> GetEnumerator()
        {
            return new Enumerator(_tabs);
        }

        /// <summary>
        /// 最初に見つかった指定タブページのインデックスを返します。
        /// </summary>
        /// <param name="item"></param>
        /// <returns></returns>
        public int IndexOf(INTabPage item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            return _tabs.IndexOf(item);
        }

        /// <summary>
        /// 指定したインデックスの位置にタブページを挿入します。
        /// </summary>
        /// <param name="index">タブページを挿入する位置の、0 から始まるインデックス。</param>
        /// <param name="item">挿入するタブページ。</param>
        public void Insert(int index, INTabPage item)
        {
            _tabs.InsertPage(index, item);
        }

        //
        /// <summary>
        /// 最初に見つかった指定タブページを削除します。
        /// </summary>
        /// <param name="item">タブページ</param>
        /// <returns>タブページが正常に削除された場合は true。それ以外の場合は false。</returns>
        public bool Remove(INTabPage item)
        {
            return _tabs.RemovePage(item);
        }

        /// <summary>
        /// 指定したインデックスにあるタブページを削除します。
        /// </summary>
        /// <param name="index">インデックス。</param>
        public void RemoveAt(int index)
        {
            _tabs.RemoveAt(index);
        }

        #region ** IList の実装

        bool IList.IsFixedSize
        {
            get { return false; }
        }

        bool IList.IsReadOnly
        {
            get { return false; }
        }

        object IList.this[int index]
        {
            get { return this[index]; }
            set { this[index] = value as INTabPage; }
        }

        int IList.Add(object value)
        {
            Add(value as INTabPage);
            return Count;
        }

        bool IList.Contains(object value)
        {
            return Contains(value as INTabPage);
        }

        int IList.IndexOf(object value)
        {
            return IndexOf(value as INTabPage);
        }

        void IList.Insert(int index, object value)
        {
            Insert(index, value as INTabPage);
        }

        void IList.Remove(object value)
        {
            Remove(value as INTabPage);
        }

        #endregion

        #region ** ICollection の実装

        bool ICollection.IsSynchronized
        {
            get { return false; }
        }

        object ICollection.SyncRoot
        {
            get { return null; }
        }

        bool ICollection<INTabPage>.IsReadOnly
        {
            get { return false; }
        }

        void ICollection.CopyTo(Array array, int arrayIndex)
        {
            if (null == array) { throw new ArgumentNullException(); }
            if (array.Length < _tabs.Count) { throw new ArgumentOutOfRangeException(); }
            if (arrayIndex + array.Length < _tabs.Count) { throw new ArgumentException(); }

            int i = arrayIndex;

            foreach (NWTabControl.ITabPage tabPage in _tabs)
            {
                array.SetValue(tabPage, i);
                i++;
            }
        }

        #endregion

        #region ** IEnumerable の実装

        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }

        #endregion

        #region ** Enumerator

        private class Enumerator : IEnumerator<INTabPage>
        {
            private NWTabControl.Tab[] _array = null;
            private int _current = -1;

            public Enumerator(NWTabControl.TabCollection tabs)
            {
                if (null == tabs)
                {
                    Debug.Fail("ArgumentNullException.");
                    return;
                }

                _array = new NWTabControl.Tab[tabs.Count];
                tabs.CopyTo(_array, 0);
            }

            public INTabPage Current
            {
                get
                {
                    if (null == _array) { return null; }
                    if (-1 == _current) { return null; }
                    if (_array.Length <= _current) { return null; }
                    return _array[_current].Page as INTabPage;
                }
            }

            public bool MoveNext()
            {
                if (null == _array) { return false; }
                if (0 == _array.Length) { return false; }
                if (_array.Length <= _current + 1) { return false; }

                _current++;
                return true;
            }

            public void Reset()
            {
                _current = -1;
            }

            public void Dispose()
            {
                _current = -1;
                _array = null;
            }

            #region ** IEnumerator の実装

            object IEnumerator.Current
            {
                get { return Current; }
            }

            #endregion
        }

        #endregion
    }

    #endregion
}
