﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Security.Cryptography;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Core.Parameters;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    ///--------------------------------------------------------------------------
    /// <summary>
    /// スクロールコントロールクラス
    /// </summary>
    public class FlexibleScrollableControl : UserControl
    {

        private HScrollBar _HorizontalScrollBar;
        private VScrollBar _VerticalScrollBar;
        private Panel _Panel = new Panel();

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FlexibleScrollableControl()
        {
            _HorizontalScrollBar = new HScrollBar();
            _VerticalScrollBar = new VScrollBar();

            SuspendLayout();

            _HorizontalScrollBar.Parent = this;
            _HorizontalScrollBar.Anchor = AnchorStyles.None;
            _HorizontalScrollBar.Location = new Point(16, 0);
            _HorizontalScrollBar.Name = "HorizontalScrollBar";
            _HorizontalScrollBar.TabIndex = 0;
            _HorizontalScrollBar.TabStop = false;
            _HorizontalScrollBar.Scroll += OnHorizontalScroll;
            _HorizontalScrollBar.ValueChanged += OnHorizontalScrollValueChanged;

            _VerticalScrollBar.Parent = this;
            _VerticalScrollBar.Anchor = AnchorStyles.None;
            _VerticalScrollBar.Location = new Point(0, 16);
            _VerticalScrollBar.Name = "VerticalScrollBar";
            _VerticalScrollBar.TabIndex = 1;
            _VerticalScrollBar.TabStop = false;
            _VerticalScrollBar.Scroll += OnVerticalScroll;
            _VerticalScrollBar.ValueChanged += OnVerticalScrollValueChanged;

            _Panel.BackColor = SystemColors.Control;
            _Panel.Parent = this;

            Resize += OnResize;
            ResumeLayout(false);

            _Panel.Focus();

            UpdateScrollBar();
            UpdatePanel();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public System.Windows.Forms.HScrollBar HorizontalScrollBar
        {
            get { return _HorizontalScrollBar; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public System.Windows.Forms.VScrollBar VerticalScrollBar
        {
            get { return _VerticalScrollBar; }
        }

        ///--------------------------------
        /// <summary>
        /// スクロールバーの除いたクライアント領域のサイズの取得
        /// </summary>
        public virtual Size ClientSizeWithoutScrollBar
        {
            get
            {
                int width = base.ClientSize.Width -
                    (VerticalScrollBar.Visible != false ?
                      SystemInformation.VerticalScrollBarWidth : 0);

                int height = base.ClientSize.Height -
                    (HorizontalScrollBar.Visible != false ?
                      SystemInformation.HorizontalScrollBarHeight : 0);

                if (width < 0) { width = 0; }
                if (height < 0) { height = 0; }

                return new Size(width, height);
            }
        }

        ///--------------------------------
        /// <summary>
        /// スクロールバーの除いたクライアント領域の取得
        /// </summary>
        public virtual Rectangle ClientRectangleWithoutScrollBar
        {
            get
            {
                Size size = ClientSizeWithoutScrollBar;
                return new Rectangle(0, 0, size.Width, size.Height);
            }
        }

        ///--------------------------------
        /// <summary>
        /// スクロールバーの更新
        /// </summary>
        public virtual void UpdateScrollBar()
        {
            int width = ClientSize.Width;
            int height = ClientSize.Height;

            //
            if (HorizontalScrollBar.Visible != false)
            {
                if ((width -= VerticalScrollBar.Width) < 0)
                {
                    width = 0;
                }
            }

            if (VerticalScrollBar.Visible != false)
            {
                if ((height -= HorizontalScrollBar.Height) < 0)
                {
                    height = 0;
                }
            }

            int HorizontalScrollBarWidth = ClientSize.Width;
            int HorizontalScrollBarHeight = SystemInformation.HorizontalScrollBarHeight;
            int VerticalScrollBarWidth = SystemInformation.VerticalScrollBarWidth;
            int VerticalScrollBarHeight = ClientSize.Height;

            if (HorizontalScrollBar.Visible != false &&
                VerticalScrollBar.Visible != false)
            {
                HorizontalScrollBarWidth -= HorizontalScrollBarHeight;
                VerticalScrollBarHeight -= VerticalScrollBarWidth;
            }

            HorizontalScrollBar.Location = new Point
                (Padding.Left, ClientSize.Height - HorizontalScrollBarHeight - Padding.Bottom);
            VerticalScrollBar.Location = new Point
                (ClientSize.Width - VerticalScrollBarWidth - Padding.Right, Padding.Top);

            VerticalScrollBar.Size = new Size
                (VerticalScrollBarWidth, VerticalScrollBarHeight - Padding.Vertical);
            HorizontalScrollBar.Size = new Size
                (HorizontalScrollBarWidth - Padding.Horizontal, HorizontalScrollBarHeight);

            UpdatePanel();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnHorizontalScrolled(ScrollEventArgs e)
        {
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnHorizontalScrollValueChanged()
        {
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnVerticalScrolled(ScrollEventArgs e)
        {
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnVerticalScrollValueChanged()
        {
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        /// 横スクロール処理
        /// </summary>
        protected virtual void OnHorizontalScrollValueChanged(object sender, EventArgs e)
        {
            OnHorizontalScrollValueChanged();
        }

        ///--------------------------------
        /// <summary>
        /// 縦スクロール処理
        /// </summary>
        protected virtual void OnVerticalScrollValueChanged(object sender, EventArgs e)
        {
            OnVerticalScrollValueChanged();
        }

        ///--------------------------------
        /// <summary>
        /// 横スクロール処理
        /// </summary>
        protected virtual void OnHorizontalScroll(object sender, ScrollEventArgs e)
        {
            switch (e.Type)
            {
                case ScrollEventType.ThumbPosition:
                case ScrollEventType.EndScroll:
                    break;

                case ScrollEventType.First:
                case ScrollEventType.Last:
                case ScrollEventType.LargeDecrement:
                case ScrollEventType.LargeIncrement:
                case ScrollEventType.SmallDecrement:
                case ScrollEventType.SmallIncrement:
                case ScrollEventType.ThumbTrack:
                    OnHorizontalScrolled(e);
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        /// 縦スクロール処理
        /// </summary>
        protected virtual void OnVerticalScroll(object sender, ScrollEventArgs e)
        {
            switch (e.Type)
            {
                case ScrollEventType.ThumbPosition:
                case ScrollEventType.EndScroll:
                    break;

                case ScrollEventType.First:
                case ScrollEventType.Last:
                case ScrollEventType.LargeDecrement:
                case ScrollEventType.LargeIncrement:
                case ScrollEventType.SmallDecrement:
                case ScrollEventType.SmallIncrement:
                case ScrollEventType.ThumbTrack:
                    OnVerticalScrolled(e);
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool ProcessCmdKey(ref Message m, Keys keyData)
        {
            if (Focused != false ||
                HorizontalScrollBar.Focused != false ||
                VerticalScrollBar.Focused != false)
            {

                long actualKey = (int)keyData & 0x0000FFFF;
                KeyEventArgs args = null;

                if (actualKey == (long)Keys.Up ||
                    actualKey == (long)Keys.Down ||
                    actualKey == (long)Keys.Left ||
                    actualKey == (long)Keys.Right ||
                    actualKey == (long)Keys.Tab)
                {

                    args = new KeyEventArgs(keyData);

                    if (actualKey == (long)Keys.Tab && args.Control != false)
                    {
                    }
                    else
                    {

                        OnKeyDown(args);
                        return true;
                    }
                }
            }

            return base.ProcessCmdKey(ref m, keyData);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdatePanel()
        {
            int x = 0;
            int y = 0;
            int width = 0;
            int height = 0;

            if (HorizontalScrollBar.Visible == false &&
                VerticalScrollBar.Visible == false)
            {
                _Panel.Hide();
            }
            else
            {

                width = SystemInformation.VerticalScrollBarWidth;
                height = SystemInformation.HorizontalScrollBarHeight;
                x = ClientSize.Width - width;
                y = ClientSize.Height - height;

                _Panel.Location = new Point(x, y);
                _Panel.Size = new Size(width, height);
                _Panel.Show();
            }
        }

        ///--------------------------------
        /// <summary>
        /// サイズの変更
        /// </summary>
        private void OnResize(object sender, System.EventArgs e)
        {
            UpdateScrollBar();
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class OperatableControl : FlexibleScrollableControl
    {
        private MouseButtons _LastUpButton = MouseButtons.None;
        private bool _EnabledEventLoop = false;
        private Timer _Timer = null;

        private Timer _AnimationTimer = null;
        private int _AnimationCounter = 0;

        private string[] _DropFilePaths = null;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public OperatableControl()
        {
            base.BackColor = SystemColors.Window;

            SetStyle(ControlStyles.AllPaintingInWmPaint
                      | ControlStyles.DoubleBuffer, true);

            //
            _AnimationTimer = new Timer();
            _AnimationTimer.Tick += OnAnimationTimer;
            _AnimationTimer.Interval = 500;
            _AnimationTimer.Start();

            AllowDrop = true;
            //AutoDropFiles = true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        //public virtual bool AutoDropFiles { get; set; }

        ///--------------------------------
        /// <summary>
        /// イベントループを開始する
        /// </summary>
        public void BeginEventLoop()
        {
            Point clientPoint;

            _EnabledEventLoop = true;
            while (_EnabledEventLoop != false)
            {

                clientPoint = PointToClient(MousePosition);
                CurrentOperator.OnEventLoop
                    (new MouseEventArgs(_LastUpButton,
                                          0, clientPoint.X, clientPoint.Y, 0));

                System.Threading.Thread.Sleep(1);
                System.Windows.Forms.Application.DoEvents();
            }
        }

        ///--------------------------------
        /// <summary>
        /// イベントループを終了する
        /// </summary>
        public void EndEventLoop()
        {
            _EnabledEventLoop = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int AnimationCounter
        {
            get { return _AnimationCounter; }
        }

        ///--------------------------------
        /// <summary>
        /// 描画
        /// </summary>
        protected virtual void Draw(Graphics gc)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual bool CanDropFiles(string[] filePaths, object dropTarget)
        {
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual bool DropFiles(string[] filePaths, object dropTarget)
        {
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected OperatableControlOperator CurrentOperator
        {
            get; set;
        }

        ///--------------------------------
        /// <summary>
        /// 背景の描画
        /// </summary>
        protected override void OnPaintBackground(PaintEventArgs e)
        {
            base.OnPaintBackground(e);
        }

        ///--------------------------------
        /// <summary>
        /// 描画
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);
            try
            {
                Draw(e.Graphics);
            }
            catch (Exception)
            {
            }
        }

        ///--------------------------------
        /// <summary>
        /// タイマーの起動
        /// </summary>
        protected virtual void StartTimer()
        {
            StartTimer(1000);
        }

        ///--------------------------------
        /// <summary>
        /// タイマーの起動
        /// </summary>
        protected virtual void StartTimer(int interval)
        {
            if (_Timer == null)
            {
                _Timer = new Timer();
                _Timer.Tick += OnTimer;
            }

            _Timer.Interval = interval;
            _Timer.Start();
        }

        ///--------------------------------
        /// <summary>
        /// タイマーの再起動
        /// </summary>
        protected virtual void RestartTimer()
        {
            StopTimer();
            StartTimer();
        }

        ///--------------------------------
        /// <summary>
        /// タイマーの停止
        /// </summary>
        protected virtual void StopTimer()
        {
            _Timer.Stop();
        }

        ///--------------------------------
        /// <summary>
        /// マウスボタンの押下処理
        /// </summary>
        protected override void OnMouseDown(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseDown(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnMouseDown(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// マウスボタンの押上処理
        /// </summary>
        protected override void OnMouseUp(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseUp(e);

            _LastUpButton = e.Button;
            if (CurrentOperator != null)
            {
                CurrentOperator.OnMouseUp(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// マウスボタンのダブルクリック処理
        /// </summary>
        protected override void OnDoubleClick(System.EventArgs e)
        {
            base.OnDoubleClick(e);

            if (CurrentOperator != null)
            {
                Point clientPoint = PointToClient(MousePosition);

                CurrentOperator.OnDoubleClick
                    (new MouseEventArgs(_LastUpButton,
                                          0, clientPoint.X, clientPoint.Y, 0));
            }
        }

        ///--------------------------------
        /// <summary>
        /// マウスの移動処理
        /// </summary>
        protected override void OnMouseMove(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseMove(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnMouseMove(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// マウスのホイール処理
        /// </summary>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnMouseWheel(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// キーが押された処理
        /// </summary>
        protected override void OnKeyDown(System.Windows.Forms.KeyEventArgs e)
        {
            base.OnKeyDown(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnKeyDown(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// キーが離された処理
        /// </summary>
        protected override void OnKeyUp(System.Windows.Forms.KeyEventArgs e)
        {
            base.OnKeyUp(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnKeyUp(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// 文字が入力された処理
        /// </summary>
        protected override void OnKeyPress(System.Windows.Forms.KeyPressEventArgs e)
        {
            base.OnKeyPress(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnKeyPress(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// ドロップ処理
        /// </summary>
        protected override void OnDragDrop(System.Windows.Forms.DragEventArgs e)
        {
            base.OnDragDrop(e);

            //
            if (CurrentOperator != null)
            {
                CurrentOperator.OnDragDrop(e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragEnter(System.Windows.Forms.DragEventArgs e)
        {
            base.OnDragEnter(e);

            _DropFilePaths = null;

            //
            if (e.Data.GetDataPresent(DataFormats.FileDrop) != false)
            {
                string[] filePaths = (string[])e.Data.GetData(DataFormats.FileDrop);
                Array.Sort(filePaths);

                if (CanDropFiles(filePaths, null) != false)
                {
                    _DropFilePaths = filePaths;
                }
            }

            //
            if (CurrentOperator != null)
            {
                CurrentOperator.OnDragEnter(e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragLeave(System.EventArgs e)
        {
            base.OnDragLeave(e);

            _DropFilePaths = null;

            if (CurrentOperator != null)
            {
                CurrentOperator.OnDragLeave(e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragOver(System.Windows.Forms.DragEventArgs e)
        {
            base.OnDragOver(e);

            if (e.Data.GetDataPresent(DataFormats.FileDrop) != false)
            {
                e.Effect = _DropFilePaths != null ?
                    DragDropEffects.Copy : DragDropEffects.None;
            }

            if (CurrentOperator != null)
            {
                CurrentOperator.OnDragOver(e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnGiveFeedback(System.Windows.Forms.GiveFeedbackEventArgs e)
        {
            base.OnGiveFeedback(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnGiveFeedback(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// フォーカスを失った時の処理
        /// </summary>
        protected override void OnLostFocus(System.EventArgs e)
        {
            base.OnLostFocus(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnLostFocus(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// タイマー処理
        /// </summary>
        private void OnTimer(object sender, EventArgs e)
        {
            if (CurrentOperator != null)
            {
                CurrentOperator.OnTimer(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// タイマー処理
        /// </summary>
        private void OnAnimationTimer(object sender, EventArgs e)
        {
            if (CurrentOperator != null)
            {
                CurrentOperator.OnAnimation(e);
            }
            _AnimationCounter++;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class OperatableControlOperator
    {

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public OperatableControlOperator(OperatableControl control)
        {
            OperatableControl = control;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected internal virtual void Initialize(Point beginPoint) { }

        protected internal virtual bool OnMouseDown(MouseEventArgs e)
        {
            return true;
        }
        protected internal virtual bool OnMouseUp(MouseEventArgs e)
        {
            return true;
        }
        protected internal virtual bool OnMouseMove(MouseEventArgs e)
        {
            return true;
        }
        protected internal virtual bool OnMouseWheel(MouseEventArgs e)
        {
            return true;
        }
        protected internal virtual void OnDoubleClick(MouseEventArgs e) { }

        protected internal virtual void OnKeyDown(KeyEventArgs e) { }
        protected internal virtual void OnKeyUp(KeyEventArgs e) { }
        protected internal virtual void OnKeyPress(KeyPressEventArgs e) { }

        protected internal virtual void OnDragDrop(DragEventArgs e) { }
        protected internal virtual void OnDragOver(DragEventArgs e) { }
        protected internal virtual void OnDragEnter(DragEventArgs e) { }
        protected internal virtual void OnDragLeave(EventArgs e) { }
        protected internal virtual void OnEventLoop(MouseEventArgs e) { }

        protected internal virtual void OnGiveFeedback(GiveFeedbackEventArgs e) { }

        protected internal virtual bool OnLostFocus(EventArgs e)
        {
            return true;
        }

        protected internal virtual void OnTimer(EventArgs e) { }
        protected internal virtual void OnAnimation(EventArgs e) { }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected OperatableControl OperatableControl
        {
            get; set;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected OperatableControl OwnerControl
        {
            get { return OperatableControl; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Point ToClientPoint(int x, int y)
        {
            return OperatableControl.PointToClient(new Point(x, y));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Point ToClientPoint(Point point)
        {
            return OperatableControl.PointToClient(point);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private OperatableControlOperator()
        {
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class DroppedEventArgs : CancelEventArgs
    {
        private object _DropTarget = null;
        private DropPosition _DropPosition = DropPosition.None;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DroppedEventArgs(object dropTarget, DropPosition dropPosition)
        {
            _DropTarget = dropTarget;
            _DropPosition = dropPosition;
        }

        ///
        public DroppedEventArgs(object dropTarget)
        {
            _DropTarget = dropTarget;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public object DropTarget
        {
            get { return _DropTarget; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DropPosition DropPosition
        {
            get { return _DropPosition; }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class FileDroppedEventArgs : DroppedEventArgs
    {
        private string[] _FilePaths = null;
        //private object          _DropTarget = null;
        //private DropPosition    _DropPosition = DropPosition.None;

        public FileDroppedEventArgs(string[] filePaths, object dropTarget, DropPosition dropPosition) : base(dropTarget, dropPosition)
        {
            _FilePaths = filePaths;
            //_DropTarget = dropTarget;
            //_DropPosition = dropPosition;
        }

        public FileDroppedEventArgs(string[] filePaths, object dropTarget) : base(dropTarget)
        {
            _FilePaths = filePaths;
            //_DropTarget = dropTarget;
        }

        ///
        public string[] FilePaths
        {
            get { return _FilePaths; }
        }
    }

    ///------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class KeyDictionary : Dictionary<Keys, bool>
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Add(Keys key)
        {
            Add(key, true);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool Contains(Keys key)
        {
            return ContainsKey(key);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public enum CaretMove
    {
        Up,
        Down,
        Left,
        Right,

        LeftWithoutNewLine,
        RightWithoutNewLine,

        MostUp,
        MostDown,
        MostLeft,
        MostRight,

        MostUpInClient,
        MostDownInClient,
        MostLeftInClient,
        MostRightInClient,

        PageUp,
        PageDown,
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ParameterEditor : IParameterValue
    {
        private string _Name = String.Empty;
        private IParameterValue _TargetParameter = null;
        private object _Value = null;

        public event ParameterEventHandler ValueChanged;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ParameterEditor(string name, IParameterValue targetParameter)
        {
            _Name = name;
            _TargetParameter = targetParameter;
            _Value = _TargetParameter.Value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return _Name; }
        }

        /// <summary>
        /// パラメータの属性コレクションを取得します。
        /// </summary>
        public ICollection<object> Attributes
        {
            get
            {
                return _TargetParameter.Attributes;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override string ToString()
        {
            return _TargetParameter.ToString();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Parse(string text)
        {
            _TargetParameter.Parse(text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual object ParseValue(string text)
        {
            return _TargetParameter.ParseValue(text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ValidationResult Validate()
        {
            return _TargetParameter.Validate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual ValidationResult ValidateValue(object value)
        {
            return _TargetParameter.ValidateValue(value);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual object Value
        {
            get { return _Value; }
            set
            {
                _Value = value;
                OnValueChanged();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected object InnerValue
        {
            get { return _Value; }
            set { _Value = value; }
        }

        /// <summary>
        /// パラメータキーを含めたパラメータのハッシュコードを取得します。
        /// </summary>
        /// <param name="algorithm">ハッシュアルゴリズムを指定します。</param>
        /// <param name="key">パラメータキーを指定します。</param>
        /// <param name="filter">対象パラメータのフィルタを指定します。</param>
        /// <returns>ハッシュコードを返します。</returns>
        public HashCode GetParameterHashCode(HashAlgorithm algorithm, string key, Func<IParameterValue, bool> filter)
        {
            return _TargetParameter.GetParameterHashCode(algorithm, key, filter);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnValueChanged()
        {
            if (ValueChanged != null)
            {
                ValueChanged(this, new ParameterEventArgs(string.Empty, this));
            }
        }
    }
}
