﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Windows.Forms;
using System.Drawing;
using System.Collections;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class OperatableController : FlexibleScrollableControl
    {

        protected SortedList Drawers = new SortedList();
        protected OperatableControllerOperator CurrentOperator = null;

        private MouseButtons _LastUpButton = MouseButtons.None;
        private bool _EnableEventLoop = false;
        private Timer _Timer;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public OperatableController()
        {
            base.BackColor = SystemColors.Window;

            InitializeComponent();

            SetStyle(ControlStyles.AllPaintingInWmPaint
                                | ControlStyles.DoubleBuffer, true);
        }

        ///--------------------------------
        /// <summary>
        /// 表示情報の取得
        /// 派生したコントロールはこれをオーバーライドする必要がある
        /// </summary>
        protected virtual Object GetDrawData()
        {
            return null;
        }

        ///--------------------------------
        /// <summary>
        /// Drawerの追加
        /// </summary>
        protected virtual void AddDrawer(int key, OperatableControllerDrawer drawer)
        {
            Drawers.Add(key, drawer);
            drawer.OwnerController = this;
        }

        ///--------------------------------
        /// <summary>
        /// Drawerの追加
        /// </summary>
        public void AddDrawer(OperatableControllerDrawer drawer)
        {
            int key = 1;

            while (Drawers.ContainsKey(key) != false)
            {
                key++;
            }
            AddDrawer(key, drawer);
        }

        ///--------------------------------
        /// <summary>
        /// 描画
        /// </summary>
        protected virtual void Draw(Graphics gc)
        {
            Object data = null;
            bool enabled = true;
            Rectangle rect = ClientRectangleWithoutScrollBar;

            data = GetDrawData();
            if (Enabled == false || data == null)
            {
                enabled = false;
            }

            PastDraw(gc, rect, data, enabled);

            foreach (OperatableControllerDrawer drawer in Drawers.Values)
            {
                if (drawer.Enable != false)
                {
                    drawer.Draw(gc, rect, data, enabled);
                }
            }

            PostDraw(gc, rect, data, enabled);
        }

        ///--------------------------------
        /// <summary>
        /// 描画
        /// </summary>
        protected virtual void PastDraw(Graphics gc, Rectangle clientRectangle, Object data, bool enabled)
        {
        }

        ///--------------------------------
        /// <summary>
        /// 描画
        /// </summary>
        protected virtual void PostDraw(Graphics gc, Rectangle clientRectangle, Object data, bool enabled)
        {
        }

        ///--------------------------------
        /// <summary>
        /// 背景の描画
        /// </summary>
        protected override void OnPaintBackground(PaintEventArgs e)
        {
            base.OnPaintBackground(e);
        }

        ///--------------------------------
        /// <summary>
        /// 描画
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);
            Draw(e.Graphics);
        }

        ///--------------------------------
        /// <summary>
        /// タイマーの起動
        /// </summary>
        protected virtual void StartTimer()
        {
            StartTimer(1000);
        }

        ///--------------------------------
        /// <summary>
        /// タイマーの起動
        /// </summary>
        protected virtual void StartTimer(int interval)
        {
            if (_Timer == null)
            {
                _Timer = new Timer();
                _Timer.Tick += new EventHandler(this.OnTimer);
            }

            _Timer.Interval = interval;
            _Timer.Start();
        }

        ///--------------------------------
        /// <summary>
        /// タイマーの再起動
        /// </summary>
        protected virtual void RestartTimer()
        {
            StopTimer();
            StartTimer();
        }

        ///--------------------------------
        /// <summary>
        /// タイマーの停止
        /// </summary>
        protected virtual void StopTimer()
        {
            _Timer.Stop();
        }

        ///--------------------------------
        /// <summary>
        /// イベントループを開始する
        /// </summary>
        public void BeginEventLoop()
        {
            Point clientPoint;

            _EnableEventLoop = true;
            while (_EnableEventLoop != false)
            {

                clientPoint = PointToClient(MousePosition);
                CurrentOperator.OnEventLoop
                    (new MouseEventArgs(_LastUpButton,
                                                                0, clientPoint.X, clientPoint.Y, 0));

                System.Threading.Thread.Sleep(1);
                System.Windows.Forms.Application.DoEvents();
            }
        }

        ///--------------------------------
        /// <summary>
        /// イベントループを終了する
        /// </summary>
        public void EndEventLoop()
        {
            _EnableEventLoop = false;
        }

        ///--------------------------------
        /// <summary>
        /// タイマー処理
        /// </summary>
        private void OnTimer(object sender, EventArgs e)
        {
            if (CurrentOperator != null)
            {
                CurrentOperator.OnTimer(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// マウスボタンの押下処理
        /// </summary>
        protected override void OnMouseDown(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseDown(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnMouseDown(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// マウスボタンの押上処理
        /// </summary>
        protected override void OnMouseUp(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseUp(e);

            _LastUpButton = e.Button;
            if (CurrentOperator != null)
            {
                CurrentOperator.OnMouseUp(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// マウスボタンのダブルクリック処理
        /// </summary>
        protected override void OnDoubleClick(System.EventArgs e)
        {
            base.OnDoubleClick(e);

            if (CurrentOperator != null)
            {
                Point clientPoint = PointToClient(MousePosition);

                CurrentOperator.OnDoubleClick
                    (new MouseEventArgs(_LastUpButton,
                                                                0, clientPoint.X, clientPoint.Y, 0));
            }
        }

        ///--------------------------------
        /// <summary>
        /// マウスの移動処理
        /// </summary>
        protected override void OnMouseMove(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseMove(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnMouseMove(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// マウスのホイール処理
        /// </summary>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnMouseWheel(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// キーが押された処理
        /// </summary>
        protected override void OnKeyDown(System.Windows.Forms.KeyEventArgs e)
        {
            base.OnKeyDown(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnKeyDown(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// キーが離された処理
        /// </summary>
        protected override void OnKeyUp(System.Windows.Forms.KeyEventArgs e)
        {
            base.OnKeyUp(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnKeyUp(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// 文字が入力された処理
        /// </summary>
        protected override void OnKeyPress(System.Windows.Forms.KeyPressEventArgs e)
        {
            base.OnKeyPress(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnKeyPress(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// ドロップ処理
        /// </summary>
        protected override void OnDragDrop(System.Windows.Forms.DragEventArgs e)
        {
            base.OnDragDrop(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnDragDrop(e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragEnter(System.Windows.Forms.DragEventArgs e)
        {
            base.OnDragEnter(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnDragEnter(e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragLeave(System.EventArgs e)
        {
            base.OnDragLeave(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnDragLeave(e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragOver(System.Windows.Forms.DragEventArgs e)
        {
            base.OnDragOver(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnDragOver(e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnGiveFeedback(System.Windows.Forms.GiveFeedbackEventArgs e)
        {
            base.OnGiveFeedback(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnGiveFeedback(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// フォーカスを失った時の処理
        /// </summary>
        protected override void OnLostFocus(System.EventArgs e)
        {
            base.OnLostFocus(e);

            if (CurrentOperator != null)
            {
                CurrentOperator.OnLostFocus(e);
            }
        }


        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void InitializeComponent()
        {
            //
            // OperatableController
            //
            this.Name = "OperatableController";
        }
    }


    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class OperatableControllerOperator
    {

        protected OperatableController OperatableController = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private OperatableControllerOperator() { }
        public OperatableControllerOperator(OperatableController control)
        {
            OperatableController = control;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected OperatableController OwnerController
        {
            get { return OperatableController; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Point PointToClient(int x, int y)
        {
            return OperatableController.PointToClient(new Point(x, y));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Point PointToClient(Point point)
        {
            return OperatableController.PointToClient(point);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected internal virtual void Initialize(Point beginPoint) { }

        protected internal virtual bool OnMouseDown(MouseEventArgs e)
        {
            return true;
        }
        protected internal virtual bool OnMouseUp(MouseEventArgs e)
        {
            return true;
        }
        protected internal virtual bool OnMouseMove(MouseEventArgs e)
        {
            return true;
        }
        protected internal virtual bool OnMouseWheel(MouseEventArgs e)
        {
            return true;
        }
        protected internal virtual void OnDoubleClick(MouseEventArgs e) { }

        protected internal virtual void OnKeyDown(KeyEventArgs e) { }
        protected internal virtual void OnKeyUp(KeyEventArgs e) { }
        protected internal virtual void OnKeyPress(KeyPressEventArgs e) { }

        protected internal virtual void OnDragDrop(DragEventArgs e) { }
        protected internal virtual void OnDragOver(DragEventArgs e) { }
        protected internal virtual void OnDragEnter(DragEventArgs e) { }
        protected internal virtual void OnDragLeave(EventArgs e) { }
        protected internal virtual void OnEventLoop(MouseEventArgs e) { }

        protected internal virtual void OnGiveFeedback(GiveFeedbackEventArgs e) { }

        protected internal virtual bool OnLostFocus(EventArgs e)
        {
            return true;
        }

        protected internal virtual void OnTimer(EventArgs e) { }
    }


    ///--------------------------------------------------------------------------
    /// <summary>
    /// 描画クラス
    /// </summary>
    public class OperatableControllerDrawer : Object
    {
        private OperatableController _OwnerController = null;
        private bool _Enable = true;

        ///--------------------------------
        /// <summary>
        /// 有効/無効
        /// </summary>
        public bool Enable
        {
            get { return _Enable; }
            set { _Enable = value; }
        }

        ///--------------------------------
        /// <summary>
        /// オーナーコントロールの取得/設定
        /// </summary>
        protected internal virtual OperatableController OwnerController
        {
            get { return _OwnerController; }
            set { _OwnerController = value; }
        }

        ///--------------------------------
        /// <summary>
        /// 描画
        /// </summary>
        protected internal virtual void Draw(Graphics gc, Rectangle clientRectangle, Object data, bool enabled)
        {
        }
    }
}
