﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    public partial class FloatingFormT<PageType> : NForm
        where PageType : class, IEmbeddedPage
    {
        #region ** フィールド

        private bool _adjustToPageSize = true;
        private PageType _activePage = null;

        #endregion

        public FloatingFormT()
        {
            InitializeComponent();
        }

        #region ** プロパティ

        /// <summary>
        /// ウィンドウサイズをアクティブページのサイズに合わせるかどうか、を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [DefaultValue(true)]
        public bool AdjustToPageSize
        {
            get { return _adjustToPageSize; }
            set
            {
                if (value == _adjustToPageSize) { return; }

                _adjustToPageSize = value;
                OnAdjustToPageSizeChanged(new EventArgs());
            }
        }

        /// <summary>
        /// アクティブページを取得または設定します。
        /// </summary>
        [Browsable(false)]
        [ReadOnly(true)]
        public PageType ActivePage
        {
            get { return _activePage; }
            set { ActivatePage(value, true); }
        }

        #endregion

        #region ** イベント

        public EventHandler AdjustToPageSizeChanged;
        public EventHandler ActivePageChanged;

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnAdjustToPageSizeChanged(EventArgs e)
        {
            if (null == AdjustToPageSizeChanged) { return; }

            AdjustToPageSizeChanged(this, e);
        }

        protected virtual void OnActivePageChanged(EventArgs e)
        {
            if (null == ActivePageChanged) { return; }

            ActivePageChanged(this, e);
        }

        protected override void OnActivated(EventArgs e)
        {
            base.OnActivated(e);

            if (this._activePage != null)
            {
                this._activePage.WindowActive = true;
            }
        }

        protected override void OnDeactivate(EventArgs e)
        {
            base.OnDeactivate(e);

            if (this._activePage != null)
            {
                this._activePage.WindowActive = false;
            }
        }

        #endregion

        #region ** メソッド

        public void AddPage(PageType newPage)
        {
            if (null == newPage) { throw new ArgumentNullException(); }
            if (!(newPage is Control)) { throw new ArgumentException(); }

            if (null != newPage)
            {
                Controls.Add(newPage as Control);
            }

            if (null == _activePage)
            {
                ActivatePage(newPage, true);
            }
        }

        public void RemovePage(PageType targetPage)
        {
            if (null == targetPage) { throw new ArgumentNullException(); }

            RemovePage(targetPage, true);
        }

        private void RemovePage(PageType targetPage, bool dispachEvent)
        {
            if (null == targetPage) { throw new ArgumentNullException(); }
            if (!(targetPage is Control)) { throw new ArgumentException(); }

            Controls.Remove(targetPage as Control);

            if (targetPage == _activePage)
            {
                ActivatePage(null, dispachEvent);
            }
        }

        private void ActivatePage(PageType targetPage, bool dispachEvent)
        {
            if (targetPage == _activePage) { return; }

            Control targetPageControl = targetPage as Control;

            if (null != targetPage)
            {
                if (null == targetPageControl) { throw new ArgumentException(); }
                if ((targetPageControl.Parent as NForm) != this) { throw new Exception(); }
            }

            if (_adjustToPageSize)
            {
                AdjustSize(targetPage);
            }

            if (null != _activePage)
            {
                _activePage.Active = false;
                (_activePage as Control).Dock = DockStyle.None;
            }

            if (null != targetPage)
            {
                targetPage.Active = true;
                targetPageControl.Dock = DockStyle.Fill;
            }

            _activePage = targetPage;

            if (null != targetPage)
            {
                Text = targetPageControl.Text;
                targetPageControl.Show();
            }

            if (dispachEvent)
            {
                OnActivePageChanged(new EventArgs());
            }
        }

        private void AdjustSize(PageType targetPage)
        {
            if (null == targetPage) { return; }
            if (!(targetPage is Control)) { throw new ArgumentException(); }

            Size ncSize = new Size(Bounds.Width - ClientRectangle.Width, Bounds.Height - ClientRectangle.Height);

            Size = (targetPage as Control).Size + ncSize;
        }

        #endregion
    }

    #region ** 特殊化

    public class FloatingForm : FloatingFormT<EmbeddedPage> { }

    #endregion
}
