﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Design;
using System.Drawing.Drawing2D;
using System.Data;
using System.Text;
using System.Windows.Forms;

using NintendoWare.SoundFoundation.Windows.Forms.Windowless;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    public partial class DockingFormContainerTitle : UserControl
    {
        #region ** 固定値

        private int MinimizeTopImageIndex = 1;
        private int MinimizeBottomImageIndex = 2;
        private int MinimizeLeftImageIndex = 3;
        private int MinimizeRightImageIndex = 4;

        #endregion

        #region ** フィールド

        private Control _target = null;
        private Pen _borderPen = null;
        private Brush _backgroundBrush = null;
        private bool _closeButtonVisible = false;

        private string _title = string.Empty;
        private Rectangle _titleRect = Rectangle.Empty;

        private DockingFormContainer.DockAlignment _alignment = DockingFormContainer.DockAlignment.Top;

        private bool _focused = false;

        #endregion

        public DockingFormContainerTitle()
        {
            InitializeComponent();
            UpdateDrawComponent();

            SetStyle(ControlStyles.Selectable, false);
        }

        #region ** プロパティ

        public DockingFormContainer.DockAlignment Alignment
        {
            get { return _alignment; }
            set
            {
                if (value == _alignment) { return; }

                _alignment = value;
                UpdateButtonImage();
            }
        }

        public new string Text
        {
            get { return _title; }
            set
            {
                if (value == _title) { return; }

                _title = (null != value) ? value : string.Empty;
                Invalidate();
            }
        }

        public Control TargetControl
        {
            get { return _target; }
            set
            {
                if (value == _target) { return; }

                if (null != _target)
                {
                    _target.Enter -= OnTargetEnter;
                    _target.Leave -= OnTargetLeave;
                }

                _target = value;

                if (null != _target)
                {
                    _target.Enter += OnTargetEnter;
                    _target.Leave += OnTargetLeave;
                }
            }
        }

        public bool CloseButtonVisible
        {
            get { return _closeButton.Visible; }
            set
            {
                if (value == _closeButtonVisible) { return; }

                _closeButtonVisible = value;
                UpdateButtons();
            }
        }

        #endregion

        #region ** イベント

        public event MouseEventHandler CloseButtonMouseClick;
        public event MouseEventHandler MinimizeButtonMouseClick;

        #endregion

        #region ** イベントハンドラ

        protected override void OnPaint(PaintEventArgs e)
        {
            Brush brush = SystemBrushes.Control;

            if (null != _target && _focused)
            {
                brush = _backgroundBrush;
            }

            e.Graphics.FillRectangle(brush, ClientRectangle);
            e.Graphics.DrawLine(_borderPen, ClientRectangle.Left, ClientRectangle.Bottom - 1,
                                             ClientRectangle.Right, ClientRectangle.Bottom - 1);

            e.Graphics.DrawString(_title, Font, new SolidBrush(ForeColor), _titleRect,
                                    new StringFormat(StringFormatFlags.NoWrap)
                                    {
                                        LineAlignment = StringAlignment.Center,
                                        Alignment = StringAlignment.Near,
                                    });
        }

        protected override void OnPaintBackground(PaintEventArgs e)
        {
            // 何もしない
        }

        protected override void OnSystemColorsChanged(EventArgs e)
        {
            base.OnSystemColorsChanged(e);
            UpdateDrawComponent();
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            UpdateTitleArea();
        }

        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);

            if (Visible)
            {
                UpdateButtons();
            }
        }

        private void OnTargetEnter(object sender, EventArgs e)
        {
            _focused = true;
            Invalidate();
        }

        private void OnTargetLeave(object sender, EventArgs e)
        {
            _focused = false;
            Invalidate();
        }

        private void OnTitleMouseDown(object sender, MouseEventArgs e)
        {
            OnMouseDown(e);
        }

        private void OnTitleDoubleClick(object sender, EventArgs e)
        {
            OnDoubleClick(new EventArgs());
        }

        private void OnTitleMouseClick(object sender, MouseEventArgs e)
        {
            Point screenLocation = (sender as Control).PointToScreen(e.Location);
            Point location = PointToClient(screenLocation);

            OnMouseClick(new MouseEventArgs(e.Button, e.Clicks, location.X, location.Y, e.Delta));
        }

        private void OnMinimizeButtonMouseClick(object sender, MouseEventArgs e)
        {
            if (null != MinimizeButtonMouseClick)
            {
                MinimizeButtonMouseClick(this, e);
            }
        }

        private void OnCloseButtonMouseClick(object sender, MouseEventArgs e)
        {
            if (null != CloseButtonMouseClick)
            {
                CloseButtonMouseClick(this, e);
            }
        }

        #endregion

        #region ** メソッド

        private void UpdateDrawComponent()
        {
            Color color = NWColor.Blend(NWColor.ThemeBorder, Color.White, 160);
            Color lightColor = NWColor.Blend(color, Color.White, 128);

            _borderPen = new Pen(NWColor.ThemeBorder);
            _backgroundBrush = new LinearGradientBrush(new Rectangle(0, 0, MinimumSize.Width, MinimumSize.Height),
                                                        lightColor, color, LinearGradientMode.Vertical);
        }

        private void UpdateButtons()
        {
            _closeButton.Visible = _closeButtonVisible;

            if (_closeButtonVisible)
            {
                _minimizeButton.Left = _closeButton.Left - _minimizeButton.Margin.Right - _minimizeButton.Width;
            }
            else
            {
                _minimizeButton.Left = _closeButton.Left;
            }

            UpdateTitleArea();
        }

        private void UpdateTitleArea()
        {
            _titleRect = new Rectangle(3, 0, _minimizeButton.Left - 3, Height);
        }

        private void UpdateButtonImage()
        {
            switch (_alignment)
            {
                case DockingFormContainer.DockAlignment.Top:
                    _minimizeButton.ImageIndex = MinimizeTopImageIndex;
                    break;

                case DockingFormContainer.DockAlignment.Bottom:
                    _minimizeButton.ImageIndex = MinimizeBottomImageIndex;
                    break;

                case DockingFormContainer.DockAlignment.Left:
                    _minimizeButton.ImageIndex = MinimizeLeftImageIndex;
                    break;

                case DockingFormContainer.DockAlignment.Right:
                    _minimizeButton.ImageIndex = MinimizeRightImageIndex;
                    break;
            }
        }

        #endregion

        #region ** コントロール

        public class NonFocusButton : Control
        {
            #region ** 固定値

            private enum State
            {
                Normal = 0,
                Hot,
                Selected,
                Disabled,
            }

            #endregion

            private State _state = State.Normal;
            private Brush _backBrush = null;

            private ImageList _imageList = null;
            private int _imageIndex = -1;

            public NonFocusButton()
            {
                SetStyle(ControlStyles.Selectable, false);
                SetStyle(ControlStyles.SupportsTransparentBackColor, true);
                SetStyle(ControlStyles.UserPaint, true);
            }

            #region ** プロパティ

            public ImageList ImageList
            {
                get { return _imageList; }
                set { _imageList = value; }
            }

            [Editor("Design.ImageIndexEditor, System.Design, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a", typeof(UITypeEditor))]
            [RefreshProperties(RefreshProperties.Repaint)]
            [TypeConverter(typeof(ImageIndexConverter))]
            public int ImageIndex
            {
                get { return _imageIndex; }
                set { _imageIndex = value; }
            }

            protected Image Image
            {
                get
                {
                    if (null == _imageList) { return null; }
                    if (0 > _imageIndex) { return null; }
                    if (_imageList.Images.Count <= _imageIndex) { return null; }
                    return _imageList.Images[_imageIndex];
                }
            }

            #endregion

            #region ** イベントハンドラ

            protected override void OnMouseEnter(EventArgs e)
            {
                base.OnMouseEnter(e);
                SetState(State.Hot);
            }

            protected override void OnMouseLeave(EventArgs e)
            {
                base.OnMouseLeave(e);
                SetState(State.Normal);
            }

            protected override void OnMouseMove(MouseEventArgs e)
            {
                base.OnMouseMove(e);

                if (!Capture) { return; }

                SetState(ClientRectangle.Contains(e.Location) ? State.Hot : State.Selected);
            }

            protected override void OnMouseDown(MouseEventArgs e)
            {
                base.OnMouseDown(e);

                Capture = true;

                if (MouseButtons.Left == e.Button)
                {
                    SetState(State.Selected);
                }
            }

            protected override void OnMouseUp(MouseEventArgs e)
            {
                base.OnMouseUp(e);

                Capture = false;
                SetState(ClientRectangle.Contains(e.Location) ? State.Hot : State.Normal);
            }

            protected override void OnPaint(PaintEventArgs e)
            {
                if (null == _backBrush)
                {
                    UpdateDrawComponent();
                }

                Rectangle drawRect = ClientRectangle;
                drawRect.Width -= 1;
                drawRect.Height -= 1;

                switch (_state)
                {
                    case State.Selected:
                        e.Graphics.FillRectangle(new SolidBrush(ControlPaint.LightLight(SystemColors.Highlight)), drawRect);
                        e.Graphics.DrawRectangle(SystemPens.Highlight, drawRect);
                        break;

                    case State.Hot:
                        e.Graphics.FillRectangle(new SolidBrush(ControlPaint.LightLight(ControlPaint.Light(SystemColors.Highlight))), drawRect);
                        e.Graphics.DrawRectangle(SystemPens.Highlight, drawRect);
                        break;
                }

                if (null != Image)
                {

                    Rectangle rect = ClientRectangle;
                    rect.X += (rect.Width - Image.Size.Width) / 2;
                    rect.Y += (rect.Height - Image.Size.Height) / 2;
                    rect.Width = Image.Size.Width;
                    rect.Height = Image.Size.Height;

                    e.Graphics.DrawImage(Image, rect);

                }
            }

            protected override void OnBackColorChanged(EventArgs e)
            {
                base.OnBackColorChanged(e);
                UpdateDrawComponent();
            }

            #endregion

            #region ** メソッド

            private void SetState(State newState)
            {
                if (_state == newState) { return; }

                _state = newState;
                Invalidate();
            }

            private void UpdateDrawComponent()
            {
                _backBrush = new SolidBrush(BackColor);
            }

            #endregion
        }

        #endregion
    }
}
