﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;

    public class ComponentVelocityRegion : IVelocityRegion, IDisposable
    {
        private VelocityRegion _Target;
        private int _Minimum;
        private int _Maximum;
        private string _FilePath;
        private int _OriginalKey;
        private WaveEncoding _WaveEncoding;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentVelocityRegion(VelocityRegion target)
        {
            _Target = target;
            _Target.Parameters.ParameterValueChanged += OnParameterValueChanged;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Dispose()
        {
            if (_Target != null)
            {
                _Target.Parameters.ParameterValueChanged -= OnParameterValueChanged;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IKeyRegion Parent { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Component Target
        {
            get
            {
                return _Target;
            }
            set
            {
                _Target = (VelocityRegion)value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int MinimumIntact
        {
            set
            {
                ChangeMinimum(value);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int MaximumIntact
        {
            set
            {
                ChangeMaximum(value);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Minimum
        {
            get
            {
                return _Minimum;
            }
            set
            {
                int min = value;

                if (Parent != null && Parent.Children != null)
                {
                    ComponentVelocityRegionCollection Children =
                        (ComponentVelocityRegionCollection)Parent.Children;
                    ComponentVelocityRegion prevChild = Children.PrevChild(this);
                    if (prevChild != null && prevChild.Maximum == Minimum - 1)
                    {
                        int prevMax = value - 1;
                        if (prevMax < 0)
                        {
                            prevMax = 0;
                            min = prevMax + 1;
                        }
                        if (127 < min)
                        {
                            min = 127;
                            prevMax = min - 1;
                        }
                        prevChild.ChangeMaximum(prevMax);
                    }
                }

                if (min < 0)
                {
                    min = 0;
                }
                if (127 < min)
                {
                    min = 127;
                }

                ChangeMinimum(min);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Maximum
        {
            get
            {
                return _Maximum;
            }
            set
            {
                int max = value;

                if (Parent != null && Parent.Children != null)
                {
                    ComponentVelocityRegionCollection Children =
                        (ComponentVelocityRegionCollection)Parent.Children;
                    ComponentVelocityRegion nextChild = Children.NextChild(this);
                    if (nextChild != null && nextChild.Minimum == Maximum + 1)
                    {
                        int nextMin = value + 1;
                        if (127 < nextMin)
                        {
                            nextMin = 127;
                            max = nextMin - 1;
                        }
                        if (max < 0)
                        {
                            max = 0;
                            nextMin = max + 1;
                        }
                        nextChild.ChangeMinimum(nextMin);
                    }
                }

                if (max < 0)
                {
                    max = 0;
                }
                if (127 < max)
                {
                    max = 127;
                }

                ChangeMaximum(max);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string FilePath
        {
            get
            {
                return _FilePath;
            }
            set
            {
                ChangeFilePath(value);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int OriginalKey
        {
            get
            {
                return _OriginalKey;
            }
            set
            {
                ChangeOriginalKey(value);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public WaveEncoding WaveEncoding
        {
            get
            {
                return _WaveEncoding;
            }
            set
            {
                ChangeWaveEncoding(value);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ChangeMinimum(int value)
        {
            if (Parent != null && Parent.Parent != null)
            {
                ComponentSampleMapAdapter Adapter = (ComponentSampleMapAdapter)Parent.Parent;
                Adapter.SetValue(Target, ProjectParameterNames.VelocityRegion.VelocityMin, value);
                _Minimum = ((VelocityRegion)Target).VelocityMin;
            }
            else
            {
                _Minimum = value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ChangeMaximum(int value)
        {
            if (Parent != null && Parent.Parent != null)
            {
                ComponentSampleMapAdapter Adapter = (ComponentSampleMapAdapter)Parent.Parent;
                Adapter.SetValue(Target, ProjectParameterNames.VelocityRegion.VelocityMax, value);
                _Maximum = ((VelocityRegion)Target).VelocityMax;
            }
            else
            {
                _Maximum = value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ChangeFilePath(string value)
        {
            if (Parent != null && Parent.Parent != null)
            {
                ComponentSampleMapAdapter Adapter = (ComponentSampleMapAdapter)Parent.Parent;
                Adapter.SetValue(Target, ProjectParameterNames.FilePath, value);
                _FilePath = ((VelocityRegion)Target).FilePath;
            }
            else
            {
                _FilePath = value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ChangeOriginalKey(int value)
        {
            if (Parent != null && Parent.Parent != null)
            {
                ComponentSampleMapAdapter Adapter = (ComponentSampleMapAdapter)Parent.Parent;
                Adapter.SetValue(Target, ProjectParameterNames.VelocityRegion.OriginalKey, value);
                _OriginalKey = ((VelocityRegion)Target).OriginalKey;
            }
            else
            {
                _OriginalKey = value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ChangeWaveEncoding(WaveEncoding value)
        {
            if (Parent != null && Parent.Parent != null)
            {
                ComponentSampleMapAdapter Adapter = (ComponentSampleMapAdapter)Parent.Parent;
                Adapter.SetValue(Target, ProjectParameterNames.WaveEncoding, value);
                _WaveEncoding = ((VelocityRegion)Target).Encoding;
            }
            else
            {
                _WaveEncoding = value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnParameterValueChanged(object sender, ParameterEventArgs e)
        {
            if (e.Key.EndsWith(ParameterNames.TemporaryPostfix) == true)
            {
                return;
            }

            _FilePath = _Target.FilePath;
            _WaveEncoding = _Target.Encoding;
            _OriginalKey = _Target.OriginalKey;
            _Minimum = _Target.VelocityMin;
            _Maximum = _Target.VelocityMax;

            if (Parent != null && Parent.Parent != null)
            {
                ((ComponentSampleMapAdapter)(Parent.Parent)).Update();
            }
        }
    }
}
