﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Diagnostics;
using System.Linq;
using System.Text;
using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Operations;
using NintendoWare.SoundFoundation.Parameters;
using NintendoWare.SoundFoundation.Projects;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    public class ComponentTreeAdapter : ITreeItemsSource
    {
        private ComponentTreeItemCollection _Items = null;
        private bool _SuspendComponentProcess = false;

        public event EventHandler Updated;
        public event OperationExecutedEventHandler OperationExecuted;

        private Dictionary<Component, bool> _ExpandedDictionary = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentTreeAdapter()
        {
            _Items = new ComponentTreeItemCollection();
            _Items.CollectionChanged += OnCollectionChanged;

            _ExpandedDictionary = new Dictionary<Component, bool>();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void DisposeItems()
        {
            DisposeItems(Items);
            _ExpandedDictionary.Clear();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentTreeItemCollection Items
        {
            get { return _Items; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        ITreeItemCollection ITreeItemsSource.Items
        {
            get { return Items; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool Suspend { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Update()
        {
            OnUpdated(new EventArgs());
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void BeginTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.BeginTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void EndTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.EndTransaction();
            }

            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void CancelTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public OperationHistory OperationHistory { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual string NameConvert(string name)
        {
            return name;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual void SetValue(ComponentTreeItem item, string name, string value)
        {
            Component targetComponent = item.Target;

            if (targetComponent.Parameters.ContainsKey(name) == false)
            {
                return;
            }

            //Operation: ここで Operationにする
            SetParameterOperation operation = null;
            operation = new SetParameterOperation(targetComponent.Parameters, name, value);
            operation.Execute();
            OnOperationExecuted(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void RemoveComponent(ComponentTreeItem parentItem, ComponentTreeItem item)
        {
            if (_SuspendComponentProcess != false)
            {
                return;
            }

            //
            PreserveExpandedState(item.Target, item.Expanded);

            //Operation: ここで Operationにする
            RemoveComponentOperation operation = null;
            operation = new RemoveComponentOperation(new[] { item.Target });
            operation.Execute();
            OnOperationExecuted(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void AddComponent(ComponentTreeItem parentItem, ComponentTreeItem item)
        {
            if (_SuspendComponentProcess != false)
            {
                return;
            }

            //Operation: ここで Operationにする
            ComponentTreeItem nextSiblingItem = item.NextSibling as ComponentTreeItem;
            Component parent = null;
            Component nextSibling = null;

            parent = parentItem.Target;
            nextSibling = nextSiblingItem != null ? nextSiblingItem.Target : null;

            InsertComponentOperation operation = null;
            operation = new InsertComponentOperation(parent, nextSibling,
                                                        new[] { item.Target });
            operation.Execute();
            OnOperationExecuted(operation);

            //
            item.Expanded = RestoreExpandedState(item.Target);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void RemoveItem(Component component)
        {
            ComponentTreeItem parentItem = null;
            ComponentTreeItem item = null;

            if ((item = FindTreeItem(component)) == null)
            {
                return;
            }

            //
            PreserveExpandedState(item.Target, item.Expanded);

            //
            parentItem = item.Parent;
            parentItem.Children.Remove(item);
            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void AddItem(Component component)
        {
            ComponentTreeItem parentItem = null;
            ComponentTreeItem nextSiblingItem = null;
            ComponentTreeItem item = null;
            Component nextSiblingComponent = null;

            if ((parentItem = FindTreeItem(component.Parent)) == null)
            {
                return;
            }

            nextSiblingComponent = NextSiblingComponent(component);
            nextSiblingItem = FindTreeItem(parentItem, nextSiblingComponent);

            if (nextSiblingItem != null)
            {
                if (nextSiblingItem.Target != nextSiblingComponent)
                {
                    return;
                }
            }

            //
            try
            {
                _SuspendComponentProcess = true;

                if ((item = CreateTreeItem(component)) == null)
                {
                    return;
                }

                if (InsertItem(parentItem, nextSiblingItem, item) == false)
                {
                    return;
                }

                foreach (Component childComponent in component.Children)
                {
                    if (AddItem(item, null, childComponent) == false)
                    {
                        RemoveItem(item);
                        return;
                    }
                }

                //
                item.Expanded = RestoreExpandedState(item.Target);
            }

            finally
            {
                _SuspendComponentProcess = false;
            }

            //
            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool AddItem(ComponentTreeItem parentItem, ComponentTreeItem nextSiblingItem, Component component)
        {
            ComponentTreeItem item = null;

            if ((item = CreateTreeItem(component)) == null)
            {
                return true;
            }

            if (InsertItem(parentItem, nextSiblingItem, item) == false)
            {
                return false;
            }

            //
            foreach (Component childComponent in component.Children)
            {
                if (AddItem(item, null, childComponent) == false)
                {
                    return false;
                }
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual ComponentTreeItem CreateTreeItem(Component component)
        {
            return new ComponentTreeItem(component);
        }

        public ComponentTreeItem FindTreeItem(Component component)
        {
            foreach (ComponentTreeItem item in Items)
            {
                var foundItem = FindTreeItem(item, component);
                if (foundItem != null)
                {
                    return foundItem;
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ComponentTreeItem FindTreeItem(ComponentTreeItem item, Component component)
        {
            ComponentTreeItem foundItem = null;

            if (item.Target == component)
            {
                return item;
            }

            foreach (ComponentTreeItem childItem in item.Children)
            {
                if ((foundItem = FindTreeItem(childItem, component)) != null)
                {
                    return foundItem;
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool InsertItem(ComponentTreeItem parentItem, ComponentTreeItem nextSiblingItem, ComponentTreeItem item)
        {
            int index = -1;

            if (nextSiblingItem != null)
            {
                if ((index = parentItem.Children.IndexOf(nextSiblingItem)) < 0)
                {
                    return false;
                }

                parentItem.Children.Insert(index, item);
            }
            else
            {

                parentItem.Children.Add(item);
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void RemoveItem(ComponentTreeItem item)
        {
            ComponentTreeItem parentItem = item.Parent;
            parentItem.Children.Remove(item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Component NextSiblingComponent(Component component)
        {
            Component parentComponent = component.Parent;
            int index = -1;

            if ((index = parentComponent.Children.IndexOf(component)) < 0 ||
               index + 1 >= parentComponent.Children.Count)
            {
                return null;
            }

            return parentComponent.Children[index + 1];
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnOperationExecuted(Operation operation)
        {
            if (OperationHistory != null)
            {
                OperationHistory.AddOperation(operation);
            }

            OnOperationExecuted(new OperationExecutedEventArgs(operation));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnOperationExecuted(OperationExecutedEventArgs e)
        {
            if (OperationExecuted != null)
            {
                OperationExecuted(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnUpdated()
        {
            OnUpdated(new EventArgs());
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnUpdated(EventArgs e)
        {
            if (Updated != null)
            {
                Updated(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (ComponentTreeItem newItem in e.NewItems)
                    {
                        newItem.Adapter = this;
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (ComponentTreeItem oldItem in e.OldItems)
                    {
                        oldItem.Adapter = null;
                    }
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void DisposeItems(ComponentTreeItemCollection items)
        {
            foreach (ComponentTreeItem item in items)
            {
                DisposeItems(item);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void DisposeItems(ComponentTreeItem item)
        {
            foreach (ComponentTreeItem childItem in item.Children)
            {
                DisposeItems(childItem);
            }

            ((IDisposable)item).Dispose();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void PreserveExpandedState(Component component, bool expanded)
        {
            _ExpandedDictionary[component] = expanded;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool RestoreExpandedState(Component component)
        {
            bool expanded = false;

            if (_ExpandedDictionary.ContainsKey(component) != false)
            {
                expanded = _ExpandedDictionary[component];
            }
            return expanded;
        }
    }
}
