﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Diagnostics;
using System.Text;
using System.Windows.Forms;


namespace NintendoWare.SoundFoundation.Windows.Forms
{
    /// <summary>
    /// UserControl のカスタマイズクラス
    /// </summary>
    public partial class NUserControl : UserControl
    {
        #region ** フィールド

        // 描画オブジェクト
        private BufferedGraphics _bgSurface = null;						// バックグラウンドサーフェイス

        // パラメータ
        private bool _customizeFrame = false;					// ウィンドウフレームのカスタマイズ
        private BorderStyle _borderStyle = BorderStyle.None;			// 境界線の種類
        private Color _borderColor = SystemColors.ControlDark;	// 境界線の色

        #endregion

        public NUserControl()
        {
            // プロパティの初期化
            base.DoubleBuffered = false;

            InitializeComponent();
        }

        #region ** プロパティ

        /// <summary>
        /// ウィンドウフレームをカスタマイズします。
        /// </summary>
        [Category("Appearance")]
        [Description("ウィンドウフレームをカスタマイズします。")]
        [DefaultValue(false)]
        [RefreshProperties(RefreshProperties.Repaint)]
        public bool CustomizeFrame
        {
            get { return _customizeFrame; }
            set
            {
                _customizeFrame = value;
                UpdatePadding();
            }
        }

        /// <summary>
        /// 境界線の色を取得または設定します。
        /// </summary>
        [Category("Appearance")]
        [Description("境界線の色を取得または設定します。")]
        [DefaultValue(typeof(Color), "ControlDark")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public Color BorderColor
        {
            get { return _borderColor; }
            set { _borderColor = value; }
        }

        [Category("Appearance")]
        [Description("境界線の種類を取得または設定します。")]
        [DefaultValue(BorderStyle.None)]
        [RefreshProperties(RefreshProperties.Repaint)]
        public new BorderStyle BorderStyle
        {
            get
            {
                if (!_customizeFrame) { return base.BorderStyle; }
                return _borderStyle;
            }
            set
            {
                _borderStyle = value;

                if (!_customizeFrame)
                {
                    base.BorderStyle = value;
                    return;
                }

                UpdatePadding();
            }
        }

        #endregion

        #region ** プロパティのオーバーライド

        /// <summary>
        /// ちらつきを軽減または回避するために、2 次バッファを使用してコントロールの表面を再描画するかどうかを示す値を取得または設定します。
        /// </summary>
        [DefaultValue(false)]
        protected override bool DoubleBuffered
        {
            get { return base.DoubleBuffered; }
            set { base.DoubleBuffered = value; }
        }

        #endregion

        #region ** イベント

        /// <summary>
        /// ウィンドウフレームが再描画されると発生します。
        /// </summary>
        public event PaintEventHandler PaintFrame;

        #endregion

        #region ** イベントハンドラ

        /// <summary>
        /// PaintFrame イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している System.Windows.Forms.PaintEventArgs。</param>
        protected virtual void OnPaintFrame(PaintEventArgs e)
        {
            // 境界線を描画する
            DrawBorder(e);

            if (null != PaintFrame)
            {
                PaintFrame(this, e);
            }
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        /// <summary>
        /// System.Windows.Forms.Control.HandleCreated イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している System.EventArgs。</param>
        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);
            InvalidateSurface();
        }

        /// <summary>
        /// System.Windows.Forms.Control.Paint イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している System.Windows.Forms.PaintEventArgs。</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            UpdateSurface();

            if (null == _bgSurface)
            {
                Debug.Assert(0 == ClientRectangle.Width || 0 == ClientRectangle.Height);
                return;
            }


            PaintEventArgs pe = new PaintEventArgs(_bgSurface.Graphics, ClientRectangle);

            // 背景を描画する
            base.OnPaintBackground(pe);

            base.OnPaint(pe);

            // 境界線を描画する
            if (_customizeFrame)
            {
                OnPaintFrame(pe);
            }

            _bgSurface.Render(e.Graphics);
        }

        /// <summary>
        /// コントロールの背景を描画します。
        /// </summary>
        /// <param name="e">描画するコントロールに関する情報を格納する System.Windows.Forms.PaintEventArgs。</param>
        protected override void OnPaintBackground(PaintEventArgs e)
        {
            // ちらつき軽減のため、デフォルトの処理を行わない。
            // 背景は OnPaint にて描画する
        }

        protected override void OnPaddingChanged(EventArgs e)
        {
            base.OnPaddingChanged(e);
            UpdatePadding();
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// このコントロールの指定した境界を設定する作業を実行します。
        /// </summary>
        /// <param name="x">コントロールの新しい System.Windows.Forms.Control.Left プロパティ値。</param>
        /// <param name="y">コントロールの新しい System.Windows.Forms.Control.Top プロパティ値。</param>
        /// <param name="width">コントロールの新しい System.Windows.Forms.Control.Width プロパティ値。</param>
        /// <param name="height">コントロールの新しい System.Windows.Forms.Control.Height プロパティ値。</param>
        /// <param name="specified">System.Windows.Forms.BoundsSpecified 値のビットごとの組み合わせ。</param>
        protected override void SetBoundsCore(int x, int y, int width, int height, BoundsSpecified specified)
        {
            Rectangle oldClientRectangle = ClientRectangle;

            base.SetBoundsCore(x, y, width, height, specified);

            if (this.IsHandleCreated)
            {
                InvalidateBorderRectangle(oldClientRectangle);
                InvalidateBorderRectangle(ClientRectangle);
                InvalidateSurface();
            }
        }

        private void InvalidateBorderRectangle(Rectangle clientRectangle)
        {
            Invalidate(new Rectangle(clientRectangle.Left, clientRectangle.Top, clientRectangle.Width, 1));
            Invalidate(new Rectangle(clientRectangle.Left, clientRectangle.Bottom - 1, clientRectangle.Width, 1));
            Invalidate(new Rectangle(clientRectangle.Left, clientRectangle.Top, 1, clientRectangle.Height));
            Invalidate(new Rectangle(clientRectangle.Right - 1, clientRectangle.Top, 1, clientRectangle.Height));
        }

        /// <summary>
        /// バックグラウンドサーフェイスを無効化します。
        /// </summary>
        private void InvalidateSurface()
        {
            if (null != _bgSurface)
            {
                _bgSurface.Dispose();
                _bgSurface = null;
            }
        }

        /// <summary>
        /// バックグラウンドサーフェイスを更新します。
        /// </summary>
        private void UpdateSurface()
        {
            InvalidateSurface();

            if (0 >= ClientRectangle.Width || 0 >= ClientRectangle.Height) { return; }
            _bgSurface = BufferedGraphicsManager.Current.Allocate(CreateGraphics(), ClientRectangle);
        }

        /// <summary>
        /// 境界線に合わせてパディングを更新します。
        /// </summary>
        private void UpdatePadding()
        {
            if (!_customizeFrame) { return; }
            if (BorderStyle.None == BorderStyle) { return; }

            Padding workPadding = Padding;

            if (1 > workPadding.Left)
            {
                workPadding.Left = 1;
            }
            if (1 > workPadding.Right)
            {
                workPadding.Right = 1;
            }
            if (1 > workPadding.Top)
            {
                workPadding.Top = 1;
            }
            if (1 > workPadding.Bottom)
            {
                workPadding.Bottom = 1;
            }

            base.BorderStyle = BorderStyle.None;
            Padding = workPadding;
        }

        /// <summary>
        /// 境界線を描画します。
        /// </summary>
        /// <param name="e">イベント データを格納している System.Windows.Forms.PaintEventArgs。</param>
        private void DrawBorder(PaintEventArgs e)
        {
            switch (_borderStyle)
            {
                case BorderStyle.FixedSingle:
                case BorderStyle.Fixed3D:
                    ControlPaint.DrawBorder(e.Graphics, e.ClipRectangle, _borderColor, ButtonBorderStyle.Solid);
                    break;
            }
        }

        #endregion
    }
}
