﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.Text;
using System.Windows.Forms;
using System.Windows.Forms.VisualStyles;

namespace NintendoWare.SoundFoundation.Windows.Forms.Windowless
{
    #region ** ウィンドウレスコントロール

    #region ** コアコンポーネント

    public class NWControlHost
    {
        #region ** パラメータ

        private Control _owner = null;
        private NWControlRoot _root = null;

        private NWControlStyle _style = new NWControlStyle();    // スタイル
        private INWComponentTemplate _template = null;                    // テンプレート

        // 状態
        private bool _initialized = false;

        #endregion

        public NWControlHost(Control owner)
        {
            Debug.Assert(null != owner, "unexpected error.");

            _owner = owner;

            owner.MouseEnter += OnOwnerMouseEnter;
            owner.MouseLeave += OnOwnerMouseLeave;
            owner.MouseMove += OnOwnerMouseMove;
            owner.MouseDown += OnOwnerMouseDown;
            owner.MouseDoubleClick += OnOwnerMouseDoubleClick;
            owner.MouseUp += OnOwnerMouseUp;
        }

        #region ** プロパティ

        public Control Owner
        {
            get { return _owner; }
        }

        public NWControlRoot Root
        {
            get { return _root; }
            set
            {
                if (value == _root) { return; }

                SetRootInternal(value);
                OnRootChanged(new EventArgs());
            }
        }

        #region ** テンプレート

        public INWComponentTemplate Template
        {
            get { return _template; }
            set
            {
                if (value == _template) { return; }
                _template = value;
                InitializeComponent();
            }
        }

        public NWControlStyle Style
        {
            get { return _style; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                _style = value;
            }
        }

        protected virtual Type TemplateType
        {
            get { return typeof(INWComponentTemplate); }
        }

        #endregion

        #endregion

        #region ** イベント

        // 初期化
        public event EventHandler Initialized;
        public event EventHandler Uninitialize;
        public event EventHandler Uninitialized;

        // Root イベント
        public event EventHandler RootChanged;
        public event EventHandler RootInitialized;
        public event EventHandler RootUninitialize;
        public event EventHandler RootUninitialized;

        #endregion

        #region ** イベントハンドラ

        #region ** 初期化

        protected virtual void OnInitializeComponent() { }

        protected virtual void OnInitialized(EventArgs e)
        {
            if (null != Initialized)
            {
                Initialized(this, e);
            }
        }

        protected virtual void OnUninitialize(EventArgs e)
        {
            if (null != Uninitialize)
            {
                Uninitialize(this, e);
            }
        }

        protected virtual void OnUninitialized(EventArgs e)
        {
            if (null != Uninitialized)
            {
                Uninitialized(this, e);
            }
        }

        #endregion

        #region ** ルートイベント

        protected virtual void OnRootChanged(EventArgs e)
        {
            if (null != RootChanged)
            {
                RootChanged(this, e);
            }
        }

        private void OnRootInitialized(object sender, EventArgs e)
        {
            if (null != RootInitialized)
            {
                RootInitialized(this, e);
            }
        }

        private void OnRootUninitialize(object sender, EventArgs e)
        {
            if (null != RootUninitialize)
            {
                RootUninitialize(this, e);
            }
        }

        private void OnRootUninitialized(object sender, EventArgs e)
        {
            if (null != RootUninitialized)
            {
                RootUninitialized(this, e);
            }
        }

        #endregion

        #region ** オーナーコントロールのイベント

        private void OnOwnerMouseEnter(object sender, EventArgs e)
        {
            if (null == _root) { return; }
            _root.PerformMouseEnter();
        }

        private void OnOwnerMouseLeave(object sender, EventArgs e)
        {
            if (null == _root) { return; }
            _root.PerformMouseLeave();
        }

        private void OnOwnerMouseMove(object sender, MouseEventArgs e)
        {
            if (null == _root) { return; }
            _root.PerformMouseMove(e);
        }

        private void OnOwnerMouseDown(object sender, MouseEventArgs e)
        {
            if (null == _root) { return; }
            _root.PerformMouseDown(e);
        }

        private void OnOwnerMouseDoubleClick(object sender, MouseEventArgs e)
        {
            if (null == _root) { return; }
            _root.PerformMouseDoubleClick(e);
        }

        private void OnOwnerMouseUp(object sender, MouseEventArgs e)
        {
            if (null == _root) { return; }
            _root.PerformMouseUp(e);
        }

        #endregion

        #endregion

        #region ** メソッド

        #region ** 初期化

        protected void InitializeComponent()
        {
            if (null != _template)
            {
                InitializeComponent(_template);
                return;
            }

            if (Style.ContainsKey(GetType()))
            {
                _template = (Style[GetType()] as NWControlHostTemplateFactory).CreateInstance(this);
            }

            if (null == _template) { throw new Exception("template is not registered."); }

            UninitializeComponent();
            InitializeComponentInternal(_template);
        }

        protected void InitializeComponent(INWComponentTemplate template)
        {
            UninitializeComponent();
            InitializeComponentInternal(_template);
        }

        private void InitializeComponentInternal(INWComponentTemplate template)
        {
            Debug.Assert(!_initialized, "unexpected error.");
            Debug.Assert(null != template, "unexpected error.");

            if (!TemplateType.IsInstanceOfType(template))
            {
                throw new Exception("invalid control template.");
            }

            _template.InitializeComponent();

            OnInitializeComponent();

            _initialized = true;
            OnInitialized(new EventArgs());
        }

        private void UninitializeComponent()
        {
            if (!_initialized) { return; }

            OnUninitialize(new EventArgs());

            SetRootInternal(null);

            _initialized = false;
            OnUninitialized(new EventArgs());
        }

        #endregion

        #region ** ルート操作

        private void SetRootInternal(NWControlRoot newRoot)
        {
            if (_root == newRoot) { return; }

            if (null != _root)
            {
                _root.UninitializeComponent();

                _root.Initialized -= OnRootInitialized;
                _root.Uninitialize -= OnRootUninitialize;
                _root.Uninitialized -= OnRootUninitialized;
            }

            _root = newRoot;

            if (null != _root)
            {
                _root.Initialized += OnRootInitialized;
                _root.Uninitialize += OnRootUninitialize;
                _root.Uninitialized += OnRootUninitialized;
            }
        }

        #endregion

        public void ApplyStyle()
        {
            ResetTemplate();
            InitializeComponent();

            if (null == Root) { return; }

            Root.ApplyStyle();
        }

        private void ResetTemplate()
        {
            _template = null;
        }

        #endregion
    }

    public class NWControlRoot : NWControl
    {
        #region ** パラメータ

        private NWControl _mouseActiveControl = null;

        #endregion

        public NWControlRoot(NWControlHost host) : this(host, string.Empty) { }
        public NWControlRoot(NWControlHost host, string name)
            : base(host, name)
        {
            Host.Root = this;
        }

        #region ** プロパティ

        public NWControl MouseActive
        {
            get { return _mouseActiveControl; }
            set
            {
                if (_mouseActiveControl == value) { return; }

                if (null != _mouseActiveControl)
                {
                    _mouseActiveControl.PerformMouseLeave();
                }

                _mouseActiveControl = value;

                if (null != value)
                {
                    value.PerformMouseEnter();
                }
            }
        }

        #endregion

        #region ** イベント

        public event EventHandler Invalidated;

        #endregion

        #region ** イベントハンドラ

        #region ** NWControl マウスイベントのオーバーライド

        protected override void OnMouseLeave()
        {
            base.OnMouseLeave();

            MouseActive = null;
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);


            if ((e.Button & MouseButtons.Left) != 0) { return; }

            NWHitResult hr = NcHitTest(e.Location);

            if (null == hr.Control)
            {
                hr = HitTest(e.Location);
            }

            MouseActive = hr.Control;
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            // 複数ボタンが押されている場合は処理しない
            switch (e.Button)
            {
                case MouseButtons.Left:
                case MouseButtons.Right:
                case MouseButtons.Middle:
                    break;

                default:
                    return;
            }


            NWHitResult hr = NcHitTest(e.Location);

            if (null == hr.Control)
            {
                hr = HitTest(e.Location);
            }

            MouseActive = hr.Control;
            if (null == MouseActive) { return; }    // 対象コントロールがない場合は処理しない

            MouseActive.PerformMouseDown(e);
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            // アクティブコントロールにて MouseUp を処理する
            NWHitResult hr = NcHitTest(e.Location);

            if (null == hr.Control)
            {
                hr = HitTest(e.Location);
            }

            if (null == MouseActive) { return; }            // アクティブコントロールがない場合は処理しない
            if (MouseActive != hr.Control) { return; }        // 対象コントロールと一致しない場合は処理しない

            MouseActive.PerformMouseUp(e);
        }

        protected override void OnMouseDoubleClick(MouseEventArgs e)
        {
            // 複数ボタンが押されている場合は処理しない
            switch (e.Button)
            {
                case MouseButtons.Left:
                case MouseButtons.Right:
                case MouseButtons.Middle:
                    break;

                default:
                    return;
            }


            NWHitResult hr = NcHitTest(e.Location);

            if (null == hr.Control)
            {
                hr = HitTest(e.Location);
            }

            MouseActive = hr.Control;
            if (null == MouseActive) { return; }    // 対象コントロールがない場合は処理しない

            MouseActive.PerformMouseDoubleClick(e);
        }

        protected override void OnMouseActivating(EventArgs e)
        {
            base.OnMouseActivating(e);

            if (null != MouseActive)
            {
                MouseActive.PerformMouseActivating();
            }
        }

        #endregion

        protected virtual void OnInvalidated(EventArgs e)
        {
            if (null != Invalidated)
            {
                Invalidated(this, e);
            }
        }

        #endregion

        #region ** メソッド

        #region ** 描画操作

        protected override void Invalidate(Rectangle bounds)
        {
            Host.Owner.Invalidate(bounds);
            OnInvalidated(new EventArgs());
        }

        protected override void Invalidate(NWControl control)
        {
            Host.Owner.Invalidate(control.AbsoluteBounds);
            OnInvalidated(new EventArgs());
        }

        #endregion

        #endregion
    }

    /// <summary>
    /// ウィンドウレスコントロール
    /// </summary>
    public class NWControl
    {
        public const int InvalidIndex = ControlCollection.InvalidIndex;

        #region ** パラメータ

        // リンク
        private NWControlHost _host = null;                           // コントロールホスト
        private NWControl _parent = null;                           // 親コントロール
        private ControlCollectionImpl _ncControls = new ControlCollectionImpl();    // 非クライアントコントロール
        private ControlCollectionImpl _controls = new ControlCollectionImpl();    // 子コントロール

        // パラメータ
        private string _name = string.Empty;         // 名前
        private Rectangle _bounds = Rectangle.Empty;      // コントロール領域
        private Rectangle _clientRect = Rectangle.Empty;      // クライアント領域
        private Size _minimumSize = Size.Empty;           // 最小サイズ
        private INWControlTemplate _template = null;                 // コントロールテンプレート
        private object _layoutData = null;                 // レイアウトデータ

        // 状態
        private bool _initialized = false;                    // 初期化状態
        private NWControlState _state = NWControlState.Normal;    // コントロールの状態
        private bool _mouseDown = false;                    // マウス押下状態

        // レイアウト状態
        private bool _layoutDirty = false;            // レイアウト Dirty フラグ
        private bool _layoutLocked = false;            // レイアウトロック

        // スクロール状態
        private Point _scrollPosition = Point.Empty;  // スクロール位置
        private Point _scrollOriginWork = Point.Empty;  // スクロールロック解除時に適用するスクロール位置
        private bool _scrollLocked = false;        // スクロールロック

        // リソース
        private ResourceDictionary _resources = new ResourceDictionary();

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="host">コントロールをホストするクラス</param>
        /// <param name="name">コントロールの名前</param>
        public NWControl(NWControlHost host) : this(host, string.Empty) { }
        public NWControl(NWControlHost host, string name)
        {
            if (null == host)
            {
                Debug.Fail("host must be not null.");
                throw new ArgumentNullException();
            }
            if (null == name) { throw new ArgumentNullException(); }

            _host = host;
            _name = name;

            _controls.Inserted += OnChildControlInserted;
            _controls.Removed += OnChildControlRemoved;
            _ncControls.Inserted += OnChildNcControlInserted;
            _ncControls.Removed += OnChildNcControlRemoved;
        }

        #region ** プロパティ

        public string Name
        {
            get { return _name; }
        }

        #region ** リンク

        public Control Owner
        {
            get { return Host.Owner; }
        }

        public NWControlHost Host
        {
            get { return _host; }
        }

        public NWControlRoot Root
        {
            get { return Host.Root; }
        }

        public NWControl Parent
        {
            get { return _parent; }
            set
            {
                if (null != _parent)
                {
                    _parent.Controls.Remove(this);
                    _parent = null;
                }

                if (null != value)
                {
                    value.Controls.Add(this);
                }
            }
        }

        public ControlCollection Controls
        {
            get { return _controls; }
        }

        internal ControlCollection NcControls
        {
            get { return _ncControls; }
        }

        protected virtual IList<NWControl> OrderedControls
        {
            get { return _controls; }
        }

        #endregion

        #region ** テンプレート

        public INWControlTemplate Template
        {
            get { return _template; }
            set
            {
                if (value == _template) { return; }
                _template = value;
                InitializeComponent();
            }
        }

        public object LayoutData
        {
            get
            {
                if (null == Parent) { return _layoutData; }
                if (null != _layoutData) { return _layoutData; }
                return Parent.LayoutData;
            }
            set { _layoutData = value; }
        }

        protected virtual INWControlTemplate DefaultTemplate
        {
            get { return new NWControlTemplate(this); }
        }

        protected virtual Type TemplateType
        {
            get { return typeof(INWControlTemplate); }
        }

        #endregion

        #region ** 状態

        public virtual bool Visible
        {
            get
            {
                if (0 >= Width && 0 >= Height) { return false; }
                if (null == _parent) { return true; }

                Rectangle scrollRect = _parent.VisibleRectangle;
                scrollRect.Offset(-_parent.ClientRectangle.Left, -_parent.ClientRectangle.Top);

                return scrollRect.IntersectsWith(Bounds);
            }
        }

        public virtual NWControlState State
        {
            get { return _state; }
        }

        public virtual bool Enabled
        {
            get { return ((_state & NWControlState.Disabled) == 0); }
            set
            {
                if (value)
                {
                    SetState(_state & ~NWControlState.Disabled);
                }
                else
                {
                    SetState(_state | NWControlState.Disabled);
                }
            }
        }

        public virtual bool Hot
        {
            get { return ((_state & NWControlState.Hot) != 0); }
        }

        public virtual bool Selected
        {
            get { return ((_state & NWControlState.Selected) != 0); }
            set
            {
                if (value)
                {
                    SetState(_state | NWControlState.Selected);
                }
                else
                {
                    SetState(_state & ~NWControlState.Selected);
                }
            }
        }

        public virtual bool Focused
        {
            get { return ((_state & NWControlState.Focused) != 0); }
        }

        protected bool IsMouseDown
        {
            get { return _mouseDown; }
        }

        protected bool LayoutLocked
        {
            get
            {
                if (null == Parent) { return _layoutLocked; }
                return (_layoutLocked) ? true : Parent.LayoutLocked;
            }
        }

        protected bool LayoutDirty
        {
            get
            {
                if (null == Parent) { return _layoutDirty; }
                return (_layoutDirty) ? true : Parent.LayoutDirty;
            }
        }

        #endregion

        #region ** 座標

        public int Left
        {
            get { return Bounds.Left; }
        }

        public int Top
        {
            get { return Bounds.Top; }
        }

        public int Right
        {
            get { return Bounds.Right; }
        }

        public int Bottom
        {
            get { return Bounds.Bottom; }
        }

        public int Width
        {
            get { return Bounds.Width; }
        }

        public int Height
        {
            get { return Bounds.Height; }
        }

        public virtual Rectangle AbsoluteBounds
        {
            get
            {
                if (null == _parent) { return Bounds; }
                return _parent.RectangleToScreen(Bounds);
            }
        }

        public Rectangle Bounds
        {
            get { return _bounds; }
            set
            {
                if (value == _bounds) { return; }

                SetBounds(value);
            }
        }

        public Rectangle ClientRectangle
        {
            get { return _clientRect; }
        }

        public Rectangle VisibleRectangle
        {
            get
            {
                if (null == Parent)
                {
                    Rectangle scrollRect = ScrollRectangle;
                    scrollRect.Offset(-ClientRectangle.Left, -ClientRectangle.Top);
                    return scrollRect;
                }

                Rectangle parentScrollRect = Parent.ScrollRectangle;
                parentScrollRect.Offset(-Parent.ClientRectangle.Left, -Parent.ClientRectangle.Top);

                Rectangle work = Parent.RectangleToScreen(parentScrollRect);
                work.Intersect(RectangleToScreen(ScrollRectangle));

                if (work.IsEmpty) { return work; }

                work = RectangleToClient(work);
                work.Offset(ClientRectangle.Left, ClientRectangle.Top);

                return work;
            }
        }

        public Size MinimumSize
        {
            get { return _minimumSize; }
            set { _minimumSize = value; }
        }

        #region ** スクロール状態

        public Point ScrollPosition
        {
            get { return _scrollPosition; }
            set
            {
                if (_scrollLocked)
                {
                    _scrollOriginWork = value;
                    return;
                }

                _scrollPosition = value;
                Invalidate();
            }
        }

        public Rectangle ScrollRectangle
        {
            get
            {
                Rectangle work = ClientRectangle;
                work.Offset(_scrollPosition.X, _scrollPosition.Y);
                return work;
            }
        }

        public bool IsScrollLocked
        {
            get { return _scrollLocked; }
        }

        #endregion

        #endregion

        #endregion

        #region ** イベント

        // 初期化
        public event EventHandler Initialized;
        public event EventHandler Uninitialize;
        public event EventHandler Uninitialized;

        // レイアウト
        protected event NWControlLayoutEventHandler Layout;
        protected event NWControlNcLayoutEventHandler NcCalcSize;

        // コントロール操作
        public event NWControlEventHandler ControlInserted;
        public event NWControlEventHandler ControlRemoved;
        public event EventHandler Resize;

        // 状態変更
        public event NWControlStateEventHandler StateChanged;
        protected event EventHandler MouseDownChanged;

        // ユーザ操作
        public event EventHandler MouseActivating;
        public event EventHandler Click;
        public event MouseEventHandler MouseDown;
        public event MouseEventHandler MouseClick;
        public event MouseEventHandler MouseDoubleClick;

        #endregion

        #region ** イベントハンドラ

        #region ** 初期化

        protected virtual void OnInitializeComponent() { }

        protected virtual void OnInitialize() { }

        protected virtual void OnInitialized(EventArgs e)
        {
            if (null != Initialized)
            {
                Initialized(this, e);
            }
        }

        protected virtual void OnUninitialize(EventArgs e)
        {
            if (null != Uninitialize)
            {
                Uninitialize(this, e);
            }
        }

        protected virtual void OnUninitialized(EventArgs e)
        {
            if (null != Uninitialized)
            {
                Uninitialized(this, e);
            }
        }

        #endregion

        #region ** レイアウト

        protected virtual void OnLayout(NWControlLayoutEventArgs e)
        {
            if (null != _template)
            {
                _template.PerformLayout(e.LayoutData);
            }

            if (null != Layout)
            {
                Layout(this, e);
            }
        }

        protected virtual void OnNcLayout(NWControlNcLayoutEventArgs e)
        {
            if (null == _template) { return; }

            Rectangle clientRect = e.ClientRectangle;
            _template.PerformNcLayout(e.NewBounds, ref clientRect);

            e.ClientRectangle = clientRect;

            if (null != NcCalcSize)
            {
                NcCalcSize(this, e);
            }

            _clientRect = e.ClientRectangle;
        }

        protected virtual void OnResize(EventArgs e)
        {
            if (null != Resize)
            {
                Resize(this, e);
            }
        }

        #endregion

        #region ** 状態変化

        protected virtual void OnStateChanged(NWControlStateEventArgs e)
        {
            if (null != StateChanged)
            {
                StateChanged(this, e);
            }
        }

        protected virtual void OnMouseDownChanged(EventArgs e)
        {
            if (null != MouseDownChanged)
            {
                MouseDownChanged(this, e);
            }
        }

        #endregion

        #region ** コントロール操作

        protected virtual void OnControlInserted(NWControlEventArgs e)
        {
            if (null != ControlInserted)
            {
                ControlInserted(this, e);
            }
        }

        protected virtual void OnControlRemoved(NWControlEventArgs e)
        {
            if (null != ControlRemoved)
            {
                ControlRemoved(this, e);
            }
        }

        private void OnChildControlInserted(object sender, NWControlEventArgs e)
        {
            e.Control._parent = this;

            OnControlInserted(e);
        }

        private void OnChildControlRemoved(object sender, NWControlEventArgs e)
        {
            e.Control._parent = null;

            OnControlRemoved(e);
        }

        private void OnChildNcControlInserted(object sender, NWControlEventArgs e)
        {
            e.Control._parent = this;
        }

        private void OnChildNcControlRemoved(object sender, NWControlEventArgs e)
        {
            e.Control._parent = null;
        }

        #endregion

        #region ** マウス操作

        protected virtual void OnMouseEnter()
        {
            //Debug.WriteLine( "NWControl.OnMouseEnter" );

            if (!Enabled) { return; }

            SetState(_state | NWControlState.Hot);
        }

        protected virtual void OnMouseLeave()
        {
            //Debug.WriteLine( "NWControl.OnMouseLeave" );

            SetMouseDown(false);

            if (!Enabled) { return; }

            SetState(_state & ~NWControlState.Hot);
        }

        protected virtual void OnMouseActivating(EventArgs e)
        {
            if (null != MouseActivating)
            {
                MouseActivating(this, e);
            }
        }

        protected virtual void OnMouseMove(MouseEventArgs e) { }

        protected virtual void OnMouseDown(MouseEventArgs e)
        {
            //Debug.WriteLine( "NWControl.OnMouseDown" );

            if (!Enabled) { return; }
            if (!Visible) { return; }

            if (!Hot)
            {
                PerformMouseEnter();
            }

            SetMouseDown(true);

            if (null != MouseDown)
            {
                MouseDown(this, e);
            }
        }

        protected virtual void OnMouseUp(MouseEventArgs e)
        {
            //Debug.WriteLine( "NWControl.OnMouseUp" );

            if (IsMouseDown && AbsoluteBounds.Contains(e.Location))
            {

                if (MouseButtons.Left == e.Button)
                {
                    OnClicked(new EventArgs());
                }

                OnMouseClicked(e);
                SetMouseDown(false);

            }
        }

        protected virtual void OnClicked(EventArgs e)
        {
            if (null != Click)
            {
                Click(this, e);
            }
        }

        protected virtual void OnMouseClicked(MouseEventArgs e)
        {
            if (null != MouseClick)
            {
                MouseClick(this, e);
            }
        }

        protected virtual void OnMouseDoubleClick(MouseEventArgs e)
        {
            //Debug.WriteLine( "NWControl.OnMouseDoubleClick" );

            if (!Enabled) { return; }
            if (!Visible) { return; }

            if (null != MouseDoubleClick)
            {
                MouseDoubleClick(this, e);
            }
        }

        #endregion

        #region ** リソース操作

        protected virtual void OnUpdateResource(NWControl control, string key) { }

        #endregion

        #endregion

        #region ** メソッド

        #region ** 初期化

        public void InitializeComponent()
        {
            if (null != _template)
            {
                InitializeComponent(_template);
                return;
            }

            if (Host.Style.ContainsKey(GetType()))
            {
                _template = (Host.Style[GetType()] as NWControlTemplateFactory).CreateInstance(this);
            }

            if (null == _template)
            {
                _template = DefaultTemplate;
            }

            UninitializeComponent();
            InitializeComponentInternal(_template);
        }

        public void InitializeComponent(INWControlTemplate template)
        {
            UninitializeComponent();
            InitializeComponentInternal(_template);
        }

        public void UninitializeComponent()
        {
            if (!_initialized) { return; }

            OnUninitialize(new EventArgs());

            foreach (NWControl control in Controls)
            {
                control.UninitializeComponent();
            }

            foreach (NWControl control in NcControls)
            {
                control.UninitializeComponent();
            }

            Controls.Clear();
            NcControls.Clear();

            _initialized = false;
            OnUninitialized(new EventArgs());
        }

        private void InitializeComponentInternal(INWControlTemplate template)
        {
            Debug.Assert(!_initialized, "unexpected error.");
            Debug.Assert(null != template, "unexpected error.");

            if (!TemplateType.IsInstanceOfType(template))
            {
                throw new Exception("invalid control template.");
            }

            OnInitialize();

            _template.InitializeComponent();

            OnInitializeComponent();

            foreach (NWControl control in Controls)
            {
                control.InitializeComponent();
            }

            foreach (NWControl control in NcControls)
            {
                control.InitializeComponent();
            }

            _initialized = true;
            PerformLayout();

            OnInitialized(new EventArgs());
        }

        #endregion

        #region ** イベント発行

        public void PerformLayout()
        {
            _layoutDirty = true;
            PerformLayoutInternal(true);
        }

        public void PerformMouseEnter()
        {
            if (Hot) { return; }
            OnMouseEnter();
        }

        public void PerformMouseLeave()
        {
            if (!Hot) { return; }
            OnMouseLeave();
        }

        public void PerformMouseActivating()
        {
            OnMouseActivating(new EventArgs());
        }

        public void PerformMouseMove(MouseEventArgs e)
        {
            OnMouseMove(e);
        }

        public void PerformMouseDown(MouseEventArgs e)
        {
            OnMouseDown(e);
        }

        public void PerformMouseDoubleClick(MouseEventArgs e)
        {
            OnMouseDoubleClick(e);
        }

        public void PerformMouseUp(MouseEventArgs e)
        {
            OnMouseUp(e);
        }

        #endregion

        #region ** 描画処理

        public virtual void Draw(Graphics g)
        {
            if (!Visible) { return; }

            if (null != _template)
            {
                _template.Draw(g);
            }

            IList<NWControl> controls = OrderedControls;
            for (int i = controls.Count - 1; i >= 0; i--)
            {
                controls[i].Draw(g);
            }

            foreach (NWControl control in NcControls)
            {
                control.Draw(g);
            }
        }

        public void UpdateDrawComponent()
        {
            if (null != _template)
            {
                _template.Drawer.UpdateDrawComponent(this);
            }

            foreach (NWControl control in Controls)
            {
                control.UpdateDrawComponent();
            }

            foreach (NWControl control in NcControls)
            {
                control.UpdateDrawComponent();
            }

            Invalidate();
        }

        public void Invalidate()
        {
            Invalidate(this);
        }

        protected virtual void Invalidate(Rectangle bounds)
        {
            // Host にてオーバーライドして、オーナーコントロールの Invalidate を呼び出す
            if (null == _parent) { return; }
            _parent.Invalidate(bounds);
        }

        protected virtual void Invalidate(NWControl control)
        {
            // Host にてオーバーライドして、オーナーコントロールの Invalidate を呼び出す
            if (null == _parent) { return; }
            _parent.Invalidate(control);
        }

        #endregion

        #region ** 座標処理

        public NWHitResult HitTest(Point point)
        {
            return HitTest<NWControl>(point);
        }

        public NWHitResult HitTest<_ControlType>(Point point) where _ControlType : NWControl
        {
            if (!ScrollRectangle.Contains(point)) { return NWHitResult.NotHit; }

            foreach (NWControl control in OrderedControls)
            {

                if (!control.Visible) { continue; }
                if (!control.Bounds.Contains(point)) { continue; }

                NWHitResult hr = control.HitTest<_ControlType>(control.PointToClient(PointToScreen(point)));
                if (null != hr.Control) { return hr; }

                if (control is _ControlType)
                {
                    return new NWHitResult(control, Controls.IndexOf(control));
                }

            }

            return NWHitResult.NotHit;
        }

        public NWHitResult NcHitTest(Point point)
        {
            return NcHitTest<NWNCControl>(point);
        }

        public NWHitResult NcHitTest<_ControlType>(Point point) where _ControlType : NWNCControl
        {
            NWHitResult hr = NcHitTest<_ControlType>(_controls, point);
            if (null != hr.Control) { return hr; }

            return NcHitTest<_ControlType>(_ncControls, point);
        }

        public NWHitResult NcHitTest<_ControlType>(ControlCollection controls, Point point) where _ControlType : NWNCControl
        {
            Rectangle bounds = Bounds;
            bounds.Offset(-Left, -Top);
            if (!bounds.Contains(point)) { return NWHitResult.NotHit; }


            int index = 0;

            foreach (NWControl control in controls)
            {

                if (!control.Visible)
                {
                    index++;
                    continue;
                }

                if (!control.Bounds.Contains(point))
                {
                    index++;
                    continue;
                }

                NWHitResult hr = control.NcHitTest<_ControlType>(control.NcPointToClient(NcPointToScreen(point)));
                if (null != hr.Control) { return hr; }

                if (control is _ControlType)
                {
                    return new NWHitResult(control, index);
                }

                index++;

            }

            return NWHitResult.NotHit;
        }

        public virtual Point PointToClient(Point point)
        {
            Point work = point;

            if (null != _parent)
            {
                work = _parent.PointToClient(work);
            }

            work.Offset(-_bounds.Left - _clientRect.Left + _scrollPosition.X,
                         -_bounds.Top - _clientRect.Top + _scrollPosition.Y);
            return work;
        }

        public virtual Point PointToScreen(Point point)
        {
            Point work = point;
            work.Offset(_bounds.Left + _clientRect.Left - _scrollPosition.X, _bounds.Top + _clientRect.Top - _scrollPosition.Y);
            if (null == _parent) { return work; }

            return _parent.PointToScreen(work);
        }

        public virtual Rectangle RectangleToClient(Rectangle bounds)
        {
            Rectangle work = bounds;

            if (null != _parent)
            {
                work = _parent.RectangleToClient(work);
            }

            work.Offset(-_bounds.Left - _clientRect.Left + _scrollPosition.X,
                         -_bounds.Top - _clientRect.Top + _scrollPosition.Y);
            return work;
        }

        public virtual Rectangle RectangleToScreen(Rectangle bounds)
        {
            Rectangle work = bounds;
            work.Offset(_bounds.Left + _clientRect.Left - _scrollPosition.X, _bounds.Top + _clientRect.Top - _scrollPosition.Y);
            if (null == _parent) { return work; }

            return _parent.RectangleToScreen(work);
        }

        public virtual Point NcPointToClient(Point point)
        {
            Point work = point;

            if (null != _parent)
            {
                work = _parent.NcPointToClient(work);
            }

            work.Offset(-_bounds.Left, -_bounds.Top);
            return work;
        }

        public virtual Point NcPointToScreen(Point point)
        {
            Point work = point;
            work.Offset(_bounds.Left, _bounds.Top);
            if (null == _parent) { return work; }

            return _parent.NcPointToScreen(work);
        }

        public virtual Rectangle NcRectangleToScreen(Rectangle bounds)
        {
            Rectangle work = bounds;
            work.Offset(_bounds.Left, _bounds.Top);
            if (null == _parent) { return work; }

            return _parent.RectangleToScreen(work);
        }

        protected virtual void SetBounds(Rectangle bounds)
        {
            Rectangle newBounds = bounds;

            if (0 < MinimumSize.Width && newBounds.Width < MinimumSize.Width)
            {
                newBounds.Width = MinimumSize.Width;
            }
            if (0 < MinimumSize.Height && newBounds.Height < MinimumSize.Height)
            {
                newBounds.Height = MinimumSize.Height;
            }

            Size oldSize = _bounds.Size;

            Rectangle work = bounds;
            work.Offset(-work.Left, -work.Top);

            NWControlNcLayoutEventArgs e = new NWControlNcLayoutEventArgs(bounds, work);
            OnNcLayout(e);

            _bounds = bounds;

            if (oldSize != _bounds.Size)
            {
                OnResize(new EventArgs());
            }
        }

        #endregion

        #region ** レイアウト処理

        public void ApplyStyle()
        {
            SuspendLayout();

            ResetAllTemplates();
            InitializeComponent();

            ResumeLayout(true, true);
        }

        public virtual void SuspendLayout()
        {
            _layoutLocked = true;
        }

        public virtual void ResumeLayout()
        {
            ResumeLayout(true);
        }

        public virtual void ResumeLayout(bool performLayout)
        {
            ResumeLayout(performLayout, false);
        }

        protected virtual void ResumeLayout(bool performLayout, bool forced)
        {
            if (!_layoutLocked) { return; }

            _layoutLocked = false;

            if (performLayout)
            {
                PerformLayoutInternal(forced);
            }
        }

        protected virtual void PerformLayoutInternal(bool forced)
        {
            if (LayoutLocked || !_initialized)
            {
                _layoutDirty = true;
                return;
            }

            bool layoutDirty = LayoutDirty;

            if (forced || layoutDirty)
            {

                Rectangle clientRectWork = Bounds;
                clientRectWork.Offset(-clientRectWork.Left, -clientRectWork.Top);
                _clientRect = clientRectWork;

                OnLayout(new NWControlLayoutEventArgs(LayoutData));

                OnNcLayout(new NWControlNcLayoutEventArgs(Bounds, ClientRectangle));

            }

            foreach (NWControl control in _controls)
            {
                control.PerformLayoutInternal(forced);
            }

            Invalidate();

            _layoutDirty = false;
        }

        private void ResetAllTemplates()
        {
            _template = null;

            foreach (NWControl control in Controls)
            {
                control.ResetAllTemplates();
            }
        }

        #endregion

        #region ** スクロール処理

        public void SuspendScroll()
        {
            _scrollOriginWork = _scrollPosition;
            _scrollLocked = true;
        }

        public void ResumeScroll(bool performScroll)
        {
            if (!_scrollLocked) { return; }

            _scrollLocked = false;

            if (performScroll && _scrollPosition != _scrollOriginWork)
            {
                ScrollPosition = _scrollOriginWork;
            }
            else
            {
                _scrollOriginWork = _scrollPosition;
            }

            Debug.Assert(_scrollPosition == _scrollOriginWork, "unexpected error.");
        }

        #endregion

        #region ** 状態操作

        protected virtual void SetState(NWControlState state)
        {
            if (_state == state) { return; }

            NWControlState oldState = _state;

            _state = state;
            OnStateChanged(new NWControlStateEventArgs(this, oldState, state));

            Invalidate();
        }

        protected virtual void SetMouseDown(bool mouseDown)
        {
            if (mouseDown == _mouseDown) { return; }

            _mouseDown = mouseDown;
            OnMouseDownChanged(new EventArgs());
        }

        #endregion

        #region ** リソース操作

        public object GetResource(string key)
        {
            if (_resources.ContainsKey(key))
            {
                return _resources[key];
            }
            return (null == Parent) ? null : Parent.GetResource(key);
        }

        public void SetResource(string key, object data)
        {
            if (null == key) { throw new ArgumentNullException("key"); }
            if (0 == key.Length) { throw new ArgumentException("key"); }

            if (_resources.ContainsKey(key))
            {
                _resources[key] = data;
            }
            else
            {
                _resources.Add(key, data);
            }

            UpdateResource(this, key);
        }

        private void UpdateResource(NWControl control, string key)
        {
            OnUpdateResource(control, key);

            foreach (NWControl child in Controls)
            {
                child.UpdateResource(control, key);
            }
            foreach (NWControl child in NcControls)
            {
                child.UpdateResource(control, key);
            }
        }

        #endregion

        #endregion

        #region ** コレクション

        public class ControlCollection : Collection<NWControl>
        {
            public const int InvalidIndex = -1;

            private ControlDictionary _dictionary = new ControlDictionary();

            #region ** メソッド

            public bool ContainsKey(string key)
            {
                return _dictionary.ContainsKey(key);
            }

            public void Swap(int index1, int index2)
            {
                if (index1 < 0) { throw new ArgumentOutOfRangeException("index1"); }
                if (index2 < 0) { throw new ArgumentOutOfRangeException("index2"); }
                if (Count <= index1) { throw new ArgumentOutOfRangeException("index1"); }
                if (Count <= index2) { throw new ArgumentOutOfRangeException("index2"); }

                NWControl control1 = Items[index1];
                NWControl control2 = Items[index2];

                SetItem(index1, control2);
                SetItem(index2, control1);
            }

            public void Move(int from, int to)
            {
                if (from < 0) { throw new ArgumentOutOfRangeException("from"); }
                if (to < 0) { throw new ArgumentOutOfRangeException("to"); }
                if (Count <= from) { throw new ArgumentOutOfRangeException("from"); }
                if (Count <= to) { throw new ArgumentOutOfRangeException("to"); }

                int index = from;

                while (index != to)
                {

                    int next = (index < to) ? index++ : index--;

                    Swap(index, next);
                    index = next;

                }
            }

            #region ** Collection<> メソッドのオーバーライド

            /// <summary>
            /// System.Collections.ObjectModel.Collection<T> からすべての要素を削除します。
            /// </summary>
            protected override void ClearItems()
            {
                base.ClearItems();

                _dictionary.Clear();
            }

            /// <summary>
            /// System.Collections.ObjectModel.Collection<T> 内の指定したインデックスの位置に要素を挿入します。
            /// </summary>
            /// <param name="index">item を挿入する位置の、0 から始まるインデックス番号。</param>
            /// <param name="item">挿入するオブジェクト。参照型の場合、null の値を使用できます。</param>
            protected override void InsertItem(int index, NWControl item)
            {
                if (null == item) { throw new ArgumentNullException(); }

                base.InsertItem(index, item);

                if (item.Name.Length > 0 && !_dictionary.ContainsKey(item.Name))
                {
                    _dictionary.Add(item.Name, item);
                }
            }

            /// <summary>
            /// System.Collections.ObjectModel.Collection<T> の指定したインデックスにある要素を削除します。
            /// </summary>
            /// <param name="index">削除する要素の、0 から始まるインデックス番号。</param>
            protected override void RemoveItem(int index)
            {
                if (0 > index) { throw new ArgumentOutOfRangeException(); }
                if (index >= Count) { throw new ArgumentOutOfRangeException(); }

                string targetKey = Items[index].Name;

                base.RemoveItem(index);

                if (targetKey.Length > 0)
                {
                    _dictionary.Remove(targetKey);
                }
            }

            #endregion

            #endregion

            #region ** インデクサ

            public NWControl this[string key]
            {
                get
                {
                    if (!_dictionary.ContainsKey(key)) { return null; }
                    return _dictionary[key];
                }
            }

            #endregion
        }

        protected class ControlCollectionImpl : ControlCollection
        {
            #region ** イベント

            public event EventHandler Cleared;
            public event NWControlEventHandler Inserted;
            public event NWControlCancelEventHandler Removing;
            public event NWControlEventHandler Removed;

            #endregion

            #region ** メソッドのオーバーライド

            /// <summary>
            /// System.Collections.ObjectModel.Collection<T> からすべての要素を削除します。
            /// </summary>
            protected override void ClearItems()
            {
                while (Count > 0)
                {
                    RemoveItem(0);
                }

                base.ClearItems();

                if (null != Cleared)
                {
                    Cleared(this, new EventArgs());
                }
            }

            /// <summary>
            /// System.Collections.ObjectModel.Collection<T> 内の指定したインデックスの位置に要素を挿入します。
            /// </summary>
            /// <param name="index">item を挿入する位置の、0 から始まるインデックス番号。</param>
            /// <param name="item">挿入するオブジェクト。参照型の場合、null の値を使用できます。</param>
            protected override void InsertItem(int index, NWControl item)
            {
                if (null == item) { throw new ArgumentNullException(); }
                if (null != item.Parent) { throw new ArgumentException(); }

                base.InsertItem(index, item);

                if (null != Inserted)
                {
                    Inserted(this, new NWControlEventArgs(index, item));
                }
            }

            /// <summary>
            /// System.Collections.ObjectModel.Collection<T> の指定したインデックスにある要素を削除します。
            /// </summary>
            /// <param name="index">削除する要素の、0 から始まるインデックス番号。</param>
            protected override void RemoveItem(int index)
            {
                if (0 > index) { throw new ArgumentOutOfRangeException(); }
                if (index >= Count) { throw new ArgumentOutOfRangeException(); }

                NWControl target = Items[index];

                if (null != Removing)
                {

                    NWControlCancelEventArgs e = new NWControlCancelEventArgs(index, target);
                    Removing(this, e);

                    if (e.Cancel) { return; }

                }

                base.RemoveItem(index);

                if (null != Removed)
                {
                    Removed(this, new NWControlEventArgs(index, target));
                }
            }

            #endregion
        }

        private class ControlDictionary : Dictionary<string, NWControl> { }

        private class ResourceDictionary : Dictionary<string, object> { }

        #endregion
    }

    public class NWNCControl : NWControl
    {
        public NWNCControl(NWControlHost host) : base(host) { }
        public NWNCControl(NWControlHost host, string name) : base(host, name) { }

        #region ** プロパティ

        public override bool Visible
        {
            get
            {
                if (null == Parent) { return false; }
                if (0 == Bounds.Width && 0 == Bounds.Height) { return false; }
                return !Parent.ClientRectangle.Contains(Bounds);
            }
        }

        public override Rectangle AbsoluteBounds
        {
            get
            {
                if (null == Parent) { return Bounds; }
                return Parent.NcRectangleToScreen(Bounds);
            }
        }

        #endregion
    }

    #endregion

    #region ** ボタン

    /// <summary>
    /// ウィンドウレスボタン
    /// </summary>
    public class NWButton : NWControl
    {
        private Image _image = null;

        public NWButton(NWControlHost host) : base(host) { }
        public NWButton(NWControlHost host, string name) : base(host, name) { }

        #region ** プロパティ

        public Image Image
        {
            get { return _image; }
            set
            {
                if (value == _image) { return; }

                _image = value;
                Invalidate();
            }
        }

        protected override INWControlTemplate DefaultTemplate
        {
            get { return new NWButtonTemplate(this); }
        }

        #endregion

        #region ** イベントハンドラ

        #region ** 状態変化

        protected override void OnMouseDownChanged(EventArgs e)
        {
            if (IsMouseDown)
            {
                SetState(State | NWControlState.Selected);
            }
            else
            {
                SetState(State & ~NWControlState.Selected);
            }
        }

        #endregion

        #endregion
    }

    /// <summary>
    /// ウィンドウレスボタン（非クライアント領域用）
    /// </summary>
    public class NWNCButton : NWNCControl
    {
        private Image _image = null;

        public NWNCButton(NWControlHost host) : base(host) { }
        public NWNCButton(NWControlHost host, string name) : base(host, name) { }

        #region ** プロパティ

        public Image Image
        {
            get { return _image; }
            set
            {
                if (value == _image) { return; }

                _image = value;
                Invalidate();
            }
        }

        protected override INWControlTemplate DefaultTemplate
        {
            get { return new NWButtonTemplate(this); }
        }

        #endregion

        #region ** イベントハンドラ

        #region ** 状態変化

        protected override void OnMouseDownChanged(EventArgs e)
        {
            if (IsMouseDown)
            {
                SetState(State | NWControlState.Selected);
            }
            else
            {
                SetState(State & ~NWControlState.Selected);
            }
        }

        #endregion

        #endregion
    }

    #endregion

    #region ** 状態

    [Flags]
    public enum NWControlState
    {
        Disabled = 0x80,
        Normal = 0x00,
        Hot = 0x01,
        Selected = 0x02,
        Focused = 0x04,
    }

    #endregion

    #region ** イベントとデリゲート

    public class NWControlEventArgs : EventArgs
    {
        private int _index = NWControl.ControlCollection.InvalidIndex;
        private NWControl _control = null;

        public NWControlEventArgs(int index, NWControl control)
        {
            _index = index;
            _control = control;
        }

        #region ** プロパティ

        public int Index
        {
            get { return _index; }
        }

        public NWControl Control
        {
            get { return _control; }
        }

        #endregion
    }

    public class NWControlCancelEventArgs : CancelEventArgs
    {
        private int _index = NWControl.ControlCollection.InvalidIndex;
        private NWControl _control = null;

        public NWControlCancelEventArgs(int index, NWControl control)
        {
            _index = index;
            _control = control;
        }

        #region ** プロパティ

        public int Index
        {
            get { return _index; }
        }

        public NWControl Control
        {
            get { return _control; }
        }

        #endregion
    }

    public class NWControlStateEventArgs : EventArgs
    {
        private NWControl _control = null;
        private NWControlState _oldState = NWControlState.Normal;
        private NWControlState _newState = NWControlState.Normal;

        public NWControlStateEventArgs(NWControl control, NWControlState oldState, NWControlState newState)
        {
            _control = control;
            _oldState = oldState;
            _newState = newState;
        }

        #region ** プロパティ

        public NWControl Control
        {
            get { return _control; }
        }

        public NWControlState OldState
        {
            get { return _oldState; }
        }

        public NWControlState NewState
        {
            get { return _newState; }
        }

        #endregion
    }

    public class NWControlLayoutEventArgs : EventArgs
    {
        #region ** パラメータ

        private object _data = null;

        #endregion

        public NWControlLayoutEventArgs(object layoutData)
        {
            _data = layoutData;
        }

        #region ** プロパティ

        public object LayoutData
        {
            get { return _data; }
        }

        #endregion
    }

    public class NWControlNcLayoutEventArgs : EventArgs
    {
        #region ** パラメータ

        private Rectangle _newBounds = Rectangle.Empty;
        private Rectangle _clientRect = Rectangle.Empty;

        #endregion

        public NWControlNcLayoutEventArgs(Rectangle newBounds, Rectangle clientRect)
        {
            _newBounds = newBounds;
            _clientRect = clientRect;

            Validate();
        }

        #region ** プロパティ

        public Rectangle NewBounds
        {
            get { return _newBounds; }
        }

        public Rectangle ClientRectangle
        {
            get { return _clientRect; }
            set
            {
                _clientRect = value;
                Validate();
            }
        }

        #endregion

        #region ** メソッド

        private void Validate()
        {
            Rectangle bounds = _newBounds;
            bounds.Offset(-bounds.Left, -bounds.Top);

            if (!bounds.Contains(_clientRect)) { throw new ArgumentException(); }
        }

        #endregion
    }

    public delegate void NWControlEventHandler(object sender, NWControlEventArgs e);
    public delegate void NWControlCancelEventHandler(object sender, NWControlCancelEventArgs e);
    public delegate void NWControlStateEventHandler(object sender, NWControlStateEventArgs e);
    public delegate void NWControlLayoutEventHandler(object sender, NWControlLayoutEventArgs e);
    public delegate void NWControlNcLayoutEventHandler(object sender, NWControlNcLayoutEventArgs e);

    #endregion

    #region ** ヒットテスト

    public class NWHitResult
    {
        #region ** 固定値

        public static readonly NWHitResult NotHit = new NWHitResult();

        #endregion

        #region ** パラメータ

        private int _index = NWControl.InvalidIndex;
        private NWControl _control = null;

        #endregion

        private NWHitResult() { }
        public NWHitResult(NWControl control, int index)
        {
            _control = control;
            _index = index;
        }

        #region ** プロパティ

        public int Index
        {
            get { return _index; }
        }

        public NWControl Control
        {
            get { return _control; }
        }

        #endregion
    }

    #endregion

    #endregion

    #region ** ウィンドウレスコントロール テンプレート

    #region ** インターフェイス

    public interface INWComponentTemplate
    {
        #region ** メソッド

        void InitializeComponent();

        #endregion
    }

    public interface INWControlTemplate : INWComponentTemplate
    {
        #region ** プロパティ

        INWControlDrawer Drawer { get; set; }

        #endregion

        #region ** メソッド

        void PerformLayout(object layoutData);
        void PerformNcLayout(Rectangle newBounds, ref Rectangle clientRect);
        Size Measure(object layoutData);

        void Draw(Graphics g);

        #endregion
    }

    #endregion

    public abstract class NWControlHostTemplate : INWComponentTemplate
    {
        #region ** パラメータ

        private NWControlHost _host = null;

        #endregion

        protected NWControlHostTemplate(NWControlHost host)
        {
            Debug.Assert(null != host, "unexpected error.");
            _host = host;
        }

        #region ** プロパティ

        public NWControlHost Host
        {
            get { return _host; }
        }

        #endregion

        #region ** メソッド

        #region ** 初期化

        public virtual void InitializeComponent()
        {
            Host.Root = CreateRootControl(Host);
        }

        protected abstract NWControlRoot CreateRootControl(NWControlHost host);

        #endregion

        #endregion
    }

    public class NWControlTemplate : INWControlTemplate
    {
        #region ** パラメータ

        private NWControl _control = null;
        private INWControlDrawer _drawer = null;
        private object _layoutData = null;

        #endregion

        public NWControlTemplate(NWControl control) : this(control, new NWControlDrawer()) { }
        protected NWControlTemplate(NWControl control, INWControlDrawer drawer)
        {
            Debug.Assert(null != control, "unexpected error.");
            Debug.Assert(null != drawer, "unexpected error.");
            _control = control;
            _drawer = drawer;
        }

        #region ** プロパティ

        public NWControlHost Host
        {
            get { return Control.Host; }
        }

        public NWControl Control
        {
            get { return _control; }
        }

        public INWControlDrawer Drawer
        {
            get { return _drawer; }
            set
            {
                if (null == value) { throw new ArgumentNullException(); }
                _drawer = value;
            }
        }

        public object LayoutData
        {
            get { return _layoutData; }
        }

        #endregion

        #region ** メソッド

        #region ** 初期化

        public virtual void InitializeComponent() { }

        #endregion

        #region ** レイアウト

        public void PerformLayout(object layoutData)
        {
            _layoutData = layoutData;

            PerformLayoutInternal(layoutData);
        }

        public virtual void PerformNcLayout(Rectangle newBounds, ref Rectangle clientRect) { }
        public virtual Size Measure(object layoutData) { return Size.Empty; }

        protected virtual void PerformLayoutInternal(object layoutData) { }

        #endregion

        #region ** 描画処理

        public virtual void Draw(Graphics g)
        {
            if (null == Control) { return; }
            if (null == _drawer) { return; }

            NWControlDrawInfo drawInfo = CreateControlDrawInfo();
            if (null == drawInfo) { return; }

            _drawer.Draw(g, drawInfo);
        }

        protected virtual NWControlDrawInfo CreateControlDrawInfo()
        {
            return new NWControlDrawInfo(Control);
        }

        #endregion

        #endregion
    }

    public class NWButtonTemplate : NWControlTemplate
    {
        public NWButtonTemplate(NWControl control)
            : base(control, new NWButtonDrawer())
        {
            Debug.Assert(control is NWButton || control is NWNCButton, "unexpected error.");
        }
    }

    #region ** 描画情報

    public class NWControlDrawInfo
    {
        #region ** パラメータ

        private NWControl _control = null;

        #endregion

        public NWControlDrawInfo(NWControl control)
        {
            Debug.Assert(null != control, "unexpected error.");
            _control = control;
        }

        #region ** プロパティ

        public NWControl Control
        {
            get { return _control; }
        }

        public Rectangle ClipRect
        {
            get { return Control.RectangleToScreen(Control.VisibleRectangle); }
        }

        #endregion
    }

    #endregion

    #region ** テンプレート ファクトリ

    public interface INWComponentTemplateFactory
    {
        INWComponentTemplate CreateInstance(object component);
    }

    public class NWComponentTemplateFactory<_TemplateType, _ComponentType> : INWComponentTemplateFactory
        where _TemplateType : INWComponentTemplate
        where _ComponentType : class
    {
        #region ** フィールド

        private NWCreateComponentTemplate<_TemplateType, _ComponentType> _createTemplate = null;

        #endregion

        public NWComponentTemplateFactory(NWCreateComponentTemplate<_TemplateType, _ComponentType> createTemplate)
        {
            Debug.Assert(null != createTemplate, "unexpected error.");
            _createTemplate = createTemplate;
        }

        public _TemplateType CreateInstance(_ComponentType component)
        {
            return _createTemplate(component);
        }

        #region ** INWComponentTemplate の実装

        INWComponentTemplate INWComponentTemplateFactory.CreateInstance(object component)
        {
            return CreateInstance(component as _ComponentType);
        }

        #endregion
    }

    public class NWControlHostTemplateFactory : NWComponentTemplateFactory<INWComponentTemplate, NWControlHost>
    {
        public NWControlHostTemplateFactory(NWCreateComponentTemplate<INWComponentTemplate, NWControlHost> createTemplate)
            : base(createTemplate) { }
    }

    public class NWControlTemplateFactory : NWComponentTemplateFactory<INWControlTemplate, NWControl>
    {
        public NWControlTemplateFactory(NWCreateComponentTemplate<INWControlTemplate, NWControl> createTemplate)
            : base(createTemplate) { }
    }

    public delegate _TemplateType NWCreateComponentTemplate<_TemplateType, _ComponentType>(_ComponentType component);

    #endregion

    #region ** スタイル

    public class NWControlStyle : Dictionary<Type, INWComponentTemplateFactory> { }

    #endregion

    #endregion

    #region ** ウィンドウレスコントロール Drawer

    public interface INWControlDrawer
    {
        #region ** メソッド

        /// <summary>
        /// コントロールを描画します。
        /// </summary>
        /// <param name="g">Graphics オブジェクト</param>
        /// <param name="drawInfo">コントロールの描画情報</param>
        void Draw(Graphics g, NWControlDrawInfo drawInfo);

        /// <summary>
        /// 描画コンポーネントを更新します。
        /// <param name="control">対象コントロール</param>
        /// </summary>
        void UpdateDrawComponent(NWControl control);

        #endregion
    }

    public class NWControlDrawer : INWControlDrawer
    {
        #region ** イベントハンドラ

        /// <summary>
        /// コントロールを描画します。
        /// </summary>
        /// <param name="g">Graphics オブジェクト</param>
        /// <param name="drawInfo">コントロールの描画情報</param>
        protected virtual void OnDraw(Graphics g, NWControlDrawInfo drawInfo) { }

        /// <summary>
        /// 描画コンポーネントを更新します。
        /// </summary>
        protected virtual void OnUpdateDrawComponent(NWControl control) { }

        #endregion

        #region ** メソッド

        /// <summary>
        /// コントロールを描画します。
        /// </summary>
        /// <param name="g">Graphics オブジェクト</param>
        /// <param name="drawInfo">コントロールの描画情報</param>
        public void Draw(Graphics g, NWControlDrawInfo drawInfo)
        {
            OnDraw(g, drawInfo);
        }

        /// <summary>
        /// 描画コンポーネントを更新します。
        /// <param name="control">対象コントロール</param>
        /// </summary>
        public void UpdateDrawComponent(NWControl control)
        {
            OnUpdateDrawComponent(control);
        }

        #endregion
    }

    public class NWButtonDrawer : NWControlDrawer
    {
        #region ** イベントハンドラ

        /// <summary>
        /// コントロールを描画します。
        /// </summary>
        /// <param name="g">Graphics オブジェクト</param>
        /// <param name="drawInfo">コントロールの描画情報</param>
        protected override void OnDraw(Graphics g, NWControlDrawInfo drawInfo)
        {
            if (!(drawInfo.Control is NWButton || drawInfo.Control is NWNCButton))
            {
                Debug.Fail("target is not Button.");
                return;
            }

            if (VisualStyleRenderer.IsSupported)
            {
                DrawForVisualStyle(g, drawInfo.Control);
            }
            else
            {
                DrawForClassicStyle(g, drawInfo.Control);
            }

            DrawImage(g, drawInfo.Control);
        }

        #endregion

        #region ** メソッド

        private void DrawForVisualStyle(Graphics g, NWControl control)
        {
            PushButtonState drawState = PushButtonState.Normal;

            if (!control.Enabled)
            {
                drawState = PushButtonState.Disabled;
            }
            else if (control.Selected)
            {
                drawState = PushButtonState.Pressed;
            }
            else if (control.Hot)
            {
                drawState = PushButtonState.Hot;
            }

            ButtonRenderer.DrawButton(g, control.AbsoluteBounds, drawState);
        }

        private void DrawForClassicStyle(Graphics g, NWControl control)
        {
            ButtonState drawState = ButtonState.Normal;

            if (!control.Enabled)
            {
                drawState = ButtonState.Inactive;
            }
            else if (control.Selected)
            {
                drawState = ButtonState.Pushed;
            }

            ControlPaint.DrawButton(g, control.AbsoluteBounds, drawState);
        }

        private void DrawImage(Graphics g, NWControl control)
        {
            Debug.Assert(null != g, "unexpected error.");
            Debug.Assert(null != control, "unexpected error.");

            Image image = null;
            if (control is NWButton) { image = (control as NWButton).Image; }
            if (control is NWNCButton) { image = (control as NWNCButton).Image; }
            if (null == image) { return; }

            Rectangle rect = control.AbsoluteBounds;
            rect.X += (rect.Width - image.Size.Width) / 2;
            rect.Y += (rect.Height - image.Size.Height) / 2;
            rect.Width = image.Size.Width;
            rect.Height = image.Size.Height;

            g.DrawImage(image, rect);
        }

        #endregion
    }

    public class NWColor
    {
        #region ** プロパティ

        public static Color ThemeBorder
        {
            get
            {
                return (VisualStyleInformation.IsEnabledByUser) ?
                            VisualStyleInformation.TextControlBorder : SystemColors.ControlDark;
            }
        }

        public static Color ThemeBorderLight
        {
            get
            {
                return (VisualStyleInformation.IsEnabledByUser) ?
                            ControlPaint.Light(ThemeBorder) : SystemColors.Control;
            }
        }

        public static Color ThemeBorderLightLight
        {
            get { return ControlPaint.LightLight(ThemeBorder); }
        }

        #endregion

        #region ** メソッド

        public static Color Blend(Color color1, Color color2, int rate1)
        {
            Debug.Assert(0 <= rate1 && rate1 <= 255, "unexpected error.");
            if (0 == rate1) { return color2; }
            if (255 == rate1) { return color1; }
            return Color.FromArgb((color1.A * rate1 / 255) + (color2.A * (255 - rate1) / 255),
                                   (color1.R * rate1 / 255) + (color2.R * (255 - rate1) / 255),
                                   (color1.G * rate1 / 255) + (color2.G * (255 - rate1) / 255),
                                   (color1.B * rate1 / 255) + (color2.B * (255 - rate1) / 255));
        }

        #endregion
    }

    #endregion
}
