﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// グループの関連アイテム登録設定を検証します。
    /// </summary>
    public static class GroupItemRegisterTypeValidator
    {
        /// <summary>
        /// グループの関連アイテム登録設定を検証します。
        /// </summary>
        /// <param name="groupItem">対象グループアイテムを指定します。</param>
        /// <returns>検証結果を返します。</returns>
        public static ValidationResult Validate(GroupItemBase groupItem)
        {
            if (groupItem.Target != null)
            {
                return Validate(groupItem.Target, groupItem.RegisterType);
            }
            else
            {
                return new ValidationResult(false);
            }
        }

        /// <summary>
        /// グループの関連アイテム登録設定を検証します。
        /// </summary>
        /// <param name="groupItem">グループ登録対象のアイテムを指定します。</param>
        /// <param name="registerType">グループアイテムの登録方法を指定します。</param>
        /// <returns>検証結果を返します。</returns>
        public static ValidationResult Validate(
                                                SoundSetItem targetItem,
                                                GroupItemRegisterType registerType
                                                )
        {
            Ensure.Argument.NotNull(targetItem);

            if (targetItem is SoundSetBankBase)
            {
                return Validate(targetItem as SoundSetBankBase, registerType);
            }
            else if (targetItem is WaveSoundSetBase)
            {
                return Validate(targetItem as WaveSoundSetBase, registerType);
            }
            else if (targetItem is SequenceSoundBase)
            {
                return Validate(targetItem as SequenceSoundBase, registerType);
            }
            else if (targetItem is SequenceSoundSetBase)
            {
                return Validate(targetItem as SequenceSoundSetBase, registerType);
            }
            else if (targetItem is WaveArchiveBase)
            {
                return Validate(targetItem as WaveArchiveBase, registerType);
            }

            return ValidateUnknown(targetItem, registerType);
        }

        private static ValidationResult Validate(
                                                 SoundSetBankBase soundSetBank,
                                                 GroupItemRegisterType registerType
                                                 )
        {
            Ensure.Argument.NotNull(soundSetBank);

            if (!registerType.IsValidForBank())
            {
                string message = Resources.MessageResource.Message_InvalidGroupItemRegisterType;
                return new ValidationResult(false, message);
            }

            if (!registerType.BankAndWaveArchiveIncludes() &&
                WaveArchiveConsts.IsConstValue(soundSetBank.WaveArchiveReference))
            {
                string message =
                  Resources.MessageResource.Message_InvalidGroupItemRegisterTypeForAutoWaveArchive;
                return new ValidationResult(false, message);
            }

            return ValidationResult.NoError;
        }

        private static ValidationResult Validate(
                                                 WaveSoundSetBase waveSoundSet,
                                                 GroupItemRegisterType registerType
                                                 )
        {
            Ensure.Argument.NotNull(waveSoundSet);

            if (!registerType.IsValidForWaveSoundSet())
            {
                string message = Resources.MessageResource.Message_InvalidGroupItemRegisterType;
                return new ValidationResult(false, message);
            }

            if (!registerType.WaveSoundSetAndWaveArchiveIncludes() &&
                WaveArchiveConsts.IsConstValue(waveSoundSet.WaveArchiveReference))
            {
                string message =
                 Resources.MessageResource.Message_InvalidGroupItemRegisterTypeForAutoWaveArchive;
                return new ValidationResult(false, message);
            }

            return ValidationResult.NoError;
        }

        private static ValidationResult Validate(
                                                 SequenceSoundBase sequenceSound,
                                                 GroupItemRegisterType registerType
                                                 )
        {
            Ensure.Argument.NotNull(sequenceSound);

            string message = string.Empty;

            if (!registerType.IsValidForSequence())
            {
                message = Resources.MessageResource.Message_InvalidGroupItemRegisterType;
                return new ValidationResult(false, message);
            }

            switch (registerType)
            {
                case GroupItemRegisterType.All:
                case GroupItemRegisterType.OnlySequence:
                case GroupItemRegisterType.BankAndWaveArchive:
                    return ValidationResult.NoError;
            }

            message =
                Resources.MessageResource.Message_InvalidGroupItemRegisterTypeForSequenceSoundBankAutoWaveArchive;
            foreach (SoundSetBankBase soundSetBank in sequenceSound.SoundSetBanks)
            {
                if (soundSetBank == null)
                {
                    continue;
                }

                if (WaveArchiveConsts.IsConstValue(soundSetBank.WaveArchiveReference))
                {
                    return new ValidationResult(false, message);
                }
            }

            return ValidationResult.NoError;
        }

        private static ValidationResult Validate(
                                                 SequenceSoundSetBase sequenceSoundSet,
                                                 GroupItemRegisterType registerType
                                                 )
        {
            Ensure.Argument.NotNull(sequenceSoundSet);

            foreach (SequenceSoundBase sequenceSound in sequenceSoundSet.Children)
            {
                ValidationResult result = Validate(sequenceSound, registerType);

                if (!result.IsValid)
                {
                    return result;
                }
            }

            return ValidationResult.NoError;
        }

        private static ValidationResult Validate(
                                                 WaveArchiveBase waveArchive,
                                                 GroupItemRegisterType registerType
                                                 )
        {
            Ensure.Argument.NotNull(waveArchive);

            if (!registerType.IsValidForWaveArchive())
            {
                string message = Resources.MessageResource.Message_InvalidGroupItemRegisterType;
                return new ValidationResult(false, message);
            }

            return ValidationResult.NoError;
        }

        private static ValidationResult ValidateUnknown(
                                                        SoundSetItem soundSetItem,
                                                        GroupItemRegisterType registerType
                                                        )
        {
            Ensure.Argument.NotNull(soundSetItem);

            if (!registerType.AllIncludes())
            {
                string message = Resources.MessageResource.Message_InvalidGroupItemRegisterType;
                return new ValidationResult(false, message);
            }

            return ValidationResult.NoError;
        }
    }
}
