﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;

    public class UserParameterUtility
    {
        public static string ToString(UserParameterStructureSetting setting, ulong userParameter)
        {
            List<UserParameterStructure> structures = setting.Structures;

            if (0 < structures.Count)
            {
                switch (setting.ShowColumnType)
                {
                    case UserParameterShowColumnType.Defined:
                        string str = string.Empty;
                        foreach (UserParameterStructure structure in structures)
                        {
                            if (structure.ShowColumn == false)
                            {
                                continue;
                            }

                            if (str != string.Empty)
                            {
                                str += " : ";
                            }
                            string value = GetStringValue(structure, userParameter);
                            str += value;
                        }
                        return str;

                    case UserParameterShowColumnType.Binary:
                        return "0b" + Convert.ToString((int)userParameter, 2);

                    case UserParameterShowColumnType.Hexadecimal:
                        return "0x" + Convert.ToString((int)userParameter, 16);
                }
            }

            return string.Empty;
        }

        public static string GetStringValue(UserParameterStructure structure, ulong userParameter)
        {
            object obj = GetValue(structure, userParameter);

            if (obj is bool)
            {
                return (bool)obj == true ? "ON" : "OFF";
            }
            else if (string.IsNullOrEmpty(structure.ValueCandidate) == false)
            {
                Dictionary<string, string> dictionary = getValueCandidate(structure);
                if (dictionary.ContainsKey(obj.ToString()) == true)
                {
                    return dictionary[obj.ToString()];
                }
            }

            return obj.ToString();
        }

        public static object GetMaximum(UserParameterStructure structure)
        {
            switch (structure.StructureType)
            {
                case StructureTypes.Integer:
                    if (0 < structure.Size)
                    {
                        return int.MaxValue >> (32 - structure.Size);
                    }
                    break;

                case StructureTypes.UInteger:
                    if (0 < structure.Size)
                    {
                        return (ulong)uint.MaxValue >> (32 - structure.Size);
                    }
                    break;
            }

            return 0;
        }

        public static object GetMinimum(UserParameterStructure structure)
        {
            switch (structure.StructureType)
            {
                case StructureTypes.Integer:
                    if (0 < structure.Size)
                    {
                        return -(1 << (structure.Size - 1));
                    }
                    break;

                case StructureTypes.UInteger:
                    return 0UL;
            }

            return 0;
        }

        public static object GetValue(UserParameterStructure structure, ulong userParameter)
        {
            switch (structure.StructureType)
            {
                case StructureTypes.Integer:
                    return GetPartialIntegerValue(structure, userParameter);

                case StructureTypes.UInteger:
                    return GetPartialUIntegerValue(structure, userParameter);

                case StructureTypes.Decimal:
                    return GetPartialDecimalValue(structure, userParameter);

                case StructureTypes.Boolean:
                    return GetPartialBooleanValue(structure, userParameter);
            }

            return null;
        }

        public static int GetPartialIntegerValue(UserParameterStructure structure, ulong userParameter)
        {
            return getPartialIntegerValue(structure.Location, structure.Size, userParameter);
        }

        public static ulong GetPartialUIntegerValue(UserParameterStructure structure, ulong userParameter)
        {
            return getPartialUIntegerValue(structure.Location, structure.Size, userParameter);
        }

        public static float GetPartialDecimalValue(UserParameterStructure structure, ulong userParameter)
        {
            return getPartialDecimalValue(structure.Location, structure.Size, userParameter);
        }

        public static bool GetPartialBooleanValue(UserParameterStructure structure, ulong userParameter)
        {
            return getPartialBooleanValue(structure.Location, structure.Size, userParameter);
        }

        /// <summary>
        ///
        /// </summary>
        public static ulong SetValue(UserParameterStructure structure, ulong userParameter, object value)
        {
            ulong result = userParameter;

            switch (structure.StructureType)
            {
                case StructureTypes.Integer:
                    result = UserParameterUtility.SetPartialIntegerValue(structure, userParameter, (int)value);
                    break;

                case StructureTypes.UInteger:
                    result = UserParameterUtility.SetPartialUIntegerValue(structure, userParameter, (ulong)value);
                    break;

                case StructureTypes.Decimal:
                    result = UserParameterUtility.SetPartialDecimalValue(structure, userParameter, (float)value);
                    break;

                case StructureTypes.Boolean:
                    result = UserParameterUtility.SetPartialBooleanValue(structure, userParameter, (bool)value);
                    break;
            }

            return result;
        }

        public static ulong SetPartialIntegerValue(UserParameterStructure structure, ulong userParameter, int value)
        {
            return setPartialIntegerValue(structure.Location, structure.Size, userParameter, value);
        }

        public static ulong SetPartialUIntegerValue(UserParameterStructure structure, ulong userParameter, ulong value)
        {
            return setPartialUIntegerValue(structure.Location, structure.Size, userParameter, value);
        }

        public static ulong SetPartialDecimalValue(UserParameterStructure structure, ulong userParameter, float value)
        {
            return setPartialDecimalValue(structure.Location, structure.Size, userParameter, value);
        }

        public static ulong SetPartialBooleanValue(UserParameterStructure structure, ulong userParameter, bool value)
        {
            return setPartialBooleanValue(structure.Location, structure.Size, userParameter, value);
        }

        /// <summary>
        ///
        /// </summary>
        public static Dictionary<string, string> GetValueCandidate(UserParameterStructure structure)
        {
            return getValueCandidate(structure);
        }

        /// <summary>
        ///
        /// </summary>
        private static ulong GetMask(int size)
        {
            return (1UL << size) - 1;
        }

        private static ulong ClearBit(int location, int size, ulong uValue)
        {
            ulong mask = GetMask(size);
            uValue &= (mask << location) ^ unchecked((ulong)-1);

            return uValue;
        }

        private static ulong SetBit(int location, int size, ulong uValue, ulong value)
        {
            value &= GetMask(size);
            value <<= location;

            uValue = ClearBit(location, size, uValue);
            uValue |= value;

            return uValue;
        }

        /// <summary>
        ///
        /// </summary>
        private static int getPartialIntegerValue(int location, int size, ulong value)
        {
            ulong mask = GetMask(size);
            value = (value >> location & mask);
            ulong signMask = (ulong)(1L << (size - 1));

            if ((value & signMask) != 0)
            {
                return -(int)(signMask - (value & ~signMask));
            }
            else
            {
                return (int)(value & ~signMask);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private static ulong getPartialUIntegerValue(int location, int size, ulong value)
        {
            ulong mask = GetMask(size);
            value = (value >> location & mask);

            return value;
        }

        /// <summary>
        ///
        /// </summary>
        private static float getPartialDecimalValue(int location, int size, ulong value)
        {
            byte[] bytes = new byte[4];
            for (int index = 0; index <= 3; index++)
            {
                bytes[index] = (byte)(value & 0xFF);
                value >>= 8;
            }

            return BitConverter.ToSingle(bytes, 0);
        }

        /// <summary>
        ///
        /// </summary>
        private static bool getPartialBooleanValue(int location, int size, ulong value)
        {
            ulong mask = GetMask(size);
            value = (value >> location & mask);
            return value != 0 ? true : false;
        }

        /// <summary>
        ///
        /// </summary>
        private static ulong setPartialIntegerValue(int location, int size, ulong uValue, int value)
        {
            return SetBit(location, size, uValue, (ulong)value);
        }

        /// <summary>
        ///
        /// </summary>
        private static ulong setPartialUIntegerValue(int location, int size, ulong uValue, ulong value)
        {
            return SetBit(location, size, uValue, value);
        }

        /// <summary>
        ///
        /// </summary>
        private static ulong setPartialDecimalValue(int location, int size, ulong uValue, float value)
        {
            uValue = 0;
            byte[] bytes = BitConverter.GetBytes(value);

            for (int index = bytes.Length - 1; index >= 0; index--)
            {
                uValue <<= 8;
                uValue |= (ulong)bytes[index];
            }

            return uValue;
        }

        /// <summary>
        ///
        /// </summary>
        private static ulong setPartialBooleanValue(int location, int size, ulong uValue, bool value)
        {
            ulong tmp = (value == true ? 1UL : 0UL);

            return SetBit(location, size, uValue, tmp);
        }

        /// <summary>
        ///
        /// </summary>
        private static Dictionary<string, string> getValueCandidate(UserParameterStructure structure)
        {
            Dictionary<string, string> dictionary = new Dictionary<string, string>();
            HashSet<string> checkValue = new HashSet<string>(); // 同値検査用

            try
            {
                string text = structure.ValueCandidate;
                text = text.Replace('\r', '\n');
                text = text.Replace("\n\n", "\n");
                text = text.Replace(" ", string.Empty);
                Regex regex = new Regex(".+[=].+", RegexOptions.IgnoreCase);

                for (Match match = regex.Match(text); match.Success == true; match = match.NextMatch())
                {
                    object value = null;
                    string[] tokens = match.Value.Split(new[] { '=' }, 2);
                    if (tokens.Length == 2 &&
                        TryParseCandidate(structure.StructureType, tokens[1], out value) == true &&
                        checkValue.Contains(value.ToString()) == false)
                    {
                        // 同じ値でなければ追加します。
                        dictionary.Add(tokens[1], tokens[0]);
                        checkValue.Add(value.ToString()); // 同値検査のため追加します。
                    }
                }
            }
            catch
            {
            }

            return dictionary;
        }

        public static bool TryParseCandidate(StructureTypes type, string number, out object value)
        {
            value = null;

            switch (type)
            {
                case StructureTypes.Integer:
                    Int32 inum;
                    if (Int32.TryParse(number, out inum) == true)
                    {
                        value = inum;
                        return true;
                    }
                    break;

                case StructureTypes.UInteger:
                    UInt32 uinum;
                    if (UInt32.TryParse(number, out uinum) == true)
                    {
                        value = uinum;
                        return true;
                    }
                    break;

                case StructureTypes.Decimal:
                    float fnum;
                    if (float.TryParse(number, out fnum) == true)
                    {
                        value = fnum;
                        return true;
                    }
                    break;
            }

            return false;
        }
    }
}
