﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Diagnostics;
    using System.Linq;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;

    /// <summary>
    /// サウンドセットを削除するオペレーションです。
    /// </summary>
    public class RemoveSoundSetOperation : OperationImpl, IDisposable
    {
        private bool _disposed = false;

        private SoundProjectService _projectService;
        private string _filePath;
        private int _index = -1;
        private SoundArchiveOutputTypes outputType = SoundArchiveOutputTypes.SoundArchive;
        private bool containsOutputType = false;
        private Component _parentComponent;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="projectService">追加先のプロジェクトサービス。</param>
        /// <param name="documentRef">追加するサウンドセットドキュメントへの参照。</param>
        public RemoveSoundSetOperation(SoundProjectService projectService, string filePath)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (string.IsNullOrEmpty(filePath) == true) { throw new ArgumentNullException("filePath"); }

            _projectService = projectService;
            _filePath = filePath;
        }

        void IDisposable.Dispose()
        {
            if (_disposed) { return; }

            _filePath = null;

            _disposed = true;
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool ExecuteInternal()
        {
            if (string.IsNullOrEmpty(_filePath) == true) { return false; }

            SoundArchiveOutputTypes outputType = SoundArchiveOutputTypes.None;
            if (_projectService.RemoveSoundArchiveOutputType(_filePath, ref outputType) == true)
            {
                this.outputType = outputType;
                this.containsOutputType = true;
            }

            SoundSet soundSet = _projectService.SoundSetDocuments
                .Where(d => d.Resource.Key == _filePath)
                .Select(d => d.SoundSet)
                .First();
            Debug.Assert(soundSet != null);
            _parentComponent = soundSet.Parent;
            _index = _parentComponent.Children.IndexOf(soundSet);
            _projectService.RemoveSoundSet(_filePath);

            return true;
        }

        /// <summary>
        /// オペレーションをロールバックします。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool RollbackInternal()
        {
            if (_index < 0 || _projectService.Project.Children.Count < _index)
            {
                _projectService.AddSoundSet(_filePath, _parentComponent);
            }
            else
            {
                _projectService.InsertSoundSet(_filePath, _index, _parentComponent);
            }

            _index = 0;

            if (this.containsOutputType == true)
            {
                _projectService.AddSoundArchiveOutputType(_filePath, this.outputType);
            }

            return true;
        }
    }
}
