﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;

using NintendoWare.SoundFoundation.Operations;

namespace NintendoWare.SoundFoundation.Projects
{
    /// <summary>
    /// コンポーネントを削除するオペレーションです。
    /// </summary>
    public class RemoveComponentOperation : OperationImpl
    {
        private List<Component> _components = new List<Component>();    // 削除するコンポーネント
        private Component _target = null;                       // 削除先のコンポーネント

        // 削除コンポーネントのインデックスディクショナリ
        private Dictionary<Component, int> _indexes = new Dictionary<Component, int>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="component">削除するコンポーネント。</param>
        public RemoveComponentOperation(Component component) : this(new Component[] { component })
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="components">削除するコンポーネントの IEnumerable。</param>
        public RemoveComponentOperation(IEnumerable<Component> components)
        {
            if (null == components) { throw new ArgumentNullException("components"); }

            Component target = null;

            foreach (Component component in components)
            {

                if (null == component) { throw new ArgumentException("invalid argument"); }

                if (_indexes.ContainsKey(component)) { continue; }
                if (null != target && target != component.Parent)
                {
                    throw new ArgumentException("each component must have a same parent");
                }

                _components.Add(component);
                _indexes.Add(component, -1);

                target = component.Parent;
            }

            if (0 < _components.Count && null == target)
            {
                throw new ArgumentException("each component must have a same parent");
            }

            _target = target;
        }

        /// <summary>
        /// 削除するコンポーネントの IEnumerable を取得します。
        /// </summary>
        public IEnumerable<Component> Components
        {
            get { return _components; }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool ExecuteInternal()
        {
            try
            {
                _target.Children.BeginCompressEvent();

                foreach (Component component in _components)
                {

                    int index = component.Parent.Children.IndexOf(component);
                    if (0 > index) { throw new Exception("expected error"); }

                    _indexes[component] = index;
                    component.Parent.Children.RemoveAt(index);

                }

                _target.Children.EndCompressEvent();
            }
            catch
            {
                _target.Children.EndCompressEvent();
                RollbackInternal();
            }

            return true;
        }

        /// <summary>
        /// オペレーションをロールバックします。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool RollbackInternal()
        {
            _target.Children.BeginCompressEvent();

            for (int i = _components.Count - 1; i >= 0; i--)
            {
                Component component = _components[i];
                if (!_indexes.ContainsKey(component)) { continue; }

                int index = _indexes[component];
                if (index < 0 || _target.Children.Count < index) { continue; }

                _target.Children.Insert(_indexes[component], component);
            }

            _indexes.Clear();
            _target.Children.EndCompressEvent();
            return true;
        }
    }
}
