﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using NintendoWare.SoundFoundation.Operations;

namespace NintendoWare.SoundFoundation.Projects
{
    /// <summary>
    /// コレクション内でコンポーネントを移動するオペレーションです。
    /// </summary>
    public class MoveComponentOperation : OperationImpl
    {
        private Component[] _components = null;   // 移動するコンポーネント
        private int _targetIndex = 0;      // 移動先インデックス
        private Component _parentComponent = null;   // 親コンポーネント

        // 移動コンポーネントのインデックスディクショナリ
        private Dictionary<Component, int> _indexes = new Dictionary<Component, int>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="component">移動するコンポーネント。</param>
        /// <param name="destIndex">移動先インデックス。</param>
        public MoveComponentOperation(Component component, int targetIndex)
        {
            if (null == component) { throw new ArgumentNullException("component"); }
            if (null == component.Parent)
            {
                throw new ArgumentException("component must have a parent.");
            }

            if (-1 > targetIndex || component.Parent.Children.Count <= targetIndex)
            {
                throw new ArgumentOutOfRangeException("targetIndex");
            }

            _components = new Component[] { component };
            _parentComponent = component.Parent;
            _targetIndex = (-1 == targetIndex) ? _parentComponent.Children.Count : targetIndex;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="component">移動するコンポーネント。</param>
        /// <param name="targetComponent">移動先のコンポーネント。</param>
        public MoveComponentOperation(Component component, Component targetComponent)
        {
            if (null == component) { throw new ArgumentNullException("component"); }
            if (null == component.Parent)
            {
                throw new ArgumentException("component.Parent is invalid.");
            }

            if (null == targetComponent) { throw new ArgumentNullException("targetComponent"); }
            if (targetComponent.Parent != component.Parent)
            {
                throw new ArgumentException("targetComponent.Parent is invalid.");
            }

            _components = new Component[] { component };
            _parentComponent = component.Parent;
            _targetIndex = _parentComponent.Children.IndexOf(targetComponent);
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="components">移動するコンポーネントの IEnumerable。</param>
        /// <param name="targetIndex">移動先インデックス。</param>
        public MoveComponentOperation(IEnumerable<Component> components, int targetIndex)
        {
            if (null == components) { throw new ArgumentNullException("components"); }

            Component parent = null;

            foreach (Component component in components)
            {
                if (null == component.Parent)
                {
                    throw new ArgumentException("all components must have a same parent.");
                }

                if (null == parent)
                {
                    parent = component.Parent;
                    continue;
                }

                if (parent != component.Parent)
                {
                    throw new ArgumentException("all components must have a same Parent.");
                }
            }

            if (-1 > targetIndex || (null != parent && parent.Children.Count < targetIndex))
            {
                throw new ArgumentOutOfRangeException("targetIndex.");
            }

            _components = components.ToArray();
            _parentComponent = parent;

            if (null == parent)
            {
                _targetIndex = -1;
            }
            else
            {
                _targetIndex = (-1 == targetIndex) ? parent.Children.Count : targetIndex;
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="components">移動するコンポーネントの IEnumerable。</param>
        /// <param name="targetComponent">移動先のコンポーネント。</param>
        public MoveComponentOperation(IEnumerable<Component> components, Component targetComponent)
        {
            if (null == components) { throw new ArgumentNullException("components"); }

            if (null == targetComponent) { throw new ArgumentNullException("targetComponent"); }
            if (null == targetComponent.Parent)
            {
                throw new ArgumentException("targetComponent must have a parent.");
            }

            foreach (Component component in components)
            {
                if (targetComponent.Parent != component.Parent)
                {
                    throw new ArgumentException("component.Parent must be targetComponent.Parent.");
                }
            }

            _components = components.ToArray();
            _parentComponent = targetComponent.Parent;
            _targetIndex = _parentComponent.Children.IndexOf(targetComponent);
        }

        /// <summary>
        /// 移動先インデックスを取得します。
        /// </summary>
        public int TargetIndex
        {
            get { return _targetIndex; }
        }

        /// <summary>
        /// 移動するコンポーネントの IEnumerator を取得します。
        /// </summary>
        public IEnumerable<Component> Components
        {
            get { return _components; }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool ExecuteInternal()
        {
            try
            {
                foreach (Component component in _components)
                {
                    _indexes[component] = _parentComponent.Children.IndexOf(component);
                }

                _parentComponent.Children.Move(_components, _targetIndex);
            }
            catch
            {
                _indexes.Clear();
            }
            return true;
        }

        /// <summary>
        /// オペレーションをロールバックします。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool RollbackInternal()
        {
            for (int i = _components.Length - 1; i >= 0; i--)
            {
                Component component = _components[i];
                if (!_indexes.ContainsKey(component))
                {
                    continue;
                }

                _parentComponent.Children.Move(component, _indexes[component]);
            }

            _indexes.Clear();
            return true;
        }
    }
}
