﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;

    /// <summary>
    /// サウンドセットを追加するオペレーションです。
    /// </summary>
    public class AddSoundSetOperation : OperationImpl, IDisposable
    {
        private bool _disposed = false;

        private SoundProjectService _projectService;
        private string _filePath;
        private Component _targetComponent;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="projectService">追加先のプロジェクトサービス。</param>
        /// <param name="filePath">追加するサウンドセットのファイルパス</param>
        /// <param name="targetComponent">親Componentの指定です。nullを指定するとSoundProjectが親になります。</param>
        public AddSoundSetOperation(SoundProjectService projectService, string filePath, Component targetComponent)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (string.IsNullOrEmpty(filePath) == true) { throw new ArgumentNullException("filePath"); }

            _projectService = projectService;
            _filePath = filePath;
            _targetComponent = targetComponent;
        }

        void IDisposable.Dispose()
        {
            if (_disposed) { return; }

            _filePath = null;

            _disposed = true;
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool ExecuteInternal()
        {
            if (string.IsNullOrEmpty(_filePath) == true) { return false; }

            _projectService.AddSoundSet(_filePath, _targetComponent);

            return true;
        }

        /// <summary>
        /// オペレーションをロールバックします。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool RollbackInternal()
        {
            if (string.IsNullOrEmpty(_filePath) == true) { return false; }
            _projectService.RemoveSoundSet(_filePath);
            return true;
        }
    }
}
