﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.FileFormats.Wave;

    public class StreamSoundTrackBase : Component, ICloneable
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public WaveFile WaveFile { get; set; }

        ///--------------------------------
        /// <summary>
        /// ファイルパス
        /// </summary>
        public string FilePath
        {
            get { return GetFilePathValue(ProjectParameterNames.FilePath); }
            set { SetFilePathValue(ProjectParameterNames.FilePath, value); }
        }

        ///--------------------------------
        /// <summary>
        /// 音量
        /// </summary>
        public int Volume
        {
            get { return GetIntValue(ProjectParameterNames.Volume); }
            set { SetIntValue(ProjectParameterNames.Volume, value); }
        }

        ///--------------------------------
        /// <summary>
        /// パン
        /// </summary>
        public int Pan
        {
            get { return GetIntValue(ProjectParameterNames.Pan); }
            set { SetIntValue(ProjectParameterNames.Pan, value); }
        }

        ///--------------------------------
        /// <summary>
        /// サラウンドパン
        /// </summary>
        public int SurroundPan
        {
            get { return GetIntValue(ProjectParameterNames.SurroundPan); }
            set { SetIntValue(ProjectParameterNames.SurroundPan, value); }
        }

        ///--------------------------------
        /// <summary>
        /// フロントバイパス
        /// </summary>
        public bool FrontBypass
        {
            get { return GetBoolValue(ProjectParameterNames.FrontBypass); }
            set { SetBoolValue(ProjectParameterNames.FrontBypass, value); }
        }

        /// <summary>
        /// センド設定を取得または設定します。
        /// </summary>
        public Sends Sends
        {
            get { return this.GetValue<Sends>(ProjectParameterNames.Sound.Sends); }
            set { this.SetValue<Sends>(ProjectParameterNames.Sound.Sends, value); }
        }

        ///--------------------------------
        /// <summary>
        /// LPF (Low-pass filter)
        /// </summary>
        public int LPF
        {
            get { return GetIntValue(ProjectParameterNames.LPF); }
            set { SetIntValue(ProjectParameterNames.LPF, value); }
        }

        ///--------------------------------
        /// <summary>
        /// Biquad フィルタの種類
        /// </summary>
        public BiquadType BiquadType
        {
            get { return GetValue<BiquadType>(ProjectParameterNames.BiquadType); }
            set { SetValue<BiquadType>(ProjectParameterNames.BiquadType, value); }
        }

        ///--------------------------------
        /// <summary>
        /// Biquad フィルタ値
        /// </summary>
        public int Biquad
        {
            get { return GetIntValue(ProjectParameterNames.Biquad); }
            set { SetIntValue(ProjectParameterNames.Biquad, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public bool PreviewMute
        {
            get { return GetBoolValue(ProjectParameterNames.StreamSoundTrack.PreviewMute); }
            set { SetBoolValue(ProjectParameterNames.StreamSoundTrack.PreviewMute, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public bool PreviewSoloPlay
        {
            get { return GetBoolValue(ProjectParameterNames.StreamSoundTrack.PreviewSoloPlay); }
            set { SetBoolValue(ProjectParameterNames.StreamSoundTrack.PreviewSoloPlay, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public bool PreviousPreviewMute
        {
            get;
            set;
        }

        /// <summary>
        /// １トラックに格納数チャンネルの数 1 or 2
        /// - aac の場合はトラック毎にユーサが指定
        /// - aac 以外は、WaveFile.ChannelCount の値
        /// </summary>
        public int ChannelCount
        {
            get { return GetIntValue(ProjectParameterNames.StreamSoundTrack.ChannelCount); }
            set { SetIntValue(ProjectParameterNames.StreamSoundTrack.ChannelCount, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public virtual ICloneable Clone()
        {
            StreamSoundTrackBase track = new StreamSoundTrackBase();

            foreach (string key in track.Parameters.Keys)
            {
                if (key == ProjectParameterNames.Sound.Sends)
                {
                    continue;
                }
                else if (this.Parameters.ContainsKey(key) == true)
                {
                    track.Parameters[key].Value = this.Parameters[key].Value;
                }
            }

            track.Sends = this.Sends.Clone() as Sends;

            return track;
        }

        object ICloneable.Clone()
        {
            return this.Clone();
        }

        ///--------------------------------
        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected override void InitializeParameters()
        {
            base.InitializeParameters();

            SetInstanceFilePathValue(ProjectParameterNames.FilePath, String.Empty);
            SetInstanceIntValue(ProjectParameterNames.Volume, 127, 0, 255);
            SetInstanceIntValue(ProjectParameterNames.Pan, 64, 0, 127);
            SetInstanceIntValue(ProjectParameterNames.SurroundPan, 0, 0, 127);
            SetInstanceBoolValue(ProjectParameterNames.FrontBypass, false);
            SetInstanceIntValue(ProjectParameterNames.LPF, 64, 0, 64);
            SetInstanceBiquadTypeValue(ProjectParameterNames.BiquadType, BiquadType.None);
            SetInstanceIntValue(ProjectParameterNames.Biquad, 0, 0, 127);
            SetInstanceIntValue(ProjectParameterNames.StreamSoundTrack.ChannelCount, 2, 1, 2);

            SendsParameterValue sendsParam = new SendsParameterValue(new Sends());
            Parameters.AddValue(ProjectParameterNames.Sound.Sends, sendsParam);

            // 一時的なパラメータを初期化します。(ファイルに保存されないパラメータ)
            SetInstanceBoolValue(ProjectParameterNames.StreamSoundTrack.PreviewMute, false);
            SetInstanceBoolValue(ProjectParameterNames.StreamSoundTrack.PreviewSoloPlay, false);

            // ストリームバイナリ依存パラメータ
            this.SetParameterAttributes(ProjectParameterNames.FilePath, ParameterAttributes.ComputeHash);

            // sndedit 転送対象パラメータ（値が変化しても sndedit 用にコンバートされません）
            this.SetParameterAttributes(ProjectParameterNames.Volume, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Pan, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.SurroundPan, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.FrontBypass, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.LPF, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.BiquadType, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Biquad, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.IsEnabled, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.Sends, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.StreamSoundTrack.PreviewMute, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.StreamSoundTrack.PreviewSoloPlay, ParameterAttributes.ComputeSndeditTransferHash);
            //
            PreviousPreviewMute = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceBiquadTypeValue(string name, BiquadType value)
        {
            SetInstanceParameterValue(name, new BiquadTypeParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnParameterValueChanged(ParameterEventArgs e)
        {
            if (e.Key == ProjectParameterNames.FilePath)
            {
                WaveFile = null;
            }

            base.OnParameterValueChanged(e);
        }
    }
}
